/* ------------------------------------------------------------------------
 * $Id: SceneManager.hh,v 1.11 2001/08/28 13:18:50 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-10-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _SceneManager_hh_
#define _SceneManager_hh_

// -- 3Dwm Includes
#include "Nobel/Node.hh"
#include "Celsius/Runnable.hh"
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"

// -- Forward Declarations
class View;
class Mutex;
class NodeImpl;

// -- Class Declarations

/**
 * Scene graph manager class. The scene manager is in command of the
 * scene graph and is responsible for rendering it in a timely
 * fashion. It is created and owned by the Backplane and is usually
 * executed in a thread of its own (note that with SDL on some
 * hardware, the rendering thread must be the same that called
 * SDL_Init(), so this thread is usually the main execution thread).
 **/
class SceneManager : public Runnable {

public:

    /**
     * Constructor. 
     **/
    SceneManager(View *v);
   
    /**
     * Destructor.
     **/
    ~SceneManager();

    /**
     * Enter the main rendering loop. This call will not return
     * (except when the manager is terminated).
     **/
    void run();
    
    /**
     * Perform a two-dimensional pick operation in the scene graph. In
     * reality, this will cast a ray originating from the current view
     * through the scene graph and send the event to the closest scene
     * graph intersected by the ray.
     *
     * @@@ Need to pass an event here as well!
     *
     * @param p 2D vector position of the pick.
     **/
    //void pick(const Vector2D &p, int button, bool pressed);

    /**
     * Perform a three-dimensional pick operation in the scene graph.
     * Here, we will trace the 3D position to a specific scene graph
     * node and propagate the event from there.
     *
     * @param p 3D vector position of the pick.
     **/
    void pick(const Vector3D &p);

    /**
     *
     **/
    void handlePositional(const Vector2D &p, Nobel::Event e);

    /**
     *
     **/
    void handleNonPositional(Nobel::Event e);
    
    /**
     * Terminate the scene manager instance.
     **/
    void terminate() { _quit = true; }

    /**
     * Create a root node (composite) for a new client.
     **/
    NodeImpl *createClientRoot();

    /**
     * Retrieve the currently active server view. In the future, the
     * server might consists of several views, independently managed
     * and displaying different parts of the scene graph.
     *
     * @return the current view.
     **/
    View *getCurrentView() const {
	return _view;
    }

private:
    Mutex _mutex;
    bool _quit;
    View *_view;
    NodeImpl *_root;
    Nobel::Node_var _focus;
};

#endif /* SceneManager */
