/* ------------------------------------------------------------------------
 * Semaphore.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-19 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Semaphore_hh_
#define _Semaphore_hh_

// -- System Includes
#include <semaphore.h>

// -- Class Declarations

class Semaphore {
public:
    
    /**
     * Default constructor.
     * 
     * @param v initial value for the semaphore (defaults to zero).
     */
    Semaphore(int v = 0) { 
	sem_init(&_semaphore, 0, v);
    }
    
    /**
     * Destructor. 
     **/
    ~Semaphore() {
	sem_destroy(&_semaphore);
    }

    /**
     * Lock the semaphore (decreasing the internal value). Will block
     * if the semaphore is already locked.
     **/
    void lock() {
	sem_wait(&_semaphore);
    }

    /**
     * Unlock the semaphore (increasing the internal value). Will wake
     * any suspended threads/processes waiting for the mutex.
     **/
    void unlock() {
	sem_post(&_semaphore);
    }

    /** 
     * Attempt to lock the mutex and return the result.
     * 
     * @return true if the mutex was locked, false otherwise.
     **/
    bool tryLock() {
	return sem_trywait(&_semaphore) == 0;
    }

    /**
     * Is the semaphore locked or not?
     * 
     * @return true if the semaphore is locked.
     */
    bool isLocked() {
	return getValue() == 0; 
    }

    /**
     * Retrieve the semaphore value.
     *
     * @return the value.
     **/
    int getValue() {
	int value;
	sem_getvalue(&_semaphore, &value);
	return value;
    }

private:
    
    Semaphore(const Semaphore &s);
    Semaphore &operator = (const Semaphore &);
    
    sem_t _semaphore;
};

#endif /* Semaphore.hh */
