/* ------------------------------------------------------------------------
 * Mutex.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-19 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Mutex_hh_
#define _Mutex_hh_

// -- System Includes
#include <pthread.h>

// -- Class Declarations

/**
 * Simple thread mutex class used for mutual exclusion (protection of
 * so-called critical sections). Meant to be used with the
 * CriticalSection class for some easy thread handling semantics.
 *
 * This class has been prepared to become a base class for several
 * different mutex implementations. Then we should use a Factory class
 * for getting implementations of different thread libraries.
 **/
class Mutex {

public:

    /**
     * Default constructor.
     * 
     * @param lock should the mutex be locked? Defaults to false for 'no'.
     */
    Mutex(bool lock = false) : _locked(lock) {
	pthread_mutex_init(&_mutex, NULL);
	if (lock == true) this->lock();
    }
    
    /**
     * Destructor.
     **/
    virtual ~Mutex() {
	pthread_mutex_destroy(&_mutex);
    }
    
    /**
     * Lock the mutex. Will block if the mutex is already locked.
     **/
    void lock() {
	pthread_mutex_lock(&_mutex);
	_locked = true;
    }
    
    /**
     * Unlock the mutex. Will wake any suspended threads waiting for
     * the mutex.
     **/
    void unlock() {
	pthread_mutex_unlock(&_mutex);
	_locked = false;
    }

    /** 
     * Attempt to lock the mutex and return the result.
     * 
     * @return true if the mutex was locked, false otherwise.
     **/
    bool tryLock() {
	if (pthread_mutex_trylock(&_mutex) != 0) 
	    return false;
	_locked = true;
	return true;
    }
    
    /**
     * Is the mutex locked or not?
     * 
     * @return true if the mutex is locked.
     */
    bool isLocked() const {
	return _locked; 
    }
    
protected:

    Mutex(const Mutex &mutex);
    Mutex &operator = (const Mutex &m);

    /// POSIX thread mutex
    pthread_mutex_t _mutex;
    
    /// Is the mutex locked?
    bool _locked;
};

#endif /* Mutex.hh */
