# -*- coding: utf-8 -*-

from RestrictedManager.handlers.xorg_driver import XorgDriverHandler

from gettext import gettext as _

class NvidiaDriver(XorgDriverHandler):
    is_handler = True

    name = "nvidia"
    rationale = _("This driver is required to fully utilise the 3D potential "
                 "of NVIDIA graphics cards, as well as provide 2D "
                 "acceleration of newer cards.\n\n"
                 "If you wish to enable desktop effects, this driver is "
                 "required.\n\n"
                 "If this driver is not enabled, you will not be able to "
                 "enable desktop effects and will not be able to run software "
                 "that requires 3D acceleration, such as some games.")

    def description(self):
        return _("NVIDIA accelerated graphics driver")

    def __init__(self, module, 
        driver_package = "nvidia-glx",
        extra_options = {"AddARGBVisuals": "True", "AddARGBGLXVisuals": "True", "NoLogo": "True"}):
        XorgDriverHandler.__init__(self, module, driver_package, "nvidia",
            extra_options, "nv", ["glx"], ["dri", "GLcore"])

    def is_loaded(self):
        """Returns True if the module is loaded."""

        # if you modprobe nvidia_{new,legacy} they will still appear as
        # "nvidia" in /proc/modules for some weird reason, so we need to
        # account for that
        return "nvidia" in self._modules

    def enable_config_hook(self):
        # compiz does not work otherwise
        self.xorg_conf.getSections('Screen')[0].defaultdepth = 24

class LegacyNvidiaDriver(NvidiaDriver):
    is_handler = True

    name = "nvidia_legacy"

    def __init__(self, module):
        NvidiaDriver.__init__(self, module, "nvidia-glx-legacy", 
            {"AllowGLXWithComposite": "True", "NoLogo": "True"})

    def description(self):
        return _("NVIDIA accelerated graphics driver (legacy cards)")

class NewNvidiaDriver(NvidiaDriver):
    is_handler = True

    name = "nvidia_new"

    def __init__(self, module):
        NvidiaDriver.__init__(self, module, "nvidia-glx-new")

    def description(self):
        return _("NVIDIA accelerated graphics driver (latest cards)")
