/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#include "driver-config.h"

#define EXPORT_SYMTAB

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/sched.h>
#ifdef KERNEL_2_2
#   include <linux/slab.h>
#else
#   include <linux/malloc.h>
#endif
#include <linux/poll.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/mm.h>
#include "compat_skbuff.h"
#include <linux/sockios.h>
#include "compat_sock.h"

#define __KERNEL_SYSCALLS__
#include <asm/io.h>

#include <linux/proc_fs.h>
#include <linux/file.h>

#include "vnetInt.h"
#include "compat_netdevice.h"
#include "vmnetInt.h"


typedef struct VNetNetIF {
   VNetPort                port;
   struct net_device       dev;
   char                    devName[VNET_NAME_LEN];
   struct net_device_stats stats;
} VNetNetIF;


static void VNetNetIfFree(VNetJack *this);
static void VNetNetIfReceive(VNetJack *this, struct sk_buff *skb);
static Bool VNetNetIfCycleDetect(VNetJack *this, int generation);

static int  VNetNetifOpen(struct net_device *dev);
static int  VNetNetifProbe(struct net_device *dev);
static int  VNetNetifClose(struct net_device *dev);
static int  VNetNetifStartXmit(struct sk_buff *skb, struct net_device *dev);
static struct net_device_stats *VNetNetifGetStats(struct net_device *dev);
static int  VNetNetifSetMAC(struct net_device *dev, void *addr);
static void VNetNetifSetMulticast(struct net_device *dev);
#if 0
#ifdef KERNEL_2_3_43
static void VNetNetifTxTimeout(struct net_device *dev);
#endif
#endif

static int  VNetNetIfProcRead(char *page, char **start, off_t off,
                              int count, int *eof, void *data);

#ifndef KERNEL_2_3_43
/* softnet API emulation */


/*
 *----------------------------------------------------------------------
 *
 * netif_stop_queue --
 *
 *      Stops queue processing.
 * 
 * Results: 
 *      None.
 *
 *----------------------------------------------------------------------
 */

static inline void
netif_stop_queue(struct net_device *dev) // IN:
{
   dev->tbusy = 1;
}


/*
 *----------------------------------------------------------------------
 *
 * netif_start_queue --
 *
 *      Enables queue processing. It does not try to start received
 *      frames processing.
 * 
 * Results: 
 *      None.
 *
 *----------------------------------------------------------------------
 */

static inline void
netif_start_queue(struct net_device *dev) // IN:
{
   dev->tbusy = 0;
}


/*
 *----------------------------------------------------------------------
 *
 * netif_wake_queue --
 *
 *      Enables queue processing. It schedules receive queue processing.
 * 
 * Results: 
 *      None.
 *
 *----------------------------------------------------------------------
 */

static inline void
netif_wake_queue(struct net_device *dev) // IN:
{
   dev->tbusy = 0;
   mark_bh(NET_BH);
}
#endif


#if 0
#ifdef KERNEL_2_3_43
/*
 *----------------------------------------------------------------------
 *
 * VNetNetIfTxTimeout --
 *
 *      Enables processing of Tx queue after it was stopped for so long.
 *      It should not happen with vmnet system.
 * 
 * Results: 
 *      None.
 *
 * Side effects:
 *      Tx queue enabled, message in log.
 *
 *----------------------------------------------------------------------
 */

static void
VNetNetifTxTimeout(struct net_device *dev) // IN:
{
   static int netRateLimit = 0;
   
   if (netRateLimit < 10) {
      LOG(0, (KERN_NOTICE "%s: Transmit timeout\n", dev->name));
      netRateLimit++;
   }
   /* We cannot stuck due to hardware, so always wake up processing */
   netif_wake_queue(dev);
}
#endif
#endif


/*
 *----------------------------------------------------------------------
 *
 * VNetNetIf_Create --
 *
 *      Create a net level port to the wonderful world of virtual
 *      networking.
 * 
 * Results: 
 *      Errno. Also returns an allocated port to connect to,
 *      NULL on error.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetIf_Create(char *devName,  // IN:
                 VNetPort **ret, // OUT:
                 int hubNum)     // IN: 
{
   VNetNetIF *netIf;
   struct net_device *dev = NULL;
   int retval = 0;
   static unsigned id = 0;
   
   netIf = kmalloc(sizeof *netIf, GFP_KERNEL);
   if (!netIf) {
      retval = -ENOMEM;
      goto out;
   }

   /*
    * Initialize fields.
    */
   
   netIf->port.id = id++;   
   netIf->port.next = NULL;

   netIf->port.jack.peer = NULL;
   netIf->port.jack.numPorts = 1;
   VNetSnprintf(netIf->port.jack.name, sizeof netIf->port.jack.name,
		"netif%u", netIf->port.id);
   netIf->port.jack.private = netIf;
   netIf->port.jack.index = 0;
   netIf->port.jack.procEntry = NULL;
   netIf->port.jack.free = VNetNetIfFree;
   netIf->port.jack.rcv = VNetNetIfReceive;
   netIf->port.jack.cycleDetect = VNetNetIfCycleDetect;
   netIf->port.jack.portsChanged = NULL;
   netIf->port.jack.isBridged = NULL;
   
   /*
    * Make proc entry for this jack.
    */
   
   retval = VNetProc_MakeEntry(NULL, netIf->port.jack.name, S_IFREG,
                               &netIf->port.jack.procEntry);
   if (retval) {
      if (retval == -ENXIO) {
         netIf->port.jack.procEntry = NULL;
      } else {
         netIf->port.jack.procEntry = NULL;
         goto out;
      }
   } else {
      netIf->port.jack.procEntry->read_proc = VNetNetIfProcRead;
      netIf->port.jack.procEntry->data = netIf;
   }

   /*
    * Rest of fields.
    */
   
   netIf->port.flags = IFF_RUNNING;

   memset(netIf->port.paddr, 0, sizeof netIf->port.paddr);
   memset(netIf->port.ladrf, 0, sizeof netIf->port.ladrf);

   /* This will generate the reserved MAC address c0:00:?? where ?? == hubNum. */
   VMX86_BUILD_MAC(netIf->port.paddr, hubNum);
   
   /* Make sure the MAC is unique. */
   retval = VNetSetMACUnique(&netIf->port, netIf->port.paddr);
   if (retval) {
     goto out;
   }

   netIf->port.fileOpRead = NULL;
   netIf->port.fileOpWrite = NULL;
   netIf->port.fileOpIoctl = NULL;
   netIf->port.fileOpPoll = NULL;
   
   dev = &netIf->dev;
   
   memset(dev, 0, sizeof *dev);
   dev->priv = netIf;
   dev->init = VNetNetifProbe;
   
   memcpy(netIf->devName, devName, sizeof netIf->devName);
   NULL_TERMINATE_STRING(netIf->devName);

#ifdef KERNEL_2_3_99
   strncpy(dev->name, netIf->devName, sizeof dev->name);
   NULL_TERMINATE_STRING(dev->name);
#else
   dev->name = netIf->devName;
#endif
   memset(&netIf->stats, 0, sizeof netIf->stats);
   
   ether_setup(dev); // turns on IFF_BROADCAST, IFF_MULTICAST
   memcpy(dev->dev_addr, netIf->port.paddr, sizeof netIf->port.paddr);
   
   dev->open = &VNetNetifOpen;
   dev->hard_start_xmit = &VNetNetifStartXmit;
   dev->stop = &VNetNetifClose;
   dev->get_stats = &VNetNetifGetStats;
   dev->set_mac_address = &VNetNetifSetMAC;
   dev->set_multicast_list = &VNetNetifSetMulticast;
#ifdef KERNEL_2_3_43
   /*
    * We cannot stuck... If someone will report problems under
    * low memory conditions or some such, we should enable it.
    */
#if 0
   dev->tx_timeout = &VNetNetifTxTimeout;
   dev->watchdog_timeo = TX_TIMEOUT;
#endif
#endif   
   
   if (register_netdev(dev) != 0) {
      LOG(0, (KERN_NOTICE "%s: could not register network device\n", devName));
      retval = -ENODEV;
      goto out;
   }

   *ret = (VNetPort*)netIf;
   return 0;

out:
   if (netIf) {
      if (netIf->port.jack.procEntry) {
         VNetProc_RemoveEntry(netIf->port.jack.procEntry, NULL);
      }
      kfree(netIf);
   }
   return retval;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetIfFree --
 *
 *      Free the net interface port.
 *
 * Results: 
 *      None.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

void
VNetNetIfFree(VNetJack *this) // IN: jack
{
   VNetNetIF *netIf = (VNetNetIF*)this;
   unregister_netdev(&netIf->dev);
   if (this->procEntry) {
      VNetProc_RemoveEntry(this->procEntry, NULL);
   }
   kfree(netIf);
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetIfReceive --
 *
 *      This jack is receiving a packet. Take appropriate action.
 *
 * Results: 
 *      None.
 *
 * Side effects:
 *      Frees skb.
 *
 *----------------------------------------------------------------------
 */

void
VNetNetIfReceive(VNetJack        *this, // IN: jack
		 struct sk_buff  *skb)  // IN: packet 
{
   VNetNetIF *netIf = (VNetNetIF*)this->private;
   uint8 *dest = SKB_2_DESTMAC(skb);
   
   if (!NETDEV_UP_AND_RUNNING(&netIf->dev)) {
      goto drop_packet;
   }

   if (!VNetPacketMatch(dest,
                        netIf->dev.dev_addr,
                        (uint8 *)AllMultiFilter, 
                        netIf->dev.flags)) {
      goto drop_packet;
   }
   
   /* send to the host interface */
   skb->dev = &netIf->dev;
   skb->protocol = eth_type_trans(skb, &netIf->dev);
   netif_rx_ni(skb);
   netIf->stats.rx_packets++;

   return;
   
 drop_packet:
   dev_kfree_skb(skb);
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetIfCycleDetect --
 *
 *      Cycle detection algorithm.
 * 
 * Results: 
 *      TRUE if a cycle was detected, FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

Bool
VNetNetIfCycleDetect(VNetJack *this,       // IN: jack
                     int       generation) // IN: 
{
   VNetNetIF *netIf = (VNetNetIF*)this->private;
   return VNetCycleDetectIf(netIf->devName, generation);
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifOpen --
 *
 *      The virtual network's open dev operation. 
 *
 * Results: 
 *      errno.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetifOpen(struct net_device *dev) // IN:
{
   /*
    * The host interface is not available if the hub is bridged.
    *
    * It's actually okay to support both.  We just need
    * to tag packets when VNetXmitPacket gives them to the interface
    * so they can be dropped by VNetBridgeReceive().
    *
    *  if so return -EBUSY;
    */

   netif_start_queue(dev);
#ifndef KERNEL_2_3_43
   /* Softnet does not have interrupt hack ... */
   dev->interrupt = 0;
   /* ... and sets start for us */
   dev->start = 1;
#endif
   // xxx need to change flags
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifProbe --
 *
 *      ???
 *
 * Results: 
 *      0.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetifProbe(struct net_device *dev) // IN: unused
{
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifClose --
 *
 *      The virtual network's close dev operation. 
 *
 * Results: 
 *      errno.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetifClose(struct net_device *dev) // IN:
{
#ifndef KERNEL_2_3_43
   /* Softnet generic layer clears it for us */
   dev->start = 0;
#endif
   netif_stop_queue(dev);
   // xxx need to change flags
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifStartXmit --
 *
 *      The virtual network's start xmit dev operation. 
 *
 * Results: 
 *      ???, 0.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetifStartXmit(struct sk_buff    *skb, // IN:
                   struct net_device *dev) // IN:
{
   VNetNetIF *netIf = (VNetNetIF*)dev->priv;

   if(skb == NULL) {
      return 0;
   }

   /* 
    * Block a timer-based transmit from overlapping.  This could better be
    * done with atomic_swap(1, dev->tbusy), but set_bit() works as well.
    * If this ever occurs the queue layer is doing something evil!
    */

#ifndef KERNEL_2_3_43
   /* Softnet does not play with tbusy... */
   if (test_and_set_bit(0, (void*)&dev->tbusy) != 0)
   {
      LOG(0, (KERN_NOTICE "%s: transmitter access conflict.\n", dev->name));
      return 1;
   }
#endif
   VNetSend(&netIf->port.jack, skb);

   netIf->stats.tx_packets++;
#ifndef KERNEL_2_3_43
   dev->tbusy = 0;
#endif
   dev->trans_start = jiffies;

   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifSetMAC --
 *
 *      Sets MAC address (i.e. via ifconfig) of netif device.
 *
 * Results: 
 *      Errno.
 *
 * Side effects:
 *      The MAC address may be changed.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetifSetMAC(struct net_device *dev, // IN:
                void *p)                // IN:
{
   VNetNetIF *netIf = (VNetNetIF*)dev->priv;
   struct sockaddr const *addr = p;
   if (!VMX86_IS_STATIC_MAC(addr->sa_data)) {
      return -EINVAL;
   }
   memcpy(netIf->port.paddr, addr->sa_data, dev->addr_len);
   memcpy(dev->dev_addr, addr->sa_data, dev->addr_len);
   return 0;
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifSetMulticast --
 *
 *      Sets or clears the multicast address list.  This information
 *      comes from an array in dev->mc_list, and with a counter in
 *      dev->mc_count.
 *
 *      Since host-only network ifaces can't be bridged, it's debatable
 *      whether this is at all useful, but at least now you can turn it 
 *      on from ifconfig without getting an ioctl error.
 * Results: 
 *      Void.
 *
 * Side effects:
 *      Multicast address list might get changed.
 *
 *----------------------------------------------------------------------
 */

void
VNetNetifSetMulticast(struct net_device *dev) // IN: unused
{
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetifGetStats --
 *
 *      The virtual network's get stats dev operation. 
 *
 * Results: 
 *      A struct full of stats.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

struct net_device_stats *
VNetNetifGetStats(struct net_device *dev) // IN:
{
   VNetNetIF *netIf = (VNetNetIF*)dev->priv;
   return &(netIf->stats);
}


/*
 *----------------------------------------------------------------------
 *
 * VNetNetIfProcRead --
 *
 *      Callback for read operation on this netif entry in vnets proc fs.
 *
 * Results: 
 *      Length of read operation.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

int
VNetNetIfProcRead(char   *page,  // IN/OUT: buffer to write into
                  char  **start, // OUT: 0 if file < 4k, else offset into page
                  off_t   off,   // IN: (unused) offset of read into the file
                  int     count, // IN: (unused) maximum number of bytes to read
                  int    *eof,   // OUT: TRUE if there is nothing more to read
                  void   *data)  // IN: client data
{
   VNetNetIF *netIf = (VNetNetIF*)data; 
   int len = 0;
   
   if (!netIf) {
      return len;
   }
   
   len += VNetPrintPort(&netIf->port, page+len);

   len += sprintf(page+len, "dev %s ", netIf->devName);
   
   len += sprintf(page+len, "\n");

   *start = 0;
   *eof   = 1;
   return len;
}
