/*
 * capture.c, vi: tabstop=4
 * capture module, controled by timer functions
 *
 * Copyright (C) 1997-98 Rasca, Berlin
 * Copyright (C) 2003,04 Karl H. Beckers, Frankfurt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 * This file contains routines used for capturing individual frames.
 * Theese routines are called from the record button callback in
 * the GUI and call themselves again till they are stopped by a stop
 * button event handler, a timeout, exceeding the maximum number of
 * frames (see the various VC_... states)
 *
 */

#include "../config.h" 	/* autoconf output */

#include <stdio.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif //HAVE_STDINT_H
#include <limits.h>
#include <sys/time.h>
#include <time.h>
#include <unistd.h>
#ifdef SOLARIS
#include <X11/X.h>
#include <X11/Xlib.h>
#endif //SOLARIS
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/XWDFile.h>
#include <X11/Xproto.h>
#include <X11/Xlibint.h>
#include <X11/cursorfont.h>
#ifdef HAVE_SHMAT
#include <sys/ipc.h>
#include <sys/shm.h>
#include <X11/extensions/XShm.h>
#endif //HAVE_SHMAT
#ifdef HasDGA
#include <X11/extensions/xf86dga.h>
#endif //HasDGA
#ifdef HasVideo4Linux
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include "video.h"
#endif //HasVideo4Linux

#include "capture.h"
#include "job.h"
#include "control.h"
#include "main.h"


uint16_t mousePointerBlack[] = { 0, 49152, 40960, 36864, 34816, 33792, 33280, 33024, 32896, 32832,
33728, 37376, 43264, 51456, 1152, 1152, 576, 576, 448, 0 };
uint16_t mousePointerWhite[] = { 0, 0, 16384, 24576, 28672, 30720, 31744, 32256, 32512, 32640, 31744,
27648, 17920, 1536, 768, 768, 384, 384, 0, 0 };


static void getCurrentPointer(int *x, int *y, Job *mjob) {
    Window mrootwindow, childwindow;
    int dummy, num_atoms;
    Atom *m_atom;
    Display *dpy;
    Screen *scr;
    
    if ( ! (mjob->flags & FLG_NOGUI ) ) {
        scr = mjob->win_attr.screen;
        dpy = DisplayOfScreen(scr);
    } else {
        dpy = XOpenDisplay(NULL);
    }
    mrootwindow = DefaultRootWindow(dpy);
    
    if (XQueryPointer(dpy, mrootwindow, &mrootwindow, &childwindow,
    x, y, &dummy, &dummy, &dummy)) {
        if (mjob->flags & FLG_RUN_VERBOSE) {
            int bla, ev_base, er_base, major, minor;
            Bool xTest;
            Cursor xCursor;
            
//            printf("Mouse pointer at x/y : %i/%i\n", *x, *y);
//            printf("Relative mouse pointer at x/y : %i/%i\n", (*x - mjob->area->x), (*y - mjob->area->y));
            
            //			printf("root window pointer: %p - child window pointer: %p\n", &mrootwindow, &childwindow);
            /*			m_atom = XListProperties(dpy, childwindow, &num_atoms);
             for (bla = 0; bla < num_atoms; bla++) {
             printf("atom %i name: %s\n", bla, XGetAtomName(dpy,
             m_atom[bla]));
             } */
            
            /*			xTest = XTestQueryExtension( dpy, &ev_base, &er_base, &major, &minor);
             if ( xTest = True ) {
             int n;
            
             for (n = 0; n < (XC_num_glyphs - 1); n++) {
             xTest = 0;
            
             xCursor = XCreateFontCursor( dpy, n);
             xTest = XTestCompareCursorWithWindow( dpy, childwindow, xCursor );
             if (xTest = True ) printf("xTest: %i - glyph: %i\n", (int) xTest, n);
             }
             } else {
             printf("XTEST extension not supported\n");
             } */
            
        }
    } else {
        printf("couldn't find mouse pointer for disp: %p , rootwindow: %p\n", dpy,
        RootWindow(dpy, DefaultScreen(dpy)));
        *x = -1;
        *y = -1;
    }
    
    if ( mjob->flags & FLG_NOGUI ) {
        XCloseDisplay(dpy);
    }
}



static void paintMousePointer(int *x, int *y, Job *mjob, XImage *image) {
    if ( (*x - mjob->area->x) >= 0 && *x < (mjob->area->width + mjob->area->x) &&
    (*y - mjob->area->y) >= 0 && *y < (mjob->area->height + mjob->area->y) ) {
        int line;
        uint8_t *im_data = image->data;
        
        im_data += (image->bytes_per_line * (*y - mjob->area->y)); // shift to right line
        im_data += (image->bits_per_pixel / 8 * (*x - mjob->area->x)); // shift to right pixel
        
        switch(image->bits_per_pixel) {
            case 32: {
                uint32_t *cursor;
                int width;
                uint16_t bm_b, bm_w, mask;
                
                for (line = 0; line < 20; line++ ) {
                    if (mjob->mouseWanted == 1) {
                        bm_b = mousePointerBlack[line];
                        bm_w = mousePointerWhite[line];
                    } else {
                        bm_b = mousePointerWhite[line];
                        bm_w = mousePointerBlack[line];
                    }
                    mask = ( 0x0001 << 15 );
                    
                    for (cursor = (uint32_t*) im_data, width = 0;
                    ((width + *x) < (mjob->area->width + mjob->area->x)&&width < 16);
                    cursor++, width++) {
                        //							Boolean pointer_b_bit, pointer_w_bit;
                        
                        //							pointer_b_bit = ( ( bm_b & mask ) > 0 );
                        //							pointer_w_bit = ( ( bm_w & mask ) > 0 );
                        //							printf("%i ", pointer_b_bit, pointer_w_bit );
                        
                        if ( ( bm_b & mask ) > 0 ) {
                            *cursor &= ((~ image->red_mask) & (~ image->green_mask) & (~
                            image->blue_mask ));
                        } else if ( ( bm_w & mask ) > 0 ) {
                            *cursor |= (image->red_mask | image->green_mask | image->blue_mask );
                        }
                        mask >>= 1;
                        
                    }
                    //						printf("\n");
                    
                    im_data += image->bytes_per_line;
                }
            }
            break;
            case 24: // not sure this can occur at all ..........
                printf("input image bits_per_pixel %i not implemented with mouse pointer capture ... aborting!\n",
                image->bits_per_pixel);
                printf("Please file a bug at http://www.sourceforge.net/projects/xvidcap/\n");
                exit(1);
                break;
            case 16: {
                uint16_t *cursor;
                int width;
                uint16_t bm_b, bm_w, mask;
                
                for (line = 0; line < 16; line++ ) {
                    if (mjob->mouseWanted == 1) {
                        bm_b = mousePointerBlack[line];
                        bm_w = mousePointerWhite[line];
                    } else {
                        bm_b = mousePointerWhite[line];
                        bm_w = mousePointerBlack[line];
                    }
                    mask = ( 0x0001 << 15 );
                    
                    for (cursor = (uint16_t*) im_data, width = 0;
                    ((width + *x) < (mjob->area->width + mjob->area->x)&&width < 6);
                    cursor++, width++) {
                        //							Boolean pointer_b_bit, pointer_w_bit;
                        
                        //							pointer_b_bit = ( ( bm_b & mask ) > 0 );
                        //							pointer_w_bit = ( ( bm_w & mask ) > 0 );
                        //							printf("%i ", pointer_b_bit, pointer_w_bit );
                        
                        if ( ( bm_b & mask ) > 0 ) {
                            *cursor &= ((~ image->red_mask) & (~ image->green_mask) & (~
                            image->blue_mask ));
                        } else if ( ( bm_w & mask ) > 0 ) {
                            *cursor |= (image->red_mask | image->green_mask | image->blue_mask );
                        }
                        mask >>= 1;
                        
                    }
                    //						printf("\n");
                    
                    im_data += image->bytes_per_line;
                }
            }
            break;
            case 8: {
                uint8_t *cursor;
                int width;
                uint16_t bm_b, bm_w, mask;
                
                for (line = 0; line < 16; line++ ) {
                    if (mjob->mouseWanted == 1) {
                        bm_b = mousePointerBlack[line];
                        bm_w = mousePointerWhite[line];
                    } else {
                        bm_b = mousePointerWhite[line];
                        bm_w = mousePointerBlack[line];
                    }
                    mask = ( 0x0001 << 15 );
                    
                    for (cursor = im_data, width = 0;
                    ((width + *x) < (mjob->area->width + mjob->area->x)&&width < 6);
                    cursor++, width++) {
                        //							Boolean pointer_b_bit, pointer_w_bit;
                        
                        //							pointer_b_bit = ( ( bm_b & mask ) > 0 );
                        //							pointer_w_bit = ( ( bm_w & mask ) > 0 );
                        //							printf("%i ", pointer_b_bit, pointer_w_bit );
                        
                        if ( ( bm_b & mask ) > 0 ) {
                            *cursor = 0;
                        } else if ( ( bm_w & mask ) > 0 ) {
                            *cursor = 1;
                        }
                        mask >>= 1;
                        
                    }
                    //						printf("\n");
                    
                    im_data += image->bytes_per_line;
                }
            }
            break;
            default:
                printf("input image bits_per_pixel %i not supported with mouse pointer capture ... aborting!\n",
                image->bits_per_pixel);
                exit(1);
                
        }
        
    }
    
}



/*
 * just read new data in the image structure, the image
 * structure inclusive the data area must be allocated before
 */
static Boolean
XGetZPixmap(Display *dpy, Drawable d, XImage *image, int x, int y) {
    xGetImageReply rep;
    register xGetImageReq *req;
    long nbytes;
    
    if (!image)
        return (False);
    LockDisplay(dpy);
    GetReq(GetImage, req);
    /*
     * first set up the standard stuff in the request
     */
    req->drawable = d;
    req->x = x;
    req->y = y;
    req->width = image->width;
    req->height = image->height;
    req->planeMask = AllPlanes;
    req->format = ZPixmap;
    
    if (_XReply(dpy, (xReply *) &rep, 0, xFalse) == 0 ||
    rep.length == 0) {
        UnlockDisplay(dpy);
        SyncHandle();
        return (False);
    }
    
    nbytes = (long)rep.length << 2;
    _XReadPad(dpy, image->data, nbytes);
    
    UnlockDisplay(dpy);
    SyncHandle();
    return (True);
}

/*
 * timer callback for capturing
 */
Boolean
TCbCaptureX11(XtPointer xtp, XtIntervalId *id) {
    Job *job = (Job *) xtp;
    static char file[PATH_MAX+1];
    static XImage *image = NULL;
    static void *fp = NULL;
    long time, time1;
    struct timeval curr_time;
    static Display *dpy;
    Screen *scr;
    
    #ifdef DEBUG2
    printf("TCbCapture() pic_no=%d - state=%i\n", job->pic_no, job->state);
    #endif
    if ((job->state & VC_PAUSE) && !(job->state & VC_STEP)) {
        XVC_AddTimeout(job->time_per_frame, job->capture, job);
        
    } else if (job->state & VC_REC) {
        if (job->max_frames && (
        (job->pic_no - job->start_no )> job->max_frames - 1)) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("Stopped! pic_no=%d max_frames=%d\n",
                job->pic_no, job->max_frames);
            if (!(job->flags & FLG_RUN_VERBOSE))
                XVC_ChangeGUILabel(job->pic_no);

            if (job->flags & FLG_AUTO_CONTINUE) job->state |= VC_CONTINUE;
            goto CLEAN_X11;
        }
        job->state &= ~VC_STEP;
        gettimeofday(&curr_time, NULL);
        time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
        
        if ((!(job->flags & FLG_MULTI_IMAGE)) ||
        ((job->flags & FLG_MULTI_IMAGE) && (job->state & VC_START))) {
            if ((job->flags & FLG_MULTI_IMAGE) != 0) {
                sprintf(file, job->file, job->movie_no);
            } else {
                sprintf(file, job->file, job->pic_no);
            }
            fp = (*job->open)(file, job->open_flags);
            if (!fp) {
                perror(file);
                job->state = VC_STOP;
                return;
            }
        }
        if (job->state & VC_START) {
            /*
             * the first time this procedure is started
             * we must create a new image ..
             *
             */
            int x, y;
            
            if ( ! (job->flags & FLG_NOGUI ) ) {
                scr = job->win_attr.screen;
                dpy = DisplayOfScreen(scr);
            } else {
                dpy = XOpenDisplay(NULL);
            }
            
            if (job->mouseWanted > 0) {
                getCurrentPointer(&x, &y, job);
            }
            
            image = XGetImage(dpy, RootWindow(dpy, DefaultScreen(dpy)),
            job->area->x, job->area->y,
            job->area->width, job->area->height, AllPlanes, ZPixmap);
            if (!image) {
                printf("Can't get image: %dx%d+%d+%d\n", job->area->width,
                job->area->height, job->area->x, job->area->y);
                job->state = VC_STOP;
            } else {
                if (job->mouseWanted > 0) {
                    paintMousePointer(&x, &y, job, image);
                }
                (*job->save) (fp, image, job);
                job->state &= ~VC_START;
            }
        } else {
            int x, y;
            
            /* just read new data in the image structure
             */
            if (job->mouseWanted > 0) {
                getCurrentPointer(&x, &y, job);
            }
            
            if (XGetZPixmap(dpy,
            RootWindow(dpy, DefaultScreen(dpy)),
            image, job->area->x, job->area->y)) {
                if (job->mouseWanted > 0) {
                    paintMousePointer(&x, &y, job, image);
                }
                (*job->save) (fp, image, job);
            } else {
                printf("XGetZPixmap returned 'False'!\n");
            }
        }
        if (!(job->flags & FLG_MULTI_IMAGE))
            (*job->close) (fp);
        
        /* substract the time we needed for creating and saving
         * the frame to the file
         */
        gettimeofday(&curr_time, NULL);
        time1 = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
        // update monitor widget, here time is the time the capture took
	// time == 0 resets led_meter
	if (! time1) time1 = 1;
        XVC_FrameMonitor(job, time1);
//	printf("time: %i time_per_frame: %i\n", time1, job->time_per_frame);
	// calculate the remaining time we have till capture of next frame
        time1 = job->time_per_frame - time1;
        
        if ( time1 > 0 ) {
	    // get time again because updating frame drop meter took some time
            gettimeofday(&curr_time, NULL);
            time = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
            time = job->time_per_frame - time;
        } else {
            time = time1;
        }
        if (time < 0) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("missing %ld milli secs (%d needed per frame), pic no %d\n",
                time, job->time_per_frame, job->pic_no);
        }
	if ( time < 2 ) time = 2;

        XVC_AddTimeout(time, job->capture, job);

        job->pic_no += job->step;
        if (time >= 2 )
            XVC_ChangeGUILabel(job->pic_no);

    } else {
        int orig_state;
        
        /* clean up */
        CLEAN_X11:
            orig_state = job->state; // store state here
            
            /* may be the last update failed .. so do it here before stop
             */
            XVC_ChangeGUILabel(job->pic_no);
            XVC_FrameMonitor(job, 0);
            
            job->state = VC_STOP;
            if (image) {
                XDestroyImage(image);
                image = NULL;
            }
            /* set the sensitive stuff for the control panel
             * if we don't autocontinue */
            if ((orig_state & VC_CONTINUE) == 0) 
                XVC_StopCapture(job);
            
            /* clean up the save routines in xtoXXX.c
             */
            if (job->clean)
                (*job->clean) (job);
            if (job->flags & FLG_MULTI_IMAGE)
                if (fp)
                    (*job->close) (fp);
            fp = NULL;
            
            if ((orig_state & VC_CONTINUE) == 0) {
            /* after this we're ready to start recording again */
                job->state |= VC_READY;
            } else {
                job->movie_no += 1;
                job->pic_no = job->start_no;
                job->state &= ~VC_STOP;
                job->state |= VC_START;
                job->state |= VC_REC;
                XVC_StartCapture(job);
                return;
            }

            if ( job->flags & FLG_NOGUI ) {
                XCloseDisplay(dpy);
                exit(0);
            }
            
    }
    
    return FALSE;
}


#ifdef HAVE_SHMAT
/*
 * timer callback for capturing with shared memory
 */
Boolean
TCbCaptureSHM(XtPointer xtp, XtIntervalId *id) {
    Job *job = (Job *) xtp;
    static char file[PATH_MAX+1];
    static XImage *image = NULL;
    static XShmSegmentInfo shminfo;
    static void *fp = NULL;
    Visual *visual = job->win_attr.visual;
    unsigned int depth = job->win_attr.depth;
    long time, time1;
    struct timeval curr_time;
    static Display *dpy;
    Screen *scr;
    
    #ifdef DEBUG2
    printf("TCbCaptureSHM() pic_no=%d flags=%d state=%i\n", job->pic_no, job->flags, job->state);
    printf("VC_REC %i - VC_STOP %i\n", ( job->state & VC_REC ), (job->state & VC_STOP ) );
    #endif
    if ((job->state & VC_PAUSE) && !(job->state & VC_STEP)) {
        XVC_AddTimeout(job->time_per_frame, job->capture, job);
        
    } else if (job->state & VC_REC) {
        // trace mouse pointer ...
        int x, y;
        
        if (job->max_frames && (
        (job->pic_no - job->start_no )> job->max_frames - 1)) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("Stopped! pic_no=%d max_frames=%d\n",
                job->pic_no, job->max_frames);
            if (!(job->flags & FLG_RUN_VERBOSE))
                XVC_ChangeGUILabel(job->pic_no);

            if (job->flags & FLG_AUTO_CONTINUE) job->state |= VC_CONTINUE;
            goto CLEAN_SHM;
        }
        job->state &= ~VC_STEP;
        gettimeofday(&curr_time, NULL);
        time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
        
        if ((!(job->flags & FLG_MULTI_IMAGE)) ||
        ((job->flags & FLG_MULTI_IMAGE) && (job->state & VC_START))) {
            
            if (job->flags & FLG_MULTI_IMAGE)
                sprintf(file, job->file, job->movie_no);
            else
                sprintf(file, job->file, job->pic_no);
            fp = (*job->open)(file, job->open_flags);
            if (!fp) {
                perror(file);
                job->state = VC_STOP;
                return;
            }
        }
        if (job->state & VC_START) {
            /*
             * the first time this procedure is called so
             * we must create a new ximage ..
             */
            if ( ! (job->flags & FLG_NOGUI ) ) {
                scr = job->win_attr.screen;
                dpy = DisplayOfScreen(scr);
            } else {
                dpy = XOpenDisplay(NULL);
                scr = DefaultScreenOfDisplay(dpy);
            }
            
            if (job->mouseWanted > 0) {
                getCurrentPointer(&x, &y, job);
            }
            
            image = XShmCreateImage(dpy, visual, depth, ZPixmap, NULL,
            &shminfo, job->area->width, job->area->height);
            if (!image) {
                printf("Can't get image: %dx%d+%d+%d\n", job->area->width,
                job->area->height, job->area->x, job->area->y);
                job->state = VC_STOP;
                //                CbStop(NULL, NULL, NULL);
                return;
            }
            shminfo.shmid = shmget(IPC_PRIVATE,
            image->bytes_per_line * image->height, IPC_CREAT|0777);
            if (shminfo.shmid == -1) {
                printf("Fatal: Can't get shared memory!\n");
                exit(1);
            }
            shminfo.shmaddr = image->data = shmat(shminfo.shmid, 0, 0);
            shminfo.readOnly = False;
            
            if (XShmAttach(dpy, &shminfo) == 0) {
                printf("Fatal: Failed to attach shared memory!\n");
                /* needs some better error subroutine :) */
                exit(1);
            }
            
            if (job->mouseWanted > 0) {
                paintMousePointer(&x, &y, job, image);
            }
            
        }
        /* just read new data in the image structure
         */
        if (job->mouseWanted) {
            getCurrentPointer(&x, &y, job);
        }
        
        if (XShmGetImage(dpy,
        RootWindow(dpy, DefaultScreen(dpy)),
        image, job->area->x, job->area->y, AllPlanes)) {
            if (job->mouseWanted) {
                paintMousePointer(&x, &y, job, image);
            }
            
            (*job->save) (fp, image, job);
            job->state &= ~VC_START;
        } else {
            printf("XShmGetImage() returned 'False'!\n");
        }
        if (!(job->flags & FLG_MULTI_IMAGE))
            (*job->close) (fp);
        else if (job->flags & FLG_SYNC) {
            if (job->open == (void *(*)(char *, char*))fopen)
                fdatasync(fileno(fp));
        }
        
        /* substract the time we needed for creating and saving
         * the frame to the file
         */
        gettimeofday(&curr_time, NULL);
        time1 = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
        // update monitor widget, here time is the time the capture took
	// time == 0 resets led_meter
	if (! time1) time1 = 1;
        XVC_FrameMonitor(job, time1);
//	printf("time: %i time_per_frame: %i\n", time1, job->time_per_frame);
	// calculate the remaining time we have till capture of next frame
        time1 = job->time_per_frame - time1;
        
        if ( time1 > 0 ) {
	    // get time again because updating frame drop meter took some time
            gettimeofday(&curr_time, NULL);
            time = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
            time = job->time_per_frame - time;
        } else {
            time = time1;
        }
        if (time < 0) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("missing %ld milli secs (%d needed per frame), pic no %d\n",
                time, job->time_per_frame, job->pic_no);
        }
	if ( time < 2 ) time = 2;

        XVC_AddTimeout(time, job->capture, job);

        job->pic_no += job->step;
        if (time >= 2 )
            XVC_ChangeGUILabel(job->pic_no);

    } else {
        int orig_state;

        /* clean up */
        CLEAN_SHM:
            orig_state = job->state; // store state here

            // maybe the last update didn't succeed
            XVC_ChangeGUILabel(job->pic_no);
            XVC_FrameMonitor(job, 0);
            
            job->state = VC_STOP;
            if (image) {
                XShmDetach(dpy, &shminfo);
                image->data = NULL;
                XDestroyImage(image);
                image = NULL;
                shmdt(shminfo.shmaddr);
                shmctl(shminfo.shmid, IPC_RMID, 0);
            }

            /* set the sensitive stuff for the control panel
             * if we don't autocontinue */
            if ((orig_state & VC_CONTINUE) == 0) 
                XVC_StopCapture(job);
            
            if (job->clean) {
                (*job->clean) (job);
            }
            if (job->flags & FLG_MULTI_IMAGE)
                if (fp)
                    (*job->close) (fp);
            fp = NULL;
            
            if ((orig_state & VC_CONTINUE) == 0) {
            /* after this we're ready to start recording again */
                job->state |= VC_READY;
            } else {
                job->movie_no += 1;
                job->pic_no = job->start_no;
                job->state &= ~VC_STOP;
                job->state |= VC_START;
                job->state |= VC_REC;
                XVC_StartCapture(job);
                return;
            }
             
            if ( job->flags & FLG_NOGUI ) {
                XCloseDisplay(dpy);
                exit(0);
            }
            
    }
    
    return FALSE;
}

#endif /* HAVE_SHMAT */

#ifdef HasVideo4Linux
/*
 * timer callback for capturing direct from bttv driver (only linux)
 */
#ifndef linux
#error only for linux
#endif
/* from bttv.h */

Boolean
TCbCaptureV4L(XtPointer xtp, XtIntervalId id *) {
    Job *job = (Job *) xtp;
    static char file[PATH_MAX+1];
    static XImage *image = NULL;
    static void *fp = NULL;
    static VIDEO *video = 0;
    static int size;
    static struct video_mmap	vi_mmap;
    static struct video_mbuf	vi_memb;
    static struct video_picture	vi_pict;
    long time, time1;
    struct timeval curr_time;
    Display *dpy;
    Screen *scr;
    
    if ( ! (job->flags & FLG_NOGUI ) ) {
        scr = job->win_attr.screen;
        dpy = DisplayOfScreen(scr);
    } else {
        dpy = XOpenDisplay(NULL);
    }
    
    #ifdef DEBUG2
    printf("TCbCaptureV4L() pic_no=%d\n", job->pic_no);
    #endif
    if ((job->state & VC_PAUSE) && !(job->state & VC_STEP)) {
        XVC_AddTimeout(job->time_per_frame, job->capture, job);
        
    } else if (job->state & VC_REC) {
        if (job->max_frames && (
        (job->pic_no - job->start_no )> job->max_frames - 1)) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("Stopped! pic_no=%d max_frames=%d\n",
                job->pic_no, job->max_frames);
            if (!(job->flags & FLG_RUN_VERBOSE))
                XVC_ChangeGUILabel(job->pic_no);
            
            if (job->flags & FLG_AUTO_CONTINUE) job->state |= VC_CONTINUE;
            goto CLEAN_V4L;
        }
        job->state &= ~VC_STEP;
        gettimeofday(&curr_time, NULL);
        time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
        
        if ((!(job->flags & FLG_MULTI_IMAGE)) ||
        ((job->flags & FLG_MULTI_IMAGE) && (job->state & VC_START))) {
            if (job->flags & FLG_MULTI_IMAGE)
                sprintf(file, job->file, job->movie_no);
            else
                sprintf(file, job->file, job->pic_no);
            fp = (*job->open)(file, job->open_flags);
            if (!fp) {
                perror(file);
                job->state = VC_STOP;
                return;
            }
        }
        if (job->state & VC_START) {
            /*
             * the first time this procedure is started
             * we must prepare some stuff ..
             */
            if (!job->bpp) /* use depth of the default window */
                job->bpp = job->win_attr.depth;
            
            sync(); /* remove if your bttv driver runs stable .. */
            video = video_open(job->video_dev, O_RDWR);
            if (!video) {
                perror(job->video_dev);
                goto CLEAN_V4L;
            }
            vi_pict.depth = 0;
            /* read default values for hue, etc.. */
            ioctl(video->fd, VIDIOCGPICT, &vi_pict);
            
            printf("%d->%d %d\n", job->bpp, vi_pict.depth, vi_pict.palette);
            switch (job->bpp) {
                case 24:
                    vi_mmap.format = vi_pict.palette = VIDEO_PALETTE_RGB24;
                    break;
                case 16:
                    vi_mmap.format = vi_pict.palette = VIDEO_PALETTE_RGB565;
                    break;
                case 15:
                    vi_mmap.format = vi_pict.palette = VIDEO_PALETTE_RGB555;
                    break;
                default:
                    printf("Fatal: unsupported bpp (%d)\n", job->bpp);
                    exit(3);
                    break;
            }
            ioctl(video->fd, VIDIOCSPICT, &vi_pict);
            printf("%d->%d %d\n", job->bpp, vi_pict.depth, vi_pict.palette);
            
            
            vi_memb.size = 0;
            ioctl(video->fd, VIDIOCGMBUF, &vi_memb);
            printf("%d %d %d\n", vi_memb.size, vi_memb.frames, vi_memb.offsets);
            
            image = (XImage *)XtMalloc(sizeof(XImage));
            if (!image) {
                printf("Can't get image: %dx%d+%d+%d\n", job->area->width,
                job->area->height, job->area->x, job->area->y);
                goto CLEAN_V4L;
            }
            switch (job->bpp) {
                case 24:
                    image->red_mask = 0xFF0000;
                    image->green_mask = 0x00FF00;
                    image->blue_mask = 0x0000FF;
                    break;
                case 16:
                    image->red_mask = 0x00F800;
                    image->green_mask = 0x0007E0;
                    image->blue_mask = 0x00001F;
                    break;
                case 15:
                    image->red_mask = 0x00F800;
                    image->green_mask = 0x0007E0;
                    image->blue_mask = 0x00001F;
                    break;
                default:
                    printf("Fatal: unsupported bpp (%d)\n", job->bpp);
                    exit(3);
                    break;
            }
            image->width = job->area->width;
            image->height= job->area->height;
            image->bits_per_pixel = job->bpp;
            image->bytes_per_line = job->bpp/8 * image->width;
            image->byte_order = MSBFirst;
            size = image->width * image->height * job->bpp;
            video->size = vi_memb.size;
            video_mmap(video, 1);
            if (video->mmap == NULL) {
                perror("mmap()");
                goto CLEAN_V4L;
            }
            
            vi_mmap.width = image->width;
            vi_mmap.height= image->height;
            vi_mmap.frame = 0;
            image->data = video->mmap;
        }
        /* just read new data in the image structure
         */
        if (ioctl(video->fd, VIDIOCMCAPTURE, &vi_mmap) < 0) {
            perror("ioctl(capture)");
            /* if (vb.frame) vb.frame = 0; else vb.frame = 1; */
            goto CLEAN_V4L;
        }
        printf("syncing ..\n");
        if (ioctl(video->fd, VIDIOCSYNC, vi_mmap.frame) < 0) {
            perror("ioctl(sync)");
        }
        printf("synced()\n");
        
        (*job->save) (fp, image, job);
        job->state &= ~VC_START;
        
        if (!(job->flags & FLG_MULTI_IMAGE))
            (*job->close) (fp);
        
        /* substract the time we needed for creating and saving
         * the frame to the file
         */
        gettimeofday(&curr_time, NULL);
        time1 = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
        // update monitor widget, here time is the time the capture took
	// time == 0 resets led_meter
	if (! time1) time1 = 1;
        XVC_FrameMonitor(job, time1);
//	printf("time: %i time_per_frame: %i\n", time1, job->time_per_frame);
	// calculate the remaining time we have till capture of next frame
        time1 = job->time_per_frame - time1;
        
        if ( time1 > 0 ) {
	    // get time again because updating frame drop meter took some time
            gettimeofday(&curr_time, NULL);
            time = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
            time = job->time_per_frame - time;
        } else {
            time = time1;
        }
        if (time < 0) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("missing %ld milli secs (%d needed per frame), pic no %d\n",
                time, job->time_per_frame, job->pic_no);
        }
	if ( time < 2 ) time = 2;

        XVC_AddTimeout(time, job->capture, job);

        job->pic_no += job->step;
        if (time >= 2 )
            XVC_ChangeGUILabel(job->pic_no);

    } else {
        int orig_state;

        /* clean up */
        CLEAN_V4L:
            orig_state = job->state; // store state here

            // maybe the last update didn't succeed
            XVC_ChangeGUILabel(job->pic_no);
            XVC_FrameMonitor(job, 0);
            
            job->state = VC_STOP;
            if (image) {
                XtFree((char*)image);
                image = NULL;
            }
            if (video) {
                video_mmap(video, 0);
                video_close(video);
                video = NULL;
            }

            /* set the sensitive stuff for the control panel
             * if we don't autocontinue */
            if ((orig_state & VC_CONTINUE) == 0) 
                XVC_StopCapture(job);
            
            /* clean up the save routines in xtoXXX.c
             */
            if (job->clean)
                (*job->clean) (job);
            if (job->flags & FLG_MULTI_IMAGE)
                if (fp)
                    (*job->close) (fp);
            fp = NULL;

            if ((orig_state & VC_CONTINUE) == 0) {
            /* after this we're ready to start recording again */
                job->state |= VC_READY;
            } else {
                job->movie_no += 1;
                job->pic_no = job->start_no;
                job->state &= ~VC_STOP;
                job->state |= VC_START;
                job->state |= VC_REC;
                XVC_StartCapture(job);
                return;
            }
            
    }

    /* after this we're ready to start recording again */
    job->state |= VC_READY;
             
    if ( job->flags & FLG_NOGUI && ( ! is_filename_mutable(job->file) ) ) {
        XCloseDisplay(dpy);
    }
    
    return FALSE;
}
#endif /* HasVideo4Linux */

#ifdef HasDGA
/*
 * direct graphic access
 * this doesn't work until now and may be removed in future..!?
 *
 * IT HAS ALSO NOT BEEN REWRITTEN FOR GTK GUI SUPPORT
 */
Boolean
TCbCaptureDGA(XtPointer xtp, XtIntervalId id *) {
    Job *job = (Job *) xtp;
    static char file[PATH_MAX+1];
    static XImage *image = NULL;
    static void *fp;
    static int size;
    static Display *dpy;
    long time;
    struct timeval curr_time;
    
    #ifdef DEBUG2
    printf("TCbCaptureDGA() pic_no=%d state=%d\n", job->pic_no, job->state);
    #endif
    if ((job->state & VC_PAUSE) && !(job->state & VC_STEP)) {
        XtAppAddTimeOut(XtWidgetToApplicationContext(job->toplevel),
        job->time_per_frame, (XtTimerCallbackProc)job->capture, job);
        
    } else if (job->state & VC_REC) {
        if (job->max_frames && (
        (job->pic_no - job->start_no )> job->max_frames - 1)) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("Stopped! pic_no=%d max_frames=%d\n",
                job->pic_no, job->max_frames);
            if (!(job->flags & FLG_RUN_VERBOSE))
                ChangeLabel(job->pic_no);
            goto CLEAN_DGA;
        }
        job->state &= ~VC_STEP;
        gettimeofday(&curr_time, NULL);
        time = curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000;
        
        if ((!(job->flags & FLG_MULTI_IMAGE)) ||
        ((job->flags & FLG_MULTI_IMAGE) && (job->state & VC_START))) {
            if (job->flags & FLG_MULTI_IMAGE)
                sprintf(file, job->file, job->movie_no);
            else
                sprintf(file, job->file, job->pic_no);
            fp = (*job->open)(file, job->open_flags);
            if (!fp) {
                perror(file);
                job->state = VC_STOP;
                return;
            }
        }
        if (job->state & VC_START) {
            /*
             * the first time this procedure is started
             * we must create a new image structure ..
             */
            dpy = XtDisplay(job->toplevel);
            image = (XImage *)XtMalloc(sizeof(XImage));
            if (!image) {
                printf("Can't get image: %dx%d+%d+%d\n", job->area->width,
                job->area->height, job->area->x, job->area->y);
                goto CLEAN_DGA;
            }
            image->width = job->area->width;
            image->height= job->area->height;
            image->bits_per_pixel = 3 * 8;
            image->bytes_per_line = 3 * image->width;
            image->byte_order = MSBFirst;
            size = image->width * image->height;
            { int width, bank, ram;
              char *base;
              XF86DGAGetVideo(dpy, XDefaultScreen(dpy), (char **) &base,
              &width, &bank, &ram);
              image->data = base;
            }
            XF86DGADirectVideo(dpy, XDefaultScreen(dpy), XF86DGADirectGraphics);
            (*job->save) (fp, image, job);
            XF86DGADirectVideo(dpy, XDefaultScreen(dpy), 0);
            job->state &= ~VC_START;
        } else {
            /* just read new data in the image structure
             */
            XF86DGADirectVideo(dpy, XDefaultScreen(dpy), XF86DGADirectGraphics);
            (*job->save) (fp, image, job);
            XF86DGADirectVideo(dpy, XDefaultScreen(dpy), 0);
        }
        if (!(job->flags & FLG_MULTI_IMAGE))
            (*job->close) (fp);
        
        /* substract the time we needed for creating and saving
         * the frame to the file
         */
        gettimeofday(&curr_time, NULL);
        time = (curr_time.tv_sec * 1000 + curr_time.tv_usec / 1000) - time;
        time = job->time_per_frame - time;
        if (time < 0) {
            if (job->flags & FLG_RUN_VERBOSE)
                printf("missing %ld milli secs (%d needed per frame), pic no %d\n",
                time, job->time_per_frame, job->pic_no);
            time = 0;
        }
        XtAppAddTimeOut(XtWidgetToApplicationContext(job->toplevel),
        time, (XtTimerCallbackProc)job->capture, job);
        job->pic_no += job->step;
        /* update the label if we have time to do this */
        if (time)
            ChangeLabel(job->pic_no);
    } else {
        /* clean up */
        CLEAN_DGA:
            /* may be the last update failed .. so do it here before stop
             */
            ChangeLabel(job->pic_no);
            job->state = VC_STOP;
            if (image) {
                XtFree((char*)image);
                image = NULL;
            }
            XF86DGADirectVideo(dpy, XDefaultScreen(dpy), 0);
            /* set the sensitive stuff for the control panel
             */
            CbStop(NULL, NULL, NULL);
            
            /* clean up the save routines in xtoXXX.c
             */
            if (job->clean)
                (*job->clean) (job);
            if (job->flags & FLG_MULTI_IMAGE)
                if (fp)
                    (*job->close) (fp);
            fp = NULL;
            
            return FALSE;
    }
    return TRUE;
}
#endif /* HasDGA */
