/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import xnap.io.*;
import xnap.net.ProxySupport;
import xnap.user.UserManager;
import xnap.util.event.*;

import java.beans.*;
import java.util.*;
import org.apache.log4j.Logger;

public class Updater implements PropertyChangeListener
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    private static Updater singleton = new Updater();
    private static boolean zigaDllLoaded = false;
    private static Logger logger = Logger.getLogger(Updater.class);

    private Preferences prefs = Preferences.getInstance();

    // --- Initializer ---

    static {
	if (Preferences.isWindowsOS) {
	    try {
		System.loadLibrary("ziga");
		zigaDllLoaded = true;
	    } catch (UnsatisfiedLinkError e) {
		logger.warn("could not load ziga.dll", e);
	    }
	}
    }

    //--- Constructor(s) ---

    private Updater()
    {
    }

    //--- Method(s) ---

    public static synchronized void start() 
    {
	singleton.init();
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();

	logger.debug("Preferences.propertyChange: " + p + " " 
		     + e.getOldValue() + " -> " + e.getNewValue());
	
	if (p.equals("useSocksProxy")
	    || p.equals("socksProxyHost")
	    || p.equals("socksProxyPort")) {
	    updateSocksProxy();
	}
	else if (p.equals("useHttpProxy")
		 || p.equals("httpProxyHost")
		 || p.equals("httpProxyPort")) {
	    updateHttpProxy();
	}
	else if (p.equals("downloadThrottle") || p.equals("uploadThrottle")
		 || p.equals("throttleDownloads") 
		 || p.equals("throttleUploads")) {
	    updateThrottle();
	}
    }

    private void init()
    {
	updateSocksProxy();
	updateHttpProxy();
	updateThrottle();

	// make sure repositories are uptodate
	Repository.getInstance();
	ResumeRepository.getInstance();
	UserManager.getInstance();

	prefs.addPropertyChangeListener(this);

	// make sure everybody is instantiated
	DownloadQueue.getInstance();
	UploadQueue.getInstance();
	SearchManager.getInstance();

	int interval = Preferences.getInstance().getUpdaterInterval() * 1000;
	Timer t = new Timer();
	t.schedule(new UpdateTask(), interval, interval);
    }

    /**
     * Returns true, if operating system is Mac OS X and a version below 
     * Jaguar (10.2).
     */
    public static boolean hasMacOSXModalDialogBug()
    {
	if (isMacOSX()) {
	    String ver = System.getProperty("os.version", "");
	    return VersionParser.compare(ver, "10.2.0") < 0;
	}

	return false;
    }

    /**
     * Disable tooltips on Mac OS X and JDK 1.3.*.
     *
     * @see <a href="http://developer.apple.com/qa/qa2001/qa1153.html">Apple</a>
     */
    public static boolean hasMacOSXToolTipsBug()
    {
	if (isMacOSX()) {
	    String ver = System.getProperty("java.version", "");
	    return VersionParser.compare(ver, "1.4.0") < 0;
	}

	return false;
    }

    public static boolean isMacOSX()
    {
	return ("Mac OS X").equals(System.getProperty("os.name"));
    }
	
    public static boolean isZigaDllLoaded()
    {
	return zigaDllLoaded;
    }

    private void updateHttpProxy()
    {
	ProxySupport.removeHttp();

	if (prefs.getUseHttpProxy()) {
	    ProxySupport.putHttp(prefs.getHttpProxyHost(), 
				 prefs.getHttpProxyPort());
	}
    }

    private void updateSocksProxy()
    {
	ProxySupport.removeSocks();

	if (prefs.getUseSocksProxy()) {
	    ProxySupport.putSocks(prefs.getSocksProxyHost(), 
				  prefs.getSocksProxyPort());
	}
    }

    private void updateThrottle()
    {
	boolean throttle = prefs.getThrottleDownloads();
	long b = prefs.getDownloadThrottle() * 1024;
	ThrottledInputStream.setBandwidth((throttle) ? b : 0);

	throttle = prefs.getThrottleUploads();
	b = prefs.getUploadThrottle() * 1024;
	ThrottledOutputStream.setBandwidth((throttle) ? b : 0);
    }

    private class UpdateTask extends TimerTask
    {

	public void run()
	{
	    Debug.log("Updater: running update");
	    Repository.getInstance().updateLater();
	}

    }
}
