/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.util;

import xnap.*;
import xnap.net.*;
import xnap.util.event.StatusListener;
import xnap.util.*;

import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.Iterator;
import org.apache.log4j.Logger;

public class Searcher implements ISearchContainer, Runnable
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected static Logger logger = Logger.getLogger(Searcher.class);

    protected SearchFilter filter;
    protected SearchResultCollector srCollector;
    protected LinkedList searches;

    protected boolean connected = false;
    protected String errStatus = "";
    protected int failedCount;
    protected int finishedCount;
    protected StatusListener listener = null;
    protected Thread runner;
    protected String status = "";
    protected boolean die = false;

    //--- Constructor(s) ---

    public Searcher(ISearch[] searches, SearchFilter filter, 
		    SearchResultCollector srCollector)
    {
	this.searches = new LinkedList(Arrays.asList(searches));	
	this.filter = filter;
	this.srCollector = srCollector;
    }

    public Searcher(ISearch[] searches, SearchFilter filter)
    {
	this(searches, filter, null);
    }

    //--- Method(s) ---

    public void abort()
    {
	die = true;
    }

    public SearchResultCollector getCollector()
    {
	return srCollector;
    }

    public void setCollector(SearchResultCollector newValue)
    {
	srCollector = newValue;
    }

    public Grouper getDefaultGrouper()
    {
	return new FilesizeGrouper();
    }

    public SearchFilter getFilter()
    {
	return filter;
    }

    public int getSize()
    {
	return searches.size() + finishedCount + failedCount;
    }

    public String getStatus()
    {
	return status;
    }

    public synchronized void setStatusListener(StatusListener newValue)
    {
	listener = newValue;
    }

    public synchronized void setStatus(String newValue)
    {
	status = newValue;
	if (listener != null) {
	    listener.setStatus(newValue);
	}
    }

    public void start()
    {
	runner = new Thread(this, "Searcher " + toString());
	runner.start();

	updateStatus();
    }

    public void run()
    {
	for (Iterator i = searches.iterator(); !die && i.hasNext();) {
	    try {
		((ISearch)i.next()).connect();
	    }
	    catch (IOException e) {
		i.remove();
		appendErr(e);
		logger.warn("connect failed", e);
	    }

	    updateStatus();
	}

	connected = true;
	updateStatus();

	while (!die && searches.size() > 0) {
	    for (Iterator i = searches.iterator(); i.hasNext();) {
		ISearch s = (ISearch)i.next();
		try {
		    int c = s.available();
		    if (c == -1) {
			i.remove();
			finishedCount++;
			updateStatus();
		    } 
		    else if (c > 0) {
			for (int j = 0; j < c; j++) {
			    srCollector.add(s.next());
			}
		    }
		}
		catch (IOException e) {
		    i.remove();
		    appendErr(e);
		    updateStatus();
		    logger.warn("search failed", e);
		}
	    }

	    try {
		runner.sleep(125);
	    }
	    catch (InterruptedException e) {
	    }
	}

	// abort remaining searches
	for (Iterator i = searches.iterator(); i.hasNext();) {
	    ((ISearch)i.next()).close();
	    i.remove();
	}

	updateStatus();
    }

    protected void appendErr(Exception e)
    {
	failedCount++;
	errStatus += (((errStatus.length() == 0) ? "" : ", ") 
		      + e.getMessage());
    }

    protected void updateStatus()
    {
	StringBuffer sb = new StringBuffer();
	
	if (searches.size() > 0) {
	    sb.append(XNap.tr("Searches:", 0, 1));
	    sb.append(finishedCount + XNap.tr("succeeded,", 1));
	    sb.append(failedCount + XNap.tr("failed,", 1));
	    sb.append(searches.size() + XNap.tr("pending", 1));
	} 
	else if (finishedCount + failedCount == 0) {
	    sb.append(XNap.tr("No search started. Please load one or more plugins"));
	}
	else {
	    sb.append((XNap.tr("Finished:", 0, 1)));
	    sb.append(finishedCount + failedCount);
	    sb.append(XNap.tr("searches", 1, 0));
	    if (failedCount > 0) {
		sb.append(", ");
		sb.append(failedCount);
		sb.append(XNap.tr("failed,", 1, 0));
	    }
	} 
	if (errStatus.length() > 0) {
	    sb.append(" (");
	    sb.append(errStatus);
	    sb.append(")");
	}
	
	setStatus(sb.toString());
    }

    public String toString()
    {
	return (filter != null) ? filter.toString() : "";
    }
    
}
