/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.util;

import xnap.cmdl.*;
import xnap.net.*;
import xnap.net.event.*;

import java.beans.*;
import java.io.*;
import java.net.*;
import java.util.*;
import org.apache.log4j.Logger;

public class DownloadQueue extends TransferQueue 
    implements PropertyChangeListener  {

    //--- Constant(s) ---

    //--- Data field(s) ---

    private static DownloadQueue singleton = new DownloadQueue();

    //--- Constructor(s) ---

    private DownloadQueue()
    {
	super("DownloadQueue", false);

	logger = Logger.getLogger(DownloadQueue.class);

	prefs.addPropertyChangeListener("limitDownloads", this);
	prefs.addPropertyChangeListener("maxDownloads", this);

	updateLimit();

	Executer.addCommand(new ListDlsCmd());
	Executer.addCommand(new AbortDlCmd());
	Executer.addCommand(new ResumeDlCmd());
    }

    //--- Method(s) ---

    public static DownloadQueue getInstance()
    {
	return singleton;
    }
    
    public synchronized boolean add(IDownloadContainer d, boolean start)
    {
	logger.debug("add download: " + d);

	int i = super.indexOf(d);
	if (i == -1) {
	    super.add(d);
	    return start ? enqueue(d) : true;
	}
	else {
	    Object o = super.get(i);
	    if (o instanceof MultiDownload && d instanceof MultiDownload) {
		// add results from d to o
		MultiDownload download = (MultiDownload)o;
		download.merge((MultiDownload)d);
		return start && !download.isRunning() ? enqueue(download) : true;
	    }
	}

	return false;
    }

    public boolean add(IDownloadContainer d) 
    {
	return add(d, true);
    }

    public boolean delete(IDownloadContainer d) 
    {
	d.delete();
	return true;
    }

    public boolean resume(IDownloadContainer d) 
    {
	return enqueue(d);
    }

    public void resumeAll()
    {
	try { 
	    for (Iterator i = super.iterator(); i.hasNext();) {
		enqueue((ITransferContainer)i.next());
	    }
	}
	catch (ConcurrentModificationException e) {
	    // FIX: we should avoid this
	}
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	updateLimit();
    }
       
    public void updateLimit()
    {
	setMaxTransfers(prefs.getLimitDownloads() ? prefs.getMaxDownloads() : 0);
    }

    protected class ListDlsCmd extends AbstractCommand
    {
	public ListDlsCmd()
	{
	    putValue(CMD, new String[] {"listdownloads", "ldl", "pdown"});
	    putValue(SHORT_HELP, "Shows the download queue.");
	}
	
	public boolean execute(String[] parms)
	{
	    print();
	    return true;
	}
    }

    protected class AbortDlCmd extends AbstractCommand
    {
	public AbortDlCmd()
	{
	    putValue(CMD, new String[] { "abortdownload", "adl" });
	    putValue(PARAMETER, "id");
	    putValue(SHORT_HELP, "Aborts a download.");
	}
	
	public boolean execute(String[] argv)
	    throws SyntaxException, ParameterException
	{
	    if (lastTransfers == null) {
		lastTransfers = getTransfers();
	    }

	    int i = getInt(argv, 0, lastTransfers.length - 1);
	    abort(lastTransfers[i]);
	    
	    return true;
	}
    }

    protected class ResumeDlCmd extends AbstractCommand
    {
	public ResumeDlCmd()
	{
	    putValue(CMD, new String[] { "resumedownload", "rdl" });
	    putValue(PARAMETER, "id");
	    putValue(SHORT_HELP, "Resumes a download.");
	}
	
	public boolean execute(String[] argv)
	    throws SyntaxException, ParameterException
	{
	    if (lastTransfers == null) {
		lastTransfers = getTransfers();
	    }

	    int i = getInt(argv, 0, lastTransfers.length - 1);
	    resume((IDownloadContainer)lastTransfers[i]);
	    
	    return true;
	}
    }

}
