/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.viewer.mp3;

import xnap.XNap;
import xnap.gui.*;
import xnap.util.FileHelper;

import helliker.id3.MP3File;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;

public class MP3InfoPanel extends ViewerPanel {

    //--- Constant(s) ---

    //--- Data field(s) ---

    JTextField jtTitle;
    JTextField jtArtist;
    JTextField jtAlbum;
    JTextField jtYear;
    JTextField jtComment;

    GuessAction acGuessAction = new GuessAction();
    SaveAction acSaveAction = new SaveAction();
    ResetAction acResetAction = new ResetAction();
    
    //--- Constructor(s) ---

    public MP3InfoPanel() 
    {
	JPanel jpTag = new JPanel(new GridBagLayout());
	jpTag.setBorder(new TitledBorder(XNap.tr("ID3 Tag", 1)));
	
	GridBagHelper.addLabel(jpTag, XNap.tr("Title"));
	jtTitle = new JTextField();
	GridBagHelper.add(jpTag, jtTitle);

	GridBagHelper.addLabel(jpTag, XNap.tr("Artist"));
	jtArtist = new JTextField();
	GridBagHelper.add(jpTag, jtArtist);

	GridBagHelper.addLabel(jpTag, XNap.tr("Album"));
	jtAlbum = new JTextField();
	GridBagHelper.add(jpTag, jtAlbum);

	GridBagHelper.addLabel(jpTag, XNap.tr("Year"));
	jtYear = new JTextField();
	GridBagHelper.add(jpTag, jtYear);

	GridBagHelper.addLabel(jpTag, XNap.tr("Comment"));
	jtComment = new JTextField();
	GridBagHelper.add(jpTag, jtComment);

	JPanel jbButtons = new JPanel();
	jbButtons.setLayout(new FlowLayout(FlowLayout.RIGHT));
	
	JButton jbGuess = new JButton(acGuessAction);
	jbButtons.add(jbGuess);

	JButton jbSave = new JButton(acSaveAction);
	jbButtons.add(jbSave);

	JButton jbReset = new JButton(acResetAction);
	jbButtons.add(jbReset);

	setLayout(new BorderLayout());
	add(jpTag, BorderLayout.CENTER);
	add(jbButtons, BorderLayout.SOUTH);
    }

    //--- Method(s) ---

    public void clear()
    {
	setStatus(" ");

	jtTitle.setText("");
	jtAlbum.setText("");
	jtArtist.setText("");
	jtYear.setText("");
	jtComment.setText("");
    }

    /**
     * 
     */
    public void display()
    {
	clear();

	try { 
	    MP3File m = new MP3File(getFile());
	    
	    jtAlbum.setText(m.getAlbum());
	    jtArtist.setText(m.getArtist());
	    jtTitle.setText(m.getTitle());
	    jtYear.setText(m.getYear());
	    jtComment.setText(m.getComment());

	    // why call getFile().canRead()? an IOException should have
	    // occured if we could not read the file
	    setEnabled(getFile().canWrite());
	}
	catch (Exception e) {
	    setEnabled(false);
	    setStatus(XNap.tr("Could not read Id Tag") 
		      + " (" + e.getLocalizedMessage() + ")");
	}
	    
	finished = true;
    }

    /**
     * Writes id3 tags to file.
     */
    protected void save()
    {
	try {
	    MP3File m = new MP3File(getFile());

	    m.setTitle(jtTitle.getText());
	    m.setArtist(jtArtist.getText());
	    m.setAlbum(jtAlbum.getText());
	    m.setYear(jtYear.getText());
	    m.setComment(jtComment.getText());

	    m.writeTags();

	    setStatus(XNap.tr("Changes successfully saved."));
	}
	catch (Exception e) {
	    setStatus(XNap.tr("Could not write ID Tags to file"));
	}
    }

    /**
     * Enables or disables all controls.
     */
    public void setEnabled(boolean enabled)
    {
	super.setEnabled(enabled);
	
	acGuessAction.setEnabled(enabled);
	acSaveAction.setEnabled(enabled);
	acResetAction.setEnabled(enabled);
	 
	jtTitle.setEnabled(enabled);
	jtArtist.setEnabled(enabled);
	jtAlbum.setEnabled(enabled);
	jtYear.setEnabled(enabled);
	jtComment.setEnabled(enabled);
    }

    protected class GuessAction extends AbstractAction
    {

        public GuessAction()
	{
	    putValue(Action.NAME, XNap.tr("Guess"));
	    putValue(Action.SHORT_DESCRIPTION, XNap.tr("Guess Id Tags from filename"));
	    putValue(Action.MNEMONIC_KEY, new Integer('G'));
	    this.setEnabled(false);
	}
	
	public void actionPerformed(ActionEvent event)
	{	    
	    String name = FileHelper.name(getFile().getName());
	    
	    StringTokenizer st = new StringTokenizer(name, "-");

	    /* probably it's just the title */
	    if (st.countTokens() == 1) {
		jtTitle.setText(st.nextToken().trim());
	    }
	    /* artist and title */
	    else if (st.countTokens() == 2) {
		jtArtist.setText(st.nextToken().trim());
		jtTitle.setText(st.nextToken().trim());
	    }
	    /* artist, album and title */
	    else if (st.countTokens() == 3) {
		jtArtist.setText(st.nextToken().trim());
		jtAlbum.setText(st.nextToken().trim());
		jtTitle.setText(st.nextToken().trim());
	    }
	}
	
    }

    protected class ResetAction extends AbstractAction
    {

        public ResetAction()
	{
	    putValue(Action.NAME, XNap.tr("Reset"));
	    putValue(Action.SHORT_DESCRIPTION, XNap.tr("Reset Id Tag Changes"));
	    putValue(Action.MNEMONIC_KEY, new Integer('R'));
	    this.setEnabled(false);
	}
	
	public void actionPerformed(ActionEvent event)
	{	    
	    display();
	}

    }

    protected class SaveAction extends AbstractAction
    {

        public SaveAction()
	{
	    putValue(Action.NAME, XNap.tr("Save"));
	    putValue(Action.SHORT_DESCRIPTION, XNap.tr("Save new Id Tag"));
	    putValue(Action.MNEMONIC_KEY, new Integer('A'));
	    this.setEnabled(false);
	}
	
	public void actionPerformed(ActionEvent event)
	{
	    save();
	}
    }        
}
