/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.nap.net.msg;

import xnap.plugin.nap.Plugin;
import xnap.plugin.nap.net.*;
import xnap.plugin.nap.util.Connector;
import xnap.plugin.nap.util.NapPreferences;
import xnap.plugin.nap.util.PrivateChannel;
import xnap.plugin.nap.net.msg.client.ClientMessage;
import xnap.plugin.nap.net.msg.client.LoginMessage;
import xnap.plugin.nap.net.msg.client.NewUserLoginMessage;
import xnap.plugin.nap.net.msg.server.MessageListener;
import xnap.plugin.nap.net.msg.server.ServerMessage;
import xnap.util.ChatManager;
import xnap.util.EventVector;
import xnap.util.Preferences;
import xnap.util.UploadQueue;

import java.io.IOException;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.LinkedList;

public class MessageHandler {

    //--- Constant(s) ---

    //--- Data field(s) ---

    /**
     * Maps message types to <code>MessageListener</code> objects.
     */
    protected static Hashtable listeners = new Hashtable();
    /**
     * Maps <code>User</code> objects to <code>PrivateChannel</code> objects.
     */
    protected static Hashtable privateChannels = new Hashtable();

    /**
     * Convenience access.
     */
    protected static UploadQueue ulQueue = UploadQueue.getInstance();

    /**
     * Maps nick + filename to <code>Server</code> object.
     */
    protected static Hashtable uploadRequests = new Hashtable();

    /**
     * Stores <code>Server</code> objects. 
     */
    protected static LinkedList browseRequests = new LinkedList();

    /**
     * Queues messages.
     */
    protected static LinkedList sendQueue = new LinkedList();
    
    //--- Method(s) ---

    public static void login(Server server, boolean newUser) throws IOException
    {
	Preferences prefs = Preferences.getInstance();
	NapPreferences napPrefs = NapPreferences.getInstance();

	String nick = server.getUsername();
	String password = server.getPassword();
	String email = server.getEmail();
	String info = napPrefs.getClientInfo();
	int linkSpeed = prefs.getLinkType();
	
	if (newUser) {
	    server.send(new NewUserLoginMessage(nick, password, 
				server.getLocalPort(), info,linkSpeed, email));
	}
	else {
	    server.send(new LoginMessage(nick, password, 
				 server.getLocalPort(), info, linkSpeed));
	}
    }

    /**
     * Asynchronously sends <code>msg</code>.
     */
    public static void send(Server server, ClientMessage msg,
			    boolean lowPriority)
    {
	MessageSender.send(server, msg, lowPriority);
    }

    public static void send(Server server, ClientMessage msg)
    {
	send(server, msg, false);
    }

    /**
     * Sends <code>msg</code> to all connected servers.
     */
    public static void send(ClientMessage msg)
    {
	MessageSender.send(msg);
    }

    public static void sendLater(Server server, ClientMessage msg)
    {
	MessageSender.sendLater(server, msg);
    }

    public static void sendPending(Server server)
    {
	MessageSender.sendPending(server);
    }

    /**
     * Let's handle message itself first and then sends it to all listeners. A
     * listener can consume the message to prevent other listeners from
     * getting it.
     */
    public static void handle(ServerMessage msg)
    {
	Server server = msg.getServer();

	msg.received();

	Object[] listeners = getListeners(msg.type).toArray();
	for (int i = 0; i < listeners.length && !msg.isConsumed(); i++) {
	    ((MessageListener)listeners[i]).messageReceived(msg);
	}
    }	

    public static synchronized void addBrowseRequest(Server server)
    {
	browseRequests.addLast(server);
    }

    public static void addUploadRequest(String nick, String filename, 
					Server server)
    {
	uploadRequests.put(nick + filename, server);
    }

    public static PrivateChannel createPrivateChannel(User user, boolean add)
    {
	PrivateChannel pc = (PrivateChannel)privateChannels.get(user);
	if (pc == null) {
	    pc = new PrivateChannel(user);
	    pc.join();
	    privateChannels.put(user, pc);
	    if (add) {
		ChatManager.getInstance().addChannel(pc);
	    }
	}
	return pc;
    }

    public static PrivateChannel getPrivateChannel(User user)
    {
	return createPrivateChannel(user, false);
    }

    /**
     * Called by <code>DirectBrowseUpload</code> if an incoming 
     * connect is registered.
     */
    public static synchronized Server removeBrowseRequest()
    {
	return (browseRequests.isEmpty() 
		? null 
		: (Server)browseRequests.removeFirst());
    }

    public static void removePrivateChannel(PrivateChannel pc)
    {
	privateChannels.remove(pc.getUser());
    }

    /**
     * Called by <code>Upload</code> if an incoming connect is registered.
     */
    public static Server removeUploadRequest(String nick, String filename)
    {
	return (Server)uploadRequests.remove(nick + filename);
    }

    public static void subscribe(int type, MessageListener listener)
    {
	getListeners(type).add(listener);
    }

    public static void unsubscribe(int type, MessageListener listener) 
    {
	getListeners(type).remove(listener);
    }

    /**
     * Returns MsgQueue for corresponding msgs. Creates new MsgQueue 
     * if it doesn't exist yet.
     */
    protected static LinkedList getListeners(int type) 
    {
	LinkedList list = (LinkedList)listeners.get(new Integer(type));

	if (list == null) {
	    list = new LinkedList();
	    listeners.put(new Integer(type), list);
	}		
	
	return list;
    }

    //      public Msg waitForMsg(MsgStream ms, long timeout, int id)
    //      {
    //          long startTime = System.currentTimeMillis();
    //          while (System.currentTimeMillis() - startTime < timeout) { 
    //  	    long timeLeft = timeout - (System.currentTimeMillis() 
    //  				       - startTime);
		
    //  	    if ((timeLeft < 0) || !ms.hasNext(timeLeft)) {
    //  		return null;
    //  	    }

    //  	    Msg msg = ms.next();
    //  	    if (msg.getId() == id) {
    //  		return msg;
    //  	    }
    //  	}

    //  	return null;
    //      }

}
