/*
 * 03/30/2001
 *
 * PongMessage.java
 * Copyright (C) 2001 Frederik Zimmer
 * tristian@users.sourceforge.net
 * http://sourceforge.net/projects/ziga/
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package xnap.plugin.gnutella.net;

import xnap.plugin.gnutella.io.*;

import java.io.IOException;

public class PongMessage extends Message 
{
    private static final int PAYLOAD_LENGTH = 14;
    protected int port;
    protected byte[] ip;
    protected int numOfFiles;
    protected int numOfKB;


    public PongMessage(GUID id, int port, byte[] ip, int numOfFiles, 
		       int numOfKB) 
    {
	super(id, DescriptorHeader.PONG, PAYLOAD_LENGTH);
	
	if (port <= 0 || port > 0xFFFF) {
	    throw new IllegalArgumentException("Invalid port " + port);
	}
	this.port = port;
	this.ip = ip;
	this.numOfFiles = numOfFiles;
	this.numOfKB = numOfKB;
    }
    
    public PongMessage(DescriptorHeader header, DataInputStream in) 
	throws IOException, InvalidMessageException 
    {
	super(header);
	if (header.payloadLength != PAYLOAD_LENGTH) {
	    throw new InvalidMessageException("Payload-length too high: "
					      + header.payloadLength);
	}
	read(in);
    }
	
    public void write(DataOutputStream out)
	throws IOException 
    {
	super.write(out);
	/* fix me */

	out.writeShort((short)port);
	out.write(ip);
	out.writeInt(numOfFiles);
	out.writeInt(numOfKB);
    }
	
    public void read(DataInputStream in)
	throws IOException, InvalidMessageException 
    {
	ip = new byte[4];
		
	port = in.readUShort();
	
	if (port <= 0) {
	    throw new InvalidMessageException("Invalid port: " + port);
	}
		
	in.readFully(ip);
	numOfFiles = in.readInt();
	numOfKB = in.readInt();
    }

    public int getNumberOfFiles()
    {
	return numOfFiles;
    }
    
    public int getNumberOfKBs()
    {
	return numOfKB;
    }
	
    public String getIP() {

	if (ip.length != 4) {
	    return null;
	}
	
	StringBuffer sb = new StringBuffer();
	
	sb.append(ip[0] & 0xFF);
	sb.append('.');
	sb.append(ip[1] & 0xFF);
	sb.append('.');
	sb.append(ip[2] & 0xFF);
	sb.append('.');
	sb.append(ip[3] & 0xFF);
	
	return sb.toString();
    }
	
    public int getPort() {
	return port;
    }
	
    public boolean equals(Object obj) {
	if (obj == null || !(obj instanceof PongMessage)) {
	    return false;
	}
	PongMessage msg = (PongMessage) obj;
	
	return (header.messageID.equals(msg.header.messageID) 
		&& ip[0] == msg.ip[0] && ip[1] == msg.ip[1]
		&& ip[2] == msg.ip[2] && ip[3] == msg.ip[3]
		&& port == msg.port);
    }
}
