/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gnutella.net;

import xnap.plugin.gnutella.util.*;
import xnap.*;
import xnap.io.*;
import xnap.net.*;
import xnap.net.event.*;
import xnap.util.*;
import xnap.util.event.*;

import java.beans.*;
import java.io.*;
import java.util.*;
import java.text.*;
import java.net.*;
import org.apache.log4j.Logger;

public class Connections extends EventVector 
    implements Runnable, PropertyChangeListener, StatusChangeListener
{
    
    //--- Constant(s) ---
    
    public static final int LOGIN_INTERVAL = 300000;

    //--- Data field(s) ---

    private static Connections instance;
    protected static Logger logger = Logger.getLogger(Connections.class);

    protected transient PropertyChangeSupport propertyChange
	= new PropertyChangeSupport(this);
    
    private Preferences prefs = Preferences.getInstance();
    private GnuPreferences gnuPrefs = GnuPreferences.getInstance();
    
    private int maxConnect = gnuPrefs.getMaxConnections();
    private int maxTrying = 2 * maxConnect;
    private int connectedCount = 0;
    private int tryingCount = 0;
    
    private boolean die = false;
    private Thread runner = null;
    private Timer timer;
    public final int DISCONNECT_INTERVAL = 10 * 1000;
    private boolean enabled = false;

    private ListenerThread listener = null;

    private ServentVector connectedServents = new ServentVector();
    
    private Object lock = new Object();
    
    //--- Constructor(s) ---
    
    private Connections()
    {
	gnuPrefs.addPropertyChangeListener(this);
	listener = new ListenerThread(gnuPrefs.getLocalPort(), 
				      gnuPrefs.getLocalPort());
    }

    //--- Method(s) ---

    public static synchronized Connections getInstance()
    {
	if (instance == null) {
	    instance = new Connections();
	}

	return instance;
    }

    private void addAutoConnectServents()
    {
	addServent("connect1.gnutellanet.com:6346", true);
	addServent("connect2.gnutellanet.com:6346", true);
	addServent("connect3.gnutellanet.com:6346", true);
	addServent("connect4.gnutellanet.com:6346", true);
	addServent("gnutella.hostscache.com:6364", true);
	addServent("router.limewire.com:6346", true);
	addServent("qtellaconnect.ath.cx:6347", true);
	addServent("public.bearshare.net:6346", true);
    }

    public boolean addServent(String url, boolean connect) 
    {
        StringTokenizer s = new StringTokenizer(url, ":");

        if (s.countTokens() != 2)
            return false;

        String ip = s.nextToken();
	int port = 0;
	try {
	    port = Integer.parseInt(s.nextToken());
	} 
	catch (NumberFormatException e) {
	}

	if (ip.equals("") || port <= 0)
	    return false;
	
	Servent servent = new Servent(ip, port);
	addServent(servent);
	
	if (connect)
	    servent.start();

	return true;
    }

    public synchronized void addServent(Servent servent)
    {
	if (! super.contains(servent)) {
	    super.add(servent);
	    servent.addStatusChangeListener(this);
	    wakeup();
	}
    }

    public void addServent(String firstLine, Socket socket)
    {
	if (connectedCount < gnuPrefs.getMaxConnections()) {
	    Servent s = new Servent(firstLine, socket);
	    addServent(s);
	    doConnect(s);
	}
    }

    public EventVector getConnectedServents()
    {
	return connectedServents;
    }

    public synchronized void removeServent(int index)
    {
	if (gnuPrefs.getCleanDisconnected()) {
	    Servent s = (Servent) super.get(index);
	    s.removeStatusChangeListener(this);
	    super.remove(index);
	}
    }

    public synchronized void removeServent(Servent s)
    {
	if (gnuPrefs.getCleanDisconnected()) {
	    s.removeStatusChangeListener(this);
	    remove(s);
	}
    }

    public void setEnabled(boolean newValue)
    {
	enabled = newValue;
	
	if (enabled) {
	    if (runner != null) {
		wakeup();
	    }
	}
    }
    
    private void wakeup()
    {
	if (enabled) {
	    synchronized (lock) {
		lock.notify();
	    }
	}
    }

    public void init()
    {
	Debug.log("gnutella: Connections started");
	runner = new Thread(this);
	runner.start();
	(new Thread("ConnectionsInit")
	    {
		public void run() 
		{
		    addFromFile(gnuPrefs.getServentsFile());
		}
	    }
	 ).start();
    }
	
    public void run() 
    {
	int next = 0;
	
	while (!die) {
	    if (next > super.size())
		next = 0;

	    int start = next;
	    boolean looped = false;
	    while (super.size() > 0 && connectedCount <= maxConnect
		   && tryingCount <= maxTrying) {
		
		Debug.log("gnutella: connections next servent: " + next);
		Servent s = (Servent) super.get(next);
		
		if (s.getStatus() == Servent.STATUS_NOT_CONNECTED) {
		    doConnect(s);
		}
		
		next++;
		if (next >= super.size()) {
		    next = 0;
		    looped = true;
		}
		if (looped && next >= start)
		    break;
	    }
	    if (super.size() == 0) {
		Debug.log("gnutella: need to contact autoconnect hosts");
		//  addAutoConnectServents();
	    }
	    synchronized (lock) {
		try {
		    lock.wait();
		}
		catch (InterruptedException e) {
		}
	    }
	}
    }

    public void die()
    {
	die = true;
	synchronized (lock) {
	    lock.notify();
	}
	listener.die();
	listener = null;
	
	for (Iterator i = super.iterator(); i.hasNext();) {
	    doDisconnect((Servent) i.next());
	}

	instance = null;
    }

    public void addFromFile(String filename)
    {
	Servent s;
	ServentFile reader = new ServentFile(filename);
	if (reader.openReader()) {
	    while ((s = reader.readServent()) != null) {
		addServent(s);
	    }
	}	
	reader.close();
    }	

    public void saveToFile(String filename)
    {
	ServentFile writer = new ServentFile(filename);
	if (writer.openWriter()) {
	    /* save cache entries first */
	    int j = 0;
	    /* save connections vector */
	    for (Iterator i = iterator(); i.hasNext() && j < 100; j++) {
		writer.writeServent((Servent)i.next());
	    }
	}
	writer.close();
    }	
    
    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();
	
	if (p.equals("localPort")) {
	    
	    if (listener != null) {
		listener.die();
	    }
	    
	    listener = new ListenerThread(gnuPrefs.getLocalPort(),
					  gnuPrefs.getLocalPort());
		
	    if (listener.getPort() == 0) {
		//  setStatusText("Could not start listener (check local port)");
	    }
	}
	else if (p.equals("maxConnections")) {
	    maxConnect = gnuPrefs.getMaxConnections();
	    maxTrying = 2 * maxConnect;
	}
    }
    

    public void statusChange(StatusChangeEvent e)
    {
	switch (e.getNewStatus()) {
	case Servent.STATUS_DISCONNECTED:
	    if (e.getOldStatus() == Servent.STATUS_CONNECTED) {
		connectedCount--;
	    }
	    else if (e.getOldStatus() == Servent.STATUS_CONNECTING) {
		tryingCount--;
	    }
	    Servent servent = (Servent) e.getSource();
	    removeServent(servent);
	    connectedServents.remove(servent);
	    wakeup();
	    break;
	case Servent.STATUS_CONNECTED:
	    if (e.getOldStatus() == Servent.STATUS_CONNECTING) {
		connectedCount++;
		tryingCount--;
		servent = (Servent) e.getSource();
		connectedServents.add(servent);
	    }
	    break;
	case Servent.STATUS_NEW_STATS:
	    break;
	}
    }
    
    public synchronized void sendMessage(Message msg) 
    {
	for (Iterator i = connectedServents.iterator(); i.hasNext();) {
	    Servent con = (Servent) i.next();
	    con.send(msg, Servent.PRIORITY_NORMAL);
	}
    }
    public synchronized void broadcastMessage(Message msg, Servent source) 
    {
	for (Iterator i = connectedServents.iterator(); i.hasNext();) {
	    Servent servent = (Servent) i.next();
	    if (servent != source) {
		servent.send(msg, Servent.PRIORITY_BROADCAST);
	    }
	}
    }

    public void sendPong(PingMessage msg, Servent servent) 
    {
	if (msg.getTTL() + msg.getHops() > 2 
	    || getConnectedCount() >= gnuPrefs.getMaxConnections())
	    return;
	
	Repository repository = Repository.getInstance();
	
	PongMessage pongMsg = new PongMessage(msg.getMessageID(), 
					      (short) gnuPrefs.getLocalPort(),
  					      IPHelper.ipStringToBytearray(servent.getLocalAddress()),
					      repository.getNonNullSize(),
  					      200000);
	pongMsg.setTTL(msg.getHops());
  	servent.send(pongMsg, Servent.PRIORITY_BROADCAST);
    }

    public void searchRepository(QueryMessage msg, Servent servent)
    {
	File[] results = Repository.getInstance().search(msg.getSearchString());
	if (results != null) {
	    for (int i = 0; i < results.length; i++) {
		
	    }
	    
	}
    }
    
    /**
     * Caches incoming pong messages.
     */
    public synchronized void incomingPong(PongMessage msg) 
    {
	String ip = msg.getIP();
	
	if (IPHelper.isPrivateIP(ip))
	    return;

	//  Debug.log("Getting new Pong Host", Debug.C_NETWORK);

	if (size() < 100) {
	    addServent(new Servent(ip, msg.getPort()));
	}
    }

    public int getConnectedCount()
    {
	return connectedCount;
    }

    public void doConnect(Servent servent)
    {
	tryingCount++;
	servent.start();
    }

    public void doDisconnect(Servent servent)
    {
	servent.stop();
    }

    public ListenerThread getListener()
    {
	return listener;
    }
    
    public int getListenerPort()
    {
	if (listener != null)
	    return listener.getPort();
	return 0;
    }

    private class DisconnectTask extends TimerTask
    {
	public void run()
	{
	    synchronized (Connections.this) {
		for (Iterator i = iterator(); i.hasNext(); ) {
		    Servent s = (Servent) i.next();
		    if (s.getStatus() == Servent.STATUS_CONNECTING
			&& s.getUpTime() == 0
			&& s.getConnectTime() > DISCONNECT_INTERVAL) {
			Debug.log("DisconnectTask: disconnect");
			doDisconnect(s);
		    }
		}
	    }
	}
    }

    public static class ServentVector extends EventVector
    {
	
	public void add(Servent s) 
	{
	    super.add(s);
	}
	
	public void remove(Servent s)
	{
	    super.remove(s);
	}

    }
}
