/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin;

import xnap.cmdl.*;
import xnap.net.*;
import xnap.util.*;
import xnap.util.event.*;

import java.util.*;
import org.apache.log4j.Logger;

public class PluginManager
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected static Logger logger = Logger.getLogger(PluginManager.class);
    protected static PluginManager singleton = null;

    protected Console console = Console.getInstance();
    /**
     * Hashes class name to plugin.
     */
    protected Hashtable plugins = new Hashtable();
    protected Vector stateListeners = new Vector();
    protected HashSet packages = new HashSet();

    //--- Constructor(s) ---

    protected PluginManager()
    {
	installPlugins(JarClassLoader.getInstance().getClassNames(PluginInfo.TYPE_PLUGIN));

	Executer.addCommand(new InfoCmd());
	Executer.addCommand(new ListCmd());
	Executer.addCommand(new LoadCmd());
	Executer.addCommand(new UnloadCmd());
    }

    //--- Method(s) ---

    public static synchronized PluginManager getInstance() 
    {
	if (singleton == null) {
	    singleton = new PluginManager();
	}

	return singleton;
    }

    public synchronized void installPlugin(String className) 
    {
	try {
	    Class c = JarClassLoader.getInstance().loadClass(className);
	    if (isValidPlugin(c)) {
		// only install one plugin per package
		// this is needed for gui and cmdl plugins
		String p = c.getPackage().getName();
		if (!packages.contains(p)) {
		    logger.info("installing plugin: " + className);
		    IPlugin plugin = (IPlugin)c.newInstance();
		    packages.add(p);
		    plugins.put(className, plugin);
		}
	    }
	}
	catch (ClassNotFoundException e) {
	    logger.error("plugin not found: " + className);
	}
	catch (Throwable e) {
	    logger.error("could not load: " + className, e);
	}
    }

    public boolean isValidPlugin(Class c)
    {
	Class[] interfaces = c.getInterfaces();
	for (int i = 0; i < interfaces.length; i++) {
	    if (interfaces[i] == INetworkPlugin.class) {
		return true;
	    }
	}
	
	return false;
    }

    public synchronized void installPlugins(String[] plugins)
    {
	for (int i = 0; i < plugins.length; i++) {
	    try {
		installPlugin(plugins[i]);
	    }
	    catch (Exception e) {
		logger.warn("Counld not install plugin", e);
	    }
	}
    }

    public IPlugin getPluginByName(String name) 
    {
	for (Enumeration enum = plugins.elements(); enum.hasMoreElements();) {
	    IPlugin plugin = (IPlugin)enum.nextElement();
	    if (plugin.getName().equalsIgnoreCase(name)) {
		return plugin;
	    }
	}

	return null;
    }

    public IPlugin getPlugin(String className) 
    {
	return (IPlugin)plugins.get(className);
    }

    public synchronized IPlugin[] getPlugins() 
    {
	IPlugin[] ret = new IPlugin[plugins.size()];

	int i = 0;
	for (Enumeration enum = plugins.elements(); enum.hasMoreElements();) {
	    ret[i] = (IPlugin)enum.nextElement();
	    i++;
	}

	return ret;
    }

    public void enablePlugins(String enabledPlugins) 
    {
	StringTokenizer t = new StringTokenizer(enabledPlugins, ";");

	while (t.hasMoreTokens()) {
	    String name = t.nextToken();
	    logger.info("plugin manager: loading " + name);

	    if (plugins.containsKey(name)) {
		setEnabled(name, true);
	    }
	}
    }
    
    /**
     * Disables all plugins.
     *
     * @return semi-colon separated string of previously enabled plugins
     */
    public String disablePlugins() 
    {
	String enabledPlugins = "";
	for (Enumeration enum = plugins.elements(); enum.hasMoreElements();) {
	    IPlugin plugin = (IPlugin)enum.nextElement();
	    if (plugin.isEnabled()) {
		enabledPlugins += plugin.getClass().getName() + ";";
		setEnabled(plugin, false);
	    }
	}

	return enabledPlugins;
    }

    public void addStateListener(StateListener l) 
    {
        stateListeners.add(l);
    }

    public void removeStateListener(StateListener l) 
    {
        stateListeners.remove(l);
    }
        
    protected void fireStateChanged(IPlugin source, boolean newState)
    {
        Object[] listeners = stateListeners.toArray();

	if (listeners != null) {
	    StateEvent event;
	    if (newState)
		event = new StateEvent(source, StateEvent.OBJECT_ENABLED);
	    else
		event = new StateEvent(source, StateEvent.OBJECT_DISABLED);

	    for (int i = listeners.length - 1; i >= 0; i--) {
		if (newState)
		    ((StateListener)listeners[i]).stateEnabled(event);
		else
		    ((StateListener)listeners[i]).stateDisabled(event);
	    }
	}
    }   

    public synchronized void setEnabled(String className, boolean newValue) 
    {
	IPlugin plugin = (IPlugin)plugins.get(className);
	setEnabled(plugin, newValue);
    }
	
    public synchronized void setEnabled(IPlugin plugin, boolean newValue) 
    { 
	if (newValue) {
	    if (!plugin.isEnabled()) {
		try {
		    plugin.start();
		    fireStateChanged(plugin, true);
		} 
		catch (Throwable e) {
		    logger.error("could not enable plugin", e);
		}
	    }
	}
	else {
	    if (plugin.isEnabled()) {
		fireStateChanged(plugin, false);
		plugin.stop();
	    }
	}
    }

    public synchronized ISearch[] search(SearchFilter f, int priority)
    {
	LinkedList searches = new LinkedList();

	for (Enumeration enum = plugins.elements(); enum.hasMoreElements();) {
	    IPlugin p = (IPlugin)enum.nextElement();
	    if (p instanceof INetworkPlugin && p.isEnabled()) {
		ISearch[] array = ((INetworkPlugin)p).search(f, priority);
		if (array != null) {
		    for (int i = 0; i < array.length; i++)
			searches.add(array[i]);
		}
	    }
	}	

	AbstractSearch[] array = new AbstractSearch[searches.size()];
	System.arraycopy(searches.toArray(), 0, array, 0, array.length);

	return array;
    }

    protected class InfoCmd extends AbstractCommand
    {
	public InfoCmd()
	{
	    putValue(CMD, new String[] {"plugininfo"});
	    putValue(PARAMETER, "[plugin]");
	    putValue(SHORT_HELP, "Show info about plugin.");
	}
	
	public boolean execute(String[] argv)
	{
	    if (argv.length != 2) {
		return false;
	    }

  	    IPlugin plugin = getPluginByName(argv[1]);
  	    if (plugin == null) {
  		console.println("Could not find plugin " + argv[1]);
	    }
  	    else {
		console.println(plugin.getDescription());
	    }

	    return true;
	}
    }

    protected class ListCmd extends AbstractCommand
    {
	public ListCmd()
	{
	    putValue(CMD, new String[] {"listplugins", "lp"});
	    putValue(SHORT_HELP, "Show installed plugins.");
	}
	
	public boolean execute(String[] args)
	{
  	    IPlugin[] plugins = getPlugins();

  	    if (plugins.length == 0) {
  		console.println("No plugins installed.");
	    }
  	    else {
		String[] table = new String[plugins.length];
  		for (int i = 0; i < plugins.length; i++) {
		    StringBuffer sb = new StringBuffer();
		    sb.append(plugins[i].getName());
		    sb.append("|");
		    sb.append((plugins[i].isEnabled()) ? "enabled" 
			      : " disabled");
		    sb.append("|");
		    sb.append(plugins[i].getClass().getName());
		    table[i] = sb.toString();
		}
		int L = Formatter.LEFT;
		int[] cols = new int[] { L, L, L };
		console.println(Formatter.formatTable(table, cols));
  	    }

	    return true;
	}
    }
    
    protected class LoadCmd extends AbstractCommand
    {
	public LoadCmd()
	{
	    putValue(CMD, new String[] {"load"});
	    putValue(PARAMETER, "[plugin]");
	    putValue(SHORT_HELP, "Load plugin.");
	}
	
	public boolean execute(String[] argv)
	{
	    if (argv.length != 2) {
		return false;
	    }

  	    IPlugin plugin = getPluginByName(argv[1]);
  	    if (plugin == null) {
  		console.println("Could not find plugin " + argv[1]);
	    }
  	    else if (plugin.isEnabled()) {
  		console.println("Plugin already loaded");
	    }
  	    else {
  		setEnabled(plugin, true);
	    }

	    return true;
	}
    }

    protected class UnloadCmd extends AbstractCommand
    {
	public UnloadCmd()
	{
	    putValue(CMD, new String[] {"unload"});
	    putValue(PARAMETER, "[plugin]");
	    putValue(SHORT_HELP, "Unload plugin.");
	}
	
	public boolean execute(String[] argv)
	{
	    if (argv.length != 2) {
		return false;
	    }

  	    IPlugin plugin = getPluginByName(argv[1]);
  	    if (plugin == null) {
  		console.println("Could not find plugin " + argv[1]);
	    }
  	    else if (!plugin.isEnabled()) {
  		console.println("Plugin not loaded");
	    }
	    else {
  		setEnabled(plugin, false);
	    }

	    return true;
  	}
    }

}
