/*
 *  Java Napster version x.yz (for current version number as well as for
 *  additional information see version.txt)
 *
 *  Previous versions of this program were written by Florian Student
 *  and Michael Ransburg available at www.weblicity.de/jnapster and
 *  http://www.tux.org/~daneel/content/projects/10.shtml respectively.
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.io;

import xnap.*;
import xnap.net.*;
import xnap.util.*;

import java.beans.*;
import java.io.*;
import java.util.*;

public class ResumeRepository extends AbstractRepository
    implements PropertyChangeListener {

    //--- Constant(s) ---

    /**
     * Don't ever change this or we will loose the resume repository
     * without notice. Use allowWrite instead.
     */
    public static int VERSION = 1;
    
    //--- Data Field(s) ---

    private static ResumeRepository singleton = new ResumeRepository();
    private static DownloadQueue dlQueue = DownloadQueue.getInstance();

    private Preferences prefs = Preferences.getInstance();
    private boolean firstTime = true;

    //--- Constructor(s) ---
    
    private ResumeRepository()
    {
	super(FileHelper.getHomeDir() + "resumes", VERSION, true);

	readAndUpdateLater();
    }

    //--- Method(s) ---

    public static ResumeRepository getInstance()
    {
	return singleton;
    }
    
    public void add(ResumeFile3 f)
    {
	super.add(f);
	updateLater();
    }

    public String getDirs()
    {
	return prefs.getIncompleteDir();
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	String p = e.getPropertyName();

	if (e.getPropertyName().equals("incompleteDir")) {
	    updateLater();
	}
    }

    /**
     * <code>allowWrite</code> will be set to false, if the resume repository
     * could not be read. Ask user what to do.
     */
    protected void postRead(boolean success, String response) 
    {
	if (!allowWrite) {
	    // blocks until users confirms write or does not
	    allowWrite = Notifier.getInstance().canPurgeResumeRepository();
	}
    }

    /**
     * Adds resumes to download queue
     */
    protected void postUpdate() 
    {
	if (firstTime) {
	    for (ListIterator li = super.listIterator(); li.hasNext();) {
		File resume = (File)li.next();

		if (resume != null) {
		    ResumeFile3 file = convert(resume);

		    // update repository
		    li.set(file);

		    if (file == null) {
			continue;
		    }

		    shorten(file);

		    IDownloadContainer d = file.createDownload();
		    dlQueue.add(d, false);
		}
	    }

	    firstTime = false;
	}
    }

    /**
     * Converts different resume file versions for downward compatibility.
     */
    protected ResumeFile3 convert(File file)
    {
	if (file instanceof ResumeFile3) {
	    return (ResumeFile3)file;
	}
	else if (file instanceof ResumeFile2) {
	    // ResumeFile2 inherits ResumeFile so it does not need
	    // special treatment but we need to mention it here for
	    // jikes to generate a dependency so that it is not dropped
	    // from the distribution
	    ResumeFile2 f = (ResumeFile2)file;
	    return new ResumeFile3(f, f.getFinalSize(), convertFilter(f));
	}
	else if (file instanceof ResumeFile) {
	    ResumeFile f = (ResumeFile)file;
	    return new ResumeFile3(f, f.getFinalSize(), convertFilter(f));
	}

	return null;
    }

    public SearchFilterData convertFilter(ResumeFile file)
    {
	return SearchFilterHelper.convert(file.getFilterData());
    }

    public void shorten(ResumeFile3 file)
    {
	logger.debug("shorten " + file.getName() + ", " + file.length());
		    
	// Shorten files which are too big due to XNap's 
	// stupid (and fixed) upload and download bugs.
	FileHelper.shorten(file, (int)Math.max(file.length() 
					       - file.getFinalSize(), 0));
	
	logger.debug("add " + file.getName() + ", " + file.length());
    }

}

