/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.gui.menu.TableColumnsMenu;
import xnap.util.Debug;
import xnap.util.Preferences;

import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;

/**
 * TableHeaderListener class handles the mouse events invoked by clicking
 * on the header portion of the JTable.
 */
public class TableHeaderListener extends MouseAdapter 
    implements MouseMotionListener {

    // --- Data Field(s) ---

    private JTable jta;
    private JTableHeader header;
    private JPopupMenu jpm;
    private SortButtonRenderer renderer;
    private AbstractSortableTableModel stm;
    private boolean sort = false;

    // --- Constructor(s) ---

    public TableHeaderListener(JTable jta, JPopupMenu jpm)
    {
	this.jta = jta;
	this.header = jta.getTableHeader();
	this.stm = (AbstractSortableTableModel)jta.getModel();
	this.jpm = jpm;

	renderer = new SortButtonRenderer(header.getDefaultRenderer());
	header.setDefaultRenderer(renderer);

	header.setReorderingAllowed(true);
	header.setResizingAllowed(true);

	if (stm.getLastSortedColumn() != -1) {
	    int col = jta.convertColumnIndexToView(stm.getLastSortedColumn()); 
	    renderer.setSortedColumn(col, stm.isSortedAscending());
	}	    
    }

    // --- Method(s) ---

    public static TableHeaderListener install(JTable jta, JPopupMenu jpm)
    {
        JTableHeader jth = jta.getTableHeader();
	TableHeaderListener l = new TableHeaderListener(jta, jpm);
        jth.addMouseListener(l);
        jth.addMouseMotionListener(l);

	return l;
    }

    /**
     * Installs listener.
     */
    public static TableHeaderListener install(JTable jta)
    {
	return install(jta, null);
    }

    public static boolean isWindowsPopupTrigger(MouseEvent e) 
    {
	return (Preferences.isWindowsOS
		&& ((e.getModifiers() & MouseEvent.BUTTON3_MASK) != 0));
    }

    public void mousePressed(MouseEvent e)
    {
	if (e.isPopupTrigger()) {
	    showPopupMenu(e);
	}
	else {
	    int col = header.columnAtPoint(e.getPoint());

	    renderer.selectColumn(col);
	    header.repaint();
	    
	    sort = true;
	}

	setAutoResizeMode(e);
    }

    public void mouseReleased(MouseEvent e) 
    {
	if (e.isPopupTrigger()) {
	    showPopupMenu(e);
	}
	else if (sort) {
	    int col = header.columnAtPoint(e.getPoint());
	    if (col != -1) {
		if (jta.isEditing()) {
		    jta.getCellEditor().stopCellEditing();
		}
		int sortCol = jta.convertColumnIndexToModel(col);
		sortByColumn(col, sortCol);
		sort = false;
	    }
	    renderer.selectColumn(-1);
	    header.repaint();

	    // save sorted column
	    if (jta.getModel() instanceof AbstractDynamicTableModel) {
		((AbstractDynamicTableModel)jta.getModel()).saveSortedColumn();
	    }
	}
	else {
	    // mouse was dragged (or menu shown)
	    if (jta.getModel() instanceof AbstractDynamicTableModel) {
		((AbstractDynamicTableModel)jta.getModel()).saveColumnWidths();
	    }
	}
    }

    public void mouseDragged(MouseEvent e) 
    {
	if (sort) {
	    // abort sorting
	    sort = false;
	    
	    renderer.selectColumn(-1);
	    header.repaint();
	}

	setAutoResizeMode(e);
    }

    public void mouseMoved(MouseEvent e) 
    {
    }

    private void setAutoResizeMode(MouseEvent e)
    {
	if ((e.getModifiers() & InputEvent.SHIFT_MASK) != 0) {
	    jta.setAutoResizeMode(JTable.AUTO_RESIZE_NEXT_COLUMN);
	}
	else if ((e.getModifiers() & InputEvent.CTRL_MASK) != 0) {
	    jta.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
	}
	else {
	    jta.setAutoResizeMode(JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	}
    }

    public void showPopupMenu(MouseEvent e)
    {
	if (jpm != null) {
	    header.setDraggedColumn(null);
	    header.setResizingColumn(null);
	    jpm.show(e.getComponent(), e.getX(), e.getY());
	}
    }

    public void sortByColumn(int col, int sortCol)
    {
        boolean ascending = stm.sortByColumn(sortCol);
	renderer.setSortedColumn(col, ascending);
    }

}
