/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.gui.table;

import xnap.*;
import xnap.io.*;
import xnap.net.*;
import xnap.util.*;

import org.apache.log4j.Logger;

import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.table.*;

public class LibraryTableModel extends AbstractDynamicTableModel {

    //--- Constant(s) ---

    public static final int NAME = 0;
    public static final int FULLNAME = 1;
    public static final int PATH = 2;
    public static final int TYPE = 3;
    public static final int SIZE = 4;
    public static final int MODIFIED = 5;
    public static final int INFO = 6;

    //--- Data field(s) ---

    protected Column columns[] = new Column[] {
	new Column(XNap.tr("Name"), String.class,
		   new StringCellRenderer()),
	new Column(XNap.tr("Fullname"), String.class, 
		   new StringCellRenderer()),
	new Column(XNap.tr("Path"), String.class, 
		   new StringCellRenderer()),
	new Column(XNap.tr("Type"), String.class),
	new Column(XNap.tr("Size"), Long.class, 
		   new FilesizeCellRenderer()),
	new Column(XNap.tr("Modified"), Long.class, 
		   new TimeCellRenderer()),
	new Column(XNap.tr("Info"), String.class, 
		   new StringCellRenderer()), 

    };

    private static Logger logger = Logger.getLogger(LibraryTableModel.class);

    private ArrayList data = new ArrayList();
    private ParseThread pt;
    private long totalSize;
    private int editableColumn = -1;
    private int editableRow = -1;
    private File currentDir;

    //--- Constructor(s) ---

    public LibraryTableModel() 
    {
	super("library", Preferences.getInstance());

	setColumns(columns);
	DefaultCellEditor dce = new DefaultCellEditor(new JTextField());
	dce.setClickCountToStart(Integer.MAX_VALUE);
	columns[NAME].setCellEditor(dce);
    }

    //--- Method(s) ---

    public void add(File f) 
    {
	if (f.isDirectory()) {
	    add(f.listFiles());
	    currentDir = f;
	}
	else {
	    addFile(f);
	}
    }

    public void add(File[] list) 
    {
	int firstRow = data.size();

	for (int i = 0; i < list.length; i++) {
	    if (list[i].isFile()) {
		addFile(list[i]);
	    }
	}

	fireTableRowsInserted(firstRow, data.size() - 1);
	    
	if (pt != null) {
	    pt.die = true;
	}

	pt = new ParseThread();
	pt.start();
    }
    
    public void addFile(File file)
    {
	if (!(file instanceof RepositoryFile)) {
	    file = FileHandler.handle(file, false);
	}
	data.add(file);
	totalSize += file.length();
    }

    public boolean delete(File f)
    {
	int i = data.indexOf(f);
	if (i != -1) {
	    if (f.delete()) {
		remove(i);
		return true;
	    }
	}
	return false;
    }
    
    public void clear() 
    {
	if (pt != null) {
	    pt.die = true;
	}

        int k = data.size() - 1;
	data.clear();
        fireTableRowsDeleted(0, k);

	totalSize = 0;
    }

    public RepositoryFile get(int i)
    {
	return (RepositoryFile)data.get(mapToIndex(i));
    }

    public int getRowCount() 
    {
        return data.size();
    }

    public Object get(int i, int j) 
    {
        if (i >= data.size()) 
	    return null;

        RepositoryFile f = (RepositoryFile)data.get(i);

        if (f == null) 
	    return null;

        switch (j) {
	case NAME:
	    return f.getName();
	case FULLNAME:
	    return f.getPath();
	case PATH:
	    return f.getParent();
	case SIZE:
	    return new Long(f.length());
	case TYPE:
	    return FileHelper.extension(f.getName());
	case MODIFIED:
	    return new Long(f.lastModified());
	case INFO:
	    return f.getInfo();
	default:
	    return null;
        }
    }

    public String getTableName()
    {
	return XNap.tr("Library Table");
    }

    public long getTotalSize()
    {
	return totalSize;
    }

    public boolean isCellEditable(int row, int column)
    {
	return (row == editableRow && column == editableColumn);
    }

    public void remove(int i) 
    {
        data.remove(i);
	fireTableRowsDeleted(i, i);
    }

    public void setCellEditable(int row, int column)
    {
	editableRow = row;
	editableColumn = column;
    }

    public void setValueAt(Object obj, int row, int column)
    {
	if (!(obj instanceof String))
	    return;

	String newName = (String)obj;

	RepositoryFile f = get(row);
	if (newName.length() > 0 && f.canWrite()) {
	    try {
		f = FileHandler.handle(FileHelper.moveUnique(f, f.getParent(),
							     newName), false);
		data.set(mapToIndex(row), f);
		fireTableRowsUpdated(row, row);
	    }
	    catch (IOException ie) {
		logger.debug("Could not rename file", ie);
	    }
	}
    }

    protected class ParseThread extends Thread 
    {
	public boolean die;

	public ParseThread()
	{

	}

	public void run()
	{
	    die = false;
	    parseWholeTable();
	}

	private void parseWholeTable()
	{
	    for (int i = 0; i < data.size() && !die; i++) {
		RepositoryFile f = (RepositoryFile)data.get(i);
		f.parse();
		if (!die) {
		    LibraryTableModel.this.fireTableRowsUpdated(i, i);
		}
	    }
	}
    }
}

