/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.XNap;
import xnap.gui.util.SwingSynchronizedCache;
import xnap.net.IUser;
import xnap.net.event.StatusChangeEvent;
import xnap.net.event.StatusChangeListener;
import xnap.user.UserManager;
import xnap.util.EventVector;
import xnap.util.event.ListEvent;
import xnap.util.event.ListListener;

import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import org.apache.log4j.Logger;

public class HotlistTableModel extends UserTableModel
    implements ListListener, StatusChangeListener {

    //--- Constant(s) ---

    //--- Data field(s) ---

    private static Logger logger = Logger.getLogger(HotlistTableModel.class);
    
    private UserFilter filter = null;

    //--- Constructor(s) ---

    public HotlistTableModel() 
    {
	super("hotlist");

	SwingSynchronizedCache cache = new SwingSynchronizedCache(this);
	UserManager.getInstance().addListListener(cache);
    }

    //--- Method(s) ---

    public void elementAdded(ListEvent e) 
    {
        IUser u = (IUser)e.getElement();
	u.addStatusChangeListener(this);
	addFiltered(u);
    }

    public void elementRemoved(ListEvent e) 
    {
        IUser u = (IUser)e.getElement();
	u.removeStatusChangeListener(this);
	remove(u);
    }

    public void addFiltered(IUser u) 
    {
	if (filter == null || filter.matches(u)) {
	    add(u);
	}
    }

    public String getTableName()
    {
	return XNap.tr("Hotlist Table");
    }

    public void setFilter(UserFilter filter)
    {
	this.filter = filter;
	reload();
    }

    public void statusChange(StatusChangeEvent e)
    {
	SwingUtilities.invokeLater
	    (new StatusChangeEventHandler((IUser)e.getSource()));
    }

    public void reload()
    {
	rows.clear();
	Object[] users = UserManager.getInstance().toArray();
	for (int i = 0; i < users.length; i++) {
	    IUser u = (IUser)users[i];
	    if (filter == null || filter.matches(u)) {
		rows.add(u);
	    }
	}
	fireTableDataChanged();
    }

    //--- Inner Class(es) ---

    private class StatusChangeEventHandler implements Runnable
    {
	IUser u;

	public StatusChangeEventHandler(IUser u)
	{
	    this.u = u;
	}

	public void run()
	{
	    logger.debug("status change: " + u + " (" + u.getClass() + ")");
	    if (filter == null || filter.matches(u)) {
		int i = rows.indexOf(u);
		if (i != -1) {
		    fireTableRowsUpdated(i, i);
		}
		else {
		    add(u);
		}
	    }
	    else {
		remove(u);
	    }
	}
    }

    public interface UserFilter
    {
	boolean matches(IUser u);
    }

    public static class AllFilter implements UserFilter {

	public boolean matches(IUser u)
	{
	    return true;
	} 

	public String toString()
	{
	    return XNap.tr("All");
	}

    }

    public static class CategoryFilter implements UserFilter {

	String category;

	public CategoryFilter(String category)
	{
	    this.category = category;
	}

	public boolean matches(IUser u)
	{
	    return !u.isTemporary() && (u.getCategory().equals(category));
	} 

	public String toString()
	{
	    return category;
	}

    }

//     public static class FriendFilter implements UserFilter {

// 	public boolean matches(IUser u)
// 	{
// 	    return !u.isTemporary() && (u.getMaxUploads() > 0);
// 	} 

// 	public String toString()
// 	{
// 	    return XNap.tr("Friends");
// 	}

//     }

//     public static class HotlistFilter implements UserFilter {

// 	public boolean matches(IUser u)
// 	{
// 	    return !u.isTemporary();
// 	} 

// 	public String toString()
// 	{
// 	    return XNap.tr("Hotlist");
// 	}
//     }

    public static class OtherFilter implements UserFilter {

	UserFilter[] filters;

	public OtherFilter(UserFilter[] filters)
	{
	    this.filters = filters;
	}

	public boolean matches(IUser u)
	{
	    if (u.isTemporary()) {
		return false;
	    }

	    boolean accept = false;
	    for (int i = 0; i < filters.length && !accept; i++) {
		accept |= filters[i].matches(u);
	    }

	    return !accept;
	} 

	public String toString()
	{
	    return XNap.tr("Other");
	}

    }

}
