/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.table;

import xnap.gui.menu.TableColumnsMenu;
import xnap.util.TablePreferencesSupport;

import java.beans.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;


/**
 * Provides a sortable table model with dynamic column support.
 */
public abstract class AbstractDynamicTableModel 
    extends AbstractSortableTableModel implements PropertyChangeListener
{

    //--- Data field(s) ---

    protected Column[] columns = new Column[0];
    protected JTable jta = null;
    protected String table;
    protected TablePreferencesSupport tps;

    //--- Constructor(s) ---

    public AbstractDynamicTableModel(String table, TablePreferencesSupport tps)
    {
	this.table = table;
	this.tps = tps;

	if (tps != null) {
	    setMaintainSortOrder(tps.getTableMaintainSortOrder(table));
	    tps.addTableListener(table, this);
	}
    }

    public AbstractDynamicTableModel()
    {
    }

    //--- Method(s) ---

    /**
     * Creates a table from this model. 
     */
    public JTable createJTable()
    {
	if (jta == null) {
	    DefaultTableColumnModel dtcm = new DefaultTableColumnModel();
	    setModel(dtcm);
	    if (tps != null) {
		dtcm.addColumnModelListener(new ColumnListener());
		jta = new JTable(this, dtcm);
		setVisible(tps.getTableColumnsArray(table), jta);
		setMaintainSortOrder(tps.getTableMaintainSortOrder(table));
		TableColumnsMenu tcm
		    = new TableColumnsMenu(table, getColumnNames(), tps);
		TableHeaderListener.install(jta, tcm.getPopupMenu());
	    }
	    else {
		jta = new JTable(this);
		TableHeaderListener.install(jta);
	    }
	}
	return jta;
    }

    public JMenu createJMenu()
    {
	return new TableColumnsMenu(this);
    }

    public Class getColumnClass(int i) 
    {
        return columns[i].getDataType();
    }

    public int getColumnCount() 
    {
        return columns.length;
    }

    public String getColumnName(int i) 
    {
        return columns[i].getName();
    }

    public String[] getColumnNames()
    {
	String[] names = new String[columns.length];
	for (int i = 0; i < names.length; i++) {
	    names[i] = columns[i].getName();
	}
	return names;
    }

    public Column[] getColumns()
    {
	return columns;
    }

    public TablePreferencesSupport getPreferencesSupport()
    {
	return tps;
    }

    public String getTableIdentifier()
    {
	return table;
    }

    public abstract String getTableName();

    public void setColumns(Column[] columns)
    {
	this.columns = columns;

	if (tps != null) {
	    // restore table widths
	    int[] widths = tps.getTableColumnWidths(table);
	    for (int i = 0; i < widths.length && i < columns.length; i++) {
		columns[i].setPreferredWidth(widths[i]);
		columns[i].setWidth(widths[i]);
	    }
	    
	    // restore sort order
	    int c = tps.getTableSortedColumn(table);
	    if (c != 0) {
		setSortedAscending(c > 0);
		sortByColumn(Math.abs(c) - 1, false);
	    }
	}
    }

    public void setModel(TableColumnModel tcm)
    {
	for (int i = 0; i < columns.length; i++) {
	    columns[i].setModel(tcm);
	    columns[i].setModelIndex(i);
	}
    }

    public void setVisible(int[] visibleCols, JTable jta)
    {
	boolean[] newStatus = new boolean[columns.length];
	Arrays.fill(newStatus, false);

	for (int i = 0; i < visibleCols.length; i++) {
	    int c = visibleCols[i];
	    if (c >= 0 && c < columns.length) {
		newStatus[c] = true;
	    }
	}
	
	boolean visible = false;
	for (int i = 0; i < columns.length; i++) {
	    visible |= newStatus[i];
	    columns[i].setVisible(newStatus[i]);
	}

	if (!visible && columns.length > 0) {
	    // make sure at least one column is visible
	    columns[0].setVisible(true);
	}

//  	if (jta != null) {
//  	    jta.sizeColumnsToFit(index);
//  	}
    }

    public void setVisible(int[] visibleColumns)
    {
	setVisible(visibleColumns, null);
    }

    public void propertyChange(PropertyChangeEvent e)
    {
	if (tps != null) {
	    setVisible(tps.getTableColumnsArray(table), jta);
	    setMaintainSortOrder(tps.getTableMaintainSortOrder(table));
	}
    }

    public void saveColumnWidths()
    {
	if (tps != null) {
	    int[] widths = new int[columns.length];
	    for (int i = 0; i < widths.length; i++) {
		widths[i] = columns[i].getWidth();
	    }
	    tps.setTableColumnWidths(table, widths);
	}
    }

    public void saveSortedColumn()
    {
	if (tps != null) {
	    int col = getLastSortedColumn();
	    if (col == -1) {
		col = 0;
	    }
	    else {
		col += 1;
		col = (isSortedAscending()) ? +col : -col;
	    }
	    tps.setTableSortedColumn(table, col);
	}
    }

    protected class ColumnListener implements TableColumnModelListener {

	public void columnAdded(TableColumnModelEvent e) 
	{
	}

	public void columnMarginChanged(ChangeEvent e)
	{
	}

	public void columnMoved(TableColumnModelEvent e)
	{
	    if (tps != null) {
		TableColumnModel tcm = (TableColumnModel)e.getSource();
		int[] cols = new int[tcm.getColumnCount()];
		for (int i = 0; i < cols.length; i++) {
		    cols[i] = tcm.getColumn(i).getModelIndex();
		}
		tps.setTableColumns(table, cols);
	    }
	}

	public void columnRemoved(TableColumnModelEvent e)
	{
	}

	public void columnSelectionChanged(ListSelectionEvent e) 
	{
	}
    }
}
