/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.XNap;
import xnap.cmdl.Executer;
import xnap.gui.AbstractPanel;
import xnap.gui.XNapFrame;
import xnap.gui.event.PopupListener;
import xnap.gui.event.UserSupport;
import xnap.gui.menu.UserMenu;
import xnap.gui.table.HotlistTableModel;
import xnap.gui.table.LinkSpeedCellRenderer;
import xnap.gui.table.TableHeaderListener;
import xnap.gui.table.TimeCellRenderer;
import xnap.net.IUser;
import xnap.user.UserManager;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class HotlistPanel extends AbstractPanel 
    implements ActionListener, ListSelectionListener, UserSupport {

    // --- Data Field(s) ---

    private JTable jtaHotlist;
    private HotlistTableModel htm;
    private JSplitPane jsp;
    private JComboBox jcbFilter;

    private JScrollPane jspRight;
    private JPanel jpUserEditor;
    private UserEditorPanel uep;

    private RemoveAction acRemove = new RemoveAction();

    //--- Constructor(s) ---
	
    public HotlistPanel()
    {
	initialize();
    }

    // --- Method(s) ---

    private void initialize()
    {
	JPanel jpHotlist = new JPanel(new BorderLayout());

	// hotlist popup
	UserMenu jm = new UserMenu(this, true);
	jm.add(new JMenuItem(acRemove));
	jm.addSeparator();

	// hotlist table
	htm = new HotlistTableModel();
	jtaHotlist = htm.createJTable();
	jtaHotlist.addMouseListener(new PopupListener(jm));
	jtaHotlist.getSelectionModel().addListSelectionListener(this);
	jtaHotlist.setShowGrid(false);
	jpHotlist.add(new JScrollPane(jtaHotlist), BorderLayout.CENTER);

	// filter combo box
	JPanel jpFilter = new JPanel();
	jpFilter.setLayout(new BoxLayout(jpFilter, BoxLayout.X_AXIS));
	jpFilter.setBorder(new EmptyBorder(5, 5, 5, 5));
	jpHotlist.add(jpFilter, BorderLayout.SOUTH);

	jpFilter.add(new JLabel(XNap.tr("Show")));
	jpFilter.add(Box.createHorizontalStrut(5));

	Object[] filters = getFilters();
    	jcbFilter = new JComboBox(filters);
	jcbFilter.addActionListener(this);
    	jpFilter.add(jcbFilter);

	jspRight = new JScrollPane();

	// user info	
	uep = new UserEditorPanel();
	jpUserEditor = new JPanel(new GridBagLayout());
	GridBagHelper.add(jpUserEditor, uep);
	GridBagHelper.add(jpUserEditor, new JButton(new ApplyAction()), 
			  GridBagHelper.COMPONENT_INSETS, false, 
			  GridBagConstraints.NORTHEAST);
	GridBagHelper.addVerticalSpacer(jpUserEditor);
	
	// split pane
	jsp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
	jsp.add(jpHotlist, JSplitPane.LEFT);
	jsp.add(jspRight, JSplitPane.RIGHT);
	jsp.setDividerLocation(prefs.getHotlistDividerLocation());
	jsp.setResizeWeight(0);
	jsp.setOneTouchExpandable(true);

	setLayout(new BorderLayout());
	add(jsp, BorderLayout.CENTER);

	htm.setFilter
	    ((HotlistTableModel.UserFilter)jcbFilter.getSelectedItem());
    }

    public void savePrefs()
    {	
	prefs.setHotlistDividerLocation(jsp.getDividerLocation());
    }
    
    /**
     * Called when a filter is selected.
     */
    public void actionPerformed(ActionEvent event) 
    {
	htm.setFilter
	    ((HotlistTableModel.UserFilter)jcbFilter.getSelectedItem());
    }

    public AbstractAction[] getActions()
    {
	return new AbstractAction[] {
	    acRemove
	};
    }

    public Object[] getFilters()
    {
	HotlistTableModel.UserFilter a 
	    = new HotlistTableModel.CategoryFilter(XNap.tr("Hotlist"));
	HotlistTableModel.UserFilter b
	    = new HotlistTableModel.CategoryFilter(XNap.tr("Banned"));
	HotlistTableModel.UserFilter[] all
	    = { a, b };

	return new Object[] {
	    a,
	    b,
	    new HotlistTableModel.OtherFilter(all),
	    new HotlistTableModel.AllFilter(), 
	};
    }

    public IUser[] getUsers()
    {
	int[] rows = jtaHotlist.getSelectedRows();
	IUser[] users = new IUser[rows.length];
	for (int i = 0; i < rows.length; i++) {
	    users[i] = htm.get(rows[i]);
	}
	return users;
    }

    public void reload()
    {
	jspRight.setViewportView(null);
	htm.reload();	
    }

    public void valueChanged(ListSelectionEvent e)
    {
	if (jspRight.getViewport().getView() 
	    == jpUserEditor && uep.hasChanged()) {
	    Object[] message = new Object[] {
		XNap.tr("The settings have been changed."),
		XNap.tr("Do you want to save your changes?"), 
	    };

	    int response = JOptionPane.showConfirmDialog
		(this, message, XNap.tr("Edit User"), 
		 JOptionPane.YES_NO_OPTION);

	    if (response == JOptionPane.YES_OPTION) {
		uep.apply();
	    }
	    else if (response == JOptionPane.CANCEL_OPTION) {
		// this does not work correctly
		return;
	    }
	}

	int row = jtaHotlist.getSelectedRow();
	if (row == -1 || jtaHotlist.getSelectedRowCount() != 1) {
	    jspRight.setViewportView(null);
	}
	else {
	    uep.setUser(htm.get(row));
	    jspRight.setViewportView(jpUserEditor);
	}
    }
	
    //--- Inner Class(es) ---

    /**
     * 
     */
    private class RemoveAction extends AbstractAction {

        public RemoveAction() 
	{
            putValue(Action.NAME, XNap.tr("Remove"));
            putValue(Action.SHORT_DESCRIPTION, 
		     XNap.tr("Remove user(s) from hotlist"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("eraser.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('R'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    IUser[] users = getUsers();
	    for (int i = 0; i < users.length; i++) {
		UserManager.reset(users[i]);
	    }
	    reload();
        }

    }

    /**
     * Updates user table.
     */
    private class UpdateAction extends AbstractAction {

        public UpdateAction() 
	{
            putValue(Action.NAME, XNap.tr("Update"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("updatedescr"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("reload.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('U'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    reload();
        }

    }

    /**
     * Handles the apply button.
     */
    private class ApplyAction extends AbstractAction {

        public ApplyAction() 
	{
	    putValue(Action.NAME, XNap.tr("Apply"));
	    putValue(Action.SHORT_DESCRIPTION, XNap.tr("Applies changes."));
	}
	
        public void actionPerformed(ActionEvent event) 
	{
	    IUser u = uep.getUser();

	    uep.apply();
	    reload();

	    int row = htm.indexOf(u);
	    if (row != -1) {
		jtaHotlist.setRowSelectionInterval(row, row);
	    }
        }
	
    }

}
