/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
import java.io.*;
import java.util.*;

public class Helper
{

    /**
     * Returns the position of the first unescaped occurence of needle in 
     * haystack.
     */
    public static int find(String haystack, String needle)
    {
	int i = 0;
	while (true) {
	    i = haystack.indexOf(needle, i);
	    if (i <= 0) {
		return i;
	    }
	    else if (!haystack.substring(i - 1, i).equals("\\")) {
		return i;
	    }
	    i++;
	}
    }

    public static int findI18nDivider(String haystack)
    {
	return find(haystack, "=");
    }

    public static String getI18nCharset(String filename)
    {
	if (filename.indexOf("_ja") != -1) {
	    return "UTF-8";
	}
	else if (filename.indexOf("_zh") != -1) {
	    return "UTF-8";
	}
	else {
	    return "ISO-8859-1";
	}
    }

    public static Hashtable readI18nFile(String filename,
					 boolean ignoreComments)
	throws IOException
    {
	BufferedReader in = new BufferedReader
	    (new InputStreamReader(new FileInputStream(new File(filename)), 
				   getI18nCharset(filename)));

	System.err.println(getI18nCharset(filename));
	Hashtable table = new Hashtable();
	String s;
	while ((s = in.readLine()) != null) {
	    if (s.startsWith("#")) {
		if (ignoreComments) {
		    continue;
		}
		else {
		    s = s.substring(1);
		}
	    }

	    int i = Helper.findI18nDivider(s);
	    if (i > 0 && i < s.length() - 1) {
		String key = Helper.rawToI18nKey(s.substring(0, i));
		String value = s.substring(i + 1);

		table.put(key, value);
	    }
	}
	in.close();

	return table;
    }

    public static HashSet readKeys() 
    {
	System.out.print("Waiting for input... ");
    
	HashSet keys = new HashSet();
	try {
	    BufferedReader in 
		= new BufferedReader(new InputStreamReader(System.in));
	    String s;
	    while ((s = in.readLine()) != null) {
		if (s.trim().length() > 0) {
		    keys.add(replaceAll(s, "\\\"", "\""));
		}
	    }
	}
	catch (IOException e) {
	    System.err.println("Error: " + e.getMessage());
	    System.exit(1);
	}

	System.out.println("processing " + keys.size() + " keys");

	return keys;
    }
	
    public static String replaceAll(String s, String oldChars, String newChars)
    {
	StringBuffer sb = new StringBuffer();
	int i = 0;
	while (true) {
	    int j = s.indexOf(oldChars, i);
	    if (j != -1) {
		sb.append(s.substring(i, j));
		sb.append(newChars);
		i = j + oldChars.length();
	    }
	    else {
		sb.append(s.substring(i));
		break;
	    }
	}
	return sb.toString();
    }
		
    public static String rawToI18nKey(String key)
    {
	return replaceAll(key, "\\=", "=");
    }

    public static String i18nKeyToRaw(String key)
    {
	return replaceAll(key, "=", "\\=");
    }

    public static String javaToRaw(String s)
    {
	s = replaceAll(s, "\n", "\\n");
	s = replaceAll(s, "\t", "\\t");

	return s;
    }

    public static String rawToJava(String s)
    {
	s = replaceAll(s, "\\n", "\n");
	s = replaceAll(s, "\\t", "\t");

	return s;
    }

    public static void writeI18nFile(String filename, Hashtable table)
	throws IOException
    {
	BufferedWriter out = new BufferedWriter
	    (new OutputStreamWriter(new FileOutputStream(new File(filename)),
				    getI18nCharset(filename)));


	for (Iterator i = table.keySet().iterator(); i.hasNext();) {
	    String key = (String)i.next();
	    String value = (String)table.get(key);
	    if (value != null && value.length() > 0) {
		key = Helper.i18nKeyToRaw(key);

		out.write(Helper.javaToRaw(key));
		out.write("=");
		out.write(Helper.javaToRaw(value));
		out.newLine();
	    }
	}

	out.close();
    }

}
