/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
*/
package gov.nasa.worldwind.servers.wms;

import gov.nasa.worldwind.formats.tiff.GeotiffImageReaderSpi;
import java.io.InputStream;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.Executors;
import java.util.concurrent.ExecutorService;
import javax.imageio.spi.IIORegistry;

/**
 *
 * @author brownrigg
 * @version $Id: WMSServer.java 5011 2008-04-10 16:53:54Z rick $
 */
public class WMSServer {

    public WMSServer() {
        thePort = configuration.getListenerPort();

        threadPool = Executors.newFixedThreadPool(configuration.getThreadPoolSize());

        // register all the maps we know about...
        mapRegistry = new SimpleMapRegistry();
        Set<MapSource> maps = getConfiguration().getMapSources();
        Iterator<MapSource> iter = maps.iterator();

        // make sure our home-grown TIFF reader is registered with the ImageIO API...
        IIORegistry r = IIORegistry.getDefaultInstance();
        r.registerServiceProvider(GeotiffImageReaderSpi.inst());

        while (iter.hasNext()) {
            // initialize each MapSource's MapGenerator; add to registry only if successful...
            try {
                MapSource mapSrc = iter.next();
                registerAndInitMapSources(mapSrc);
                mapRegistry.addMapSource(mapSrc);
            }
            catch (Exception ex) {
                SysLog.inst().error("Could not instantiate or initialize MapSource: " + ex.toString());
            }
        }

        //
        // Open a port and listen indefinitely for requests...
        //
        try {
            ServerSocket socket = new ServerSocket(thePort);
            SysLog.inst().info("Server now listening on port " + thePort);

            while (true) {
                Socket s = socket.accept();
                Thread serviceThread = new WMSServiceThread(s);
                threadPool.execute(serviceThread);
            }

        } catch (Exception ex) {
            SysLog.inst().error("Error on listener socket: " + ex.toString());
            SysLog.inst().error("Server shutdown");
        }
    }

    // gettors...
    public static Configuration getConfiguration() { return configuration; }
    public static MapRegistry getMapSourceRegistry() { return mapRegistry; }

    public static void main(String[] args) {

        InputStream config = WMSServer.class.getClassLoader().getResourceAsStream(CONFIG_FILE );
        if (config == null) {
            SysLog.inst().error(CONFIG_FILE + " not found");
            System.exit(1);
        }
        configuration = new Configuration(config);
        WMSServer service = new WMSServer();
    }
    
    public void registerAndInitMapSources(MapSource mapSrc) {
        try {
            // initialize each named MapSource's MapGenerator; add to registry only if successful...
            if (mapSrc.isNamed()) {
                SysLog.inst().info("Initializing MapSource " + mapSrc.getName() + "...");
                MapGenerator mapGen = mapSrc.getMapGenerator();
                if (mapGen.initialize(mapSrc)) {
                    mapRegistry.addNamedMapSource(mapSrc);
                } else {
                    SysLog.inst().error("Failed to initialize MapSource " + mapSrc.getName() + "; will not be used.");
                }
            }

            // iterate over any children...
            Iterator<MapSource> iter = mapSrc.getChildren();
            while (iter.hasNext()) {
                registerAndInitMapSources((MapSource)iter.next());
            }
        } catch (Exception ex) {
            SysLog.inst().error("Could not instantiate or initialize MapSource: " + ex.toString());
        }
    }

    private int thePort = -1;
    private ExecutorService threadPool;
    private static Configuration configuration;
    private static MapRegistry   mapRegistry;
    private static final String CONFIG_FILE = "WEB-INF/config.xml";
}

