/*
 * gejcon.c - OpenGL/Mesa/Java context for Unix/X11
 * Copyright (c) 1996,97 IICM. All rights reserved.
 *
 * created: mpichler, 19960812
 * changed: mpichler, 19970506
 *
 * $Id: gejcon.c,v 1.18 1997/09/23 12:28:35 mpichler Exp $
 */


#ifdef __cplusplus
extern "C" {
#endif
#include "OGLCanvas.h"
#ifdef __cplusplus
} // C++
#endif
/* #include <ge3d/ge3d.h> */

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmu/WinUtil.h>    /* XmuClientWindow () */

#ifdef USEMESA
#include <mesa/GL/glx.h>
#else
#include <GL/glx.h>
#endif

#include "jutils.h"

/* if defined, a new OpenGL dedicated window will be placed atop the Canvas window
 * this switch has no influence on Mesa
 */
#define NEWWIN

/* only windows with size >= (MINWIDTH x MINHEIGHT) are
 * considered on searching the canvas window
 */
#define MINWIDTH 50
#define MINHEIGHT 50

#ifdef XMESA_MAJOR_VERSION
#define MESA_2_0  /* heavy interface change in XMesa routines */
#endif

#include <stdio.h>  /* debugging */


/* progname */
static char* progname = "OpenGL-Java-context";

/* Display */
static Display* xdpy = NULL;

/* verbose flag */
static jn_int32 gejcverbose = 0;  /* verbose already used by Java */

/* long big enough to hold a pointer */
typedef long ptrsizedint;
#ifdef __osf__
/* 64 bit pointer on DecAlpha */
#define ncontext ncontextL
#define nxcontext nxcontextL
#define nbuffer nbufferL
#define nxbuffer nxbufferL
#endif

/* helper functions */
static Window findAppWin (const char* title);
static Window findCanvas (Window xwin);


/* version control */
#define currentVersion version_1_0_8


/* care for package name */
#ifdef PACKAGE
#define HOGLCanvas name3 (H,PACKAGE,OGLCanvas)
#define OGLCanvas_createContext name2 (PACKAGE,OGLCanvas_createContext)
#define OGLCanvas_setContext name2 (PACKAGE,OGLCanvas_setContext)
#define OGLCanvas_swapBuffers name2 (PACKAGE,OGLCanvas_swapBuffers)
#define OGLCanvas_flushDisplay name2 (PACKAGE,OGLCanvas_flushDisplay)
#define OGLCanvas_mouseInside name2 (PACKAGE,OGLCanvas_mouseInside)
#define OGLCanvas_MESABACKBUF_XIMAGE name2 (PACKAGE,OGLCanvas_MESABACKBUF_XIMAGE)
#define OGLCanvas_MESABACKBUF_PIXMAP name2 (PACKAGE,OGLCanvas_MESABACKBUF_PIXMAP)
#endif


static void makeOpenGLcontext (struct HOGLCanvas* handle, Window cwin, jn_int32 flags);


/*** createContext ***/
/* create graphics context */

void OGLCanvas_createContext (struct HOGLCanvas* handle, jn_int32 flags)
{
  Window xwin, cwin;
  char titlebuf [1024];
  gejcverbose = handle->obj->verbose;


  if (!handle->obj->currentVersion & 0)  /* version control */
    return;

  if (gejcverbose)
#ifdef USEMESA
    fprintf (stderr, "GE3D OpenGL-Java-context: Mesa version\n")
#else
    fprintf (stderr, "GE3D OpenGL-Java-context: OpenGL version\n")
#endif
    ;

  xdpy = XOpenDisplay (NULL);
  if (!xdpy)
  { fprintf (stderr, "%s. Error: cannot open DISPLAY", progname);
    return;
  }

  javaString2CString (handle->obj->title, titlebuf, sizeof (titlebuf));
  if (gejcverbose)
    fprintf (stderr, "title (C-String): `%s'\n", titlebuf);
  xwin = findAppWin (titlebuf);
  if (!xwin)
  { fprintf (stderr, "%s. Error: java application window not found\n", progname);
    return;
  }
  else if (gejcverbose)
    fprintf (stderr, "got application window 0x%x\n", (int) xwin);
  /* both nwindow and ncanvas currently not needed by subsequent functions */
  /* handle to native java window */
  handle->obj->nwindow = (jn_int32) xwin;

  cwin = findCanvas (xwin);
  if (!xwin)
  { fprintf (stderr, "%s. Error: java OpenGL canvas window not found\n", progname);
    return;
  }
  /* handle to native java canvas */
  handle->obj->ncanvas = (jn_int32) cwin;

  handle->obj->ncontext = 0;
  handle->obj->nxcontext = 0;
  makeOpenGLcontext (handle, cwin, flags);
  /* always: ncontext, noglwindow */
  /* OpenGL: colormap (if INSTCOLORMAP) */
  /* Mesa:   nxcontext; nbuffer, nxbuffer (>= 2.0) */

} /* createContext */



/*** setContext ***/
/* activate graphics context */

jn_boolean OGLCanvas_setContext (struct HOGLCanvas* handle, jn_boolean shading)
{
  Window glxwin = (Window) handle->obj->noglwindow;
#ifdef USEMESA
  XMesaContext pixcon = (XMesaContext) handle->obj->ncontext;
  XMesaContext ximcon = (XMesaContext) handle->obj->nxcontext;
  XMesaContext mesacon;
# ifdef MESA_2_0
  XMesaBuffer pixbuf = (XMesaBuffer) handle->obj->nbuffer;
  XMesaBuffer ximbuf = (XMesaBuffer) handle->obj->nxbuffer;
# endif
#else
  GLXContext glxcon = (GLXContext) handle->obj->ncontext;
#endif
  jn_int32 width, height;

  if (!glxwin)
    return 0;  /* false */

  /* native code has its privileges :-): width, height are the private
   * members of Canvas' base class Component */
  width = handle->obj->width;
  height = handle->obj->height;

#ifdef USEMESA
  /* assert: at least one backbuffer available */
  if (shading)  /* shading: prefer ximage backbuffer */
    mesacon = ximcon ? ximcon : pixcon;
  else  /* wireframe: prefer pixmap backbuffer */
    mesacon = pixcon ? pixcon : ximcon;

  handle->obj->nxactive = (mesacon == ximcon);  /* use ximbuf or pixbuf */

# ifdef MESA_2_0
/* fprintf (stderr, "drawing into mesa context %p via buffer %p\n", */
/* mesacon, (mesacon == pixcon) ? pixbuf : ximbuf); */
  if (!XMesaMakeCurrent (mesacon, (mesacon == pixcon) ? pixbuf : ximbuf))
  { fprintf (stderr, "%s. Error: XMesaMakeCurrent failed!\n", progname);
    return 0;
  }
# else
  if (!XMesaMakeCurrent (mesacon))
  { fprintf (stderr, "%s. Error: XMesaMakeCurrent failed!\n", progname);
    return 0;
  }
# endif  /* Mesa */
#else
/* OpenGL */
  /* fprintf (stderr, "drawing into GLXwindow 0x%x via GLXContext %p\n", (int) glxwin, glxcon); */

# if defined(NEWWIN)
  /* resize OpenGL child window if size changed */
  if (width != handle->obj->cwidth || height != handle->obj->cheight)
  {
 /* fprintf (stderr, "resizing window 0x%x to (%d, %d)\n", (int) glxwin, (int) width, (int) height); */
    /* XResizeWindow (xdpy, glxwin, width, height);  -- completely cover parent window: no expose events */
    XMoveResizeWindow (xdpy, glxwin, 1, 1, width-2, height-2);  /* border of parent window for exposure events */
    glXWaitX ();  /* wait for X having resized the window before glXMakeCurrent */
  }
# endif

  if (!glXMakeCurrent (xdpy, glxwin, glxcon))
  { fprintf (stderr, "%s. Error: glXMakeCurrent (OpenGL) failed!\n", progname);
    fprintf (stderr, "  (GLXwindow: 0x%x, glxcontext %p)\n", (int) glxwin, glxcon);
    return 0;
  }
#endif  /* OpenGL */
/* common to OpenGL and Mesa */

/*fprintf (stderr, "canvas size (native): %d x %d\n", (int) width, (int) height);*/

  /* some OpenGL libraries automatically reshape the viewport, others don't */
  glViewport (0, 0, width, height);

  /* for easy access from window */
  handle->obj->cwidth = width;
  handle->obj->cheight = height;

/*fprintf (stderr, "after glViewport\n");*/

  return 1;  /* true */

} /* setContext */



void OGLCanvas_testDrawing ()
{
/* test: do some OpenGL drawings: blue triangle on green background */
  static int runcount = 0;

  glClearColor (0, 1, 0, 0);
  glClear (GL_COLOR_BUFFER_BIT);

fprintf (stderr, "after glClear\n");

  runcount++;
  runcount &= 0xf;

  glColor3f (0, 0, 1);
  glBegin (GL_TRIANGLES);
  glVertex2f (-1 + runcount / 15.0, -1);
  glVertex2i (1, -1);
  glVertex2f (1, runcount / 15.0);
  glEnd ();

} /* testDrawing */


/* swapBuffers */

void OGLCanvas_swapBuffers (struct HOGLCanvas* handle)
{
/*OGLCanvas_testDrawing ();*/
  /* if (double_buffered_) */
#ifdef USEMESA
# ifdef MESA_2_0
  /* must care of active buffer when having two */
/* fprintf (stderr, "swapping buffer %p\n", */
/* (void*) (handle->obj->nxactive) ? handle->obj->nxbuffer : handle->obj->nbuffer); */
  XMesaSwapBuffers ((XMesaBuffer) ((handle->obj->nxactive) ? handle->obj->nxbuffer : handle->obj->nbuffer));
# else
  XMesaSwapBuffers ();
# endif
#else
  glXSwapBuffers (xdpy, (Window) handle->obj->noglwindow);
#endif
  /* else glFlush (); */
} /* swapBuffers */


/* flushDisplay
void OGLCanvas_flushDisplay (struct HOGLCanvas* handle)
{
  XFlush (xdpy);
  XSync (xdpy, False);
}
*/

/* mouseInside */
/* set colormap when mouse enters/leaves canvas */
/* necessary for OpenGL without mulitple colormaps (e.g. on 8bit displays) */
/* could also install a colormap for Mesa (annoying colormap flashing) */

void OGLCanvas_mouseInside (struct HOGLCanvas* handle, jn_int32 flag)
{
#ifdef INSTCOLORMAP
  Colormap color_map = (Colormap) handle->obj->ncolormap;

  /* install colormap when mouse enters */
  if (flag)
    XInstallColormap (xdpy, color_map);
  else
    XUninstallColormap (xdpy, color_map);
  XFlush (xdpy);
#endif
} /* mouseInside */


/* findAppWin */
/* find application window by name (searched on xdpy) */

static Window findAppWin (const char* wintitle)
{
  Window root = RootWindowOfScreen (DefaultScreenOfDisplay (xdpy));
  Window rootout, parent;
  Window* child;
  unsigned int numchildren;

  if (gejcverbose)
    fprintf (stderr, "searching window with title `%s'\n", wintitle);

  if (!XQueryTree (xdpy, root, &rootout, &parent, &child, &numchildren))
  { fprintf (stderr, "%s. Error on XQueryTree.\n", progname);
    return 0;
  }

  if (!child || !numchildren)
  { fprintf (stderr, "%s. Error: No children got for root window.\n", progname);
    return 0;
  }

  while (numchildren--)
  {
    Atom type;
    int format;
    unsigned long nitems, bytesafter;
    unsigned char* title = 0;
    Window w;

    if (gejcverbose)
      fprintf (stderr, "examining window 0x%x\n", (int) child [numchildren]);

    w = XmuClientWindow (xdpy, child [numchildren]);

    XGetWindowProperty (
      xdpy, w, XA_WM_NAME, 0, (65536 / sizeof (long)), False,
      XA_STRING, &type, &format, &nitems, &bytesafter, &title);

    if (gejcverbose)
      fprintf (stderr, "window 0x%x has title: `%s'\n", (int) w, title ? title : (unsigned char*) "null");

    if (title && !strcmp (title, wintitle))  /* found it */
    {
      if (gejcverbose)
        fprintf (stderr, "window 0x%x seems to be the application window.\n", (int) w);
      return w;
    }
  }

  if (gejcverbose)
    fprintf (stderr, "error: window with title %s not found\n", wintitle);

  return 0;

} /* findAppWin */



/* findCanvas */
/* find canvas window that should be handled by OpenGL */
/* because of lack of more information it should pick
 * the most deeply nested subwindow; currently it chooses
 * the deepest window on the first branch by recursion
 */

static void findCanvasRecursive (Window *win, int *level)
{
  Window root, parent;
  Window* child;
  unsigned int numchildren;
  int level_in = *level;
  XWindowAttributes wattrs;

  if (!XQueryTree (xdpy, *win, &root, &parent, &child, &numchildren))
    numchildren = 0;

  while (numchildren--)
  {
    int sublevel = level_in + 1;
    Window subwindow = child [numchildren];
    /* fprintf (stderr, "found subwindow 0x%x at level %d\n", (int) subwindow, sublevel); */

    XGetWindowAttributes (xdpy, subwindow, &wattrs);
    /* fprintf (stderr, "  (sized %d x %d)\n", wattrs.width, wattrs.height); */
    if (wattrs.width < MINWIDTH || wattrs.height < MINHEIGHT)
    {
      /* fprintf (stderr, "  ... too small - ignored\n"); */
      continue;
    }

    findCanvasRecursive (&subwindow, &sublevel);
    if (sublevel > *level)
    {
      *win = subwindow;
      *level = sublevel;
    }
  }
} /* findCanvasRecursive */


static Window findCanvas (Window xwin)
{
  int level = 0;
  findCanvasRecursive (&xwin, &level);

  if (gejcverbose)
    fprintf (stderr, "window 0x%x will be taken as the canvas window.\n", (int) xwin);

  return xwin;

} /* findCanvas */


#if defined(USEMESA) || !defined(NEWWIN)

/* getVisualInfo */
/* little helper: get XVisualInfo for a Window */

static void getVisualInfo (Window win, XVisualInfo* xvisinfo)
{
  XWindowAttributes wattrs;
  Visual* visual;
  int xclass;

  XGetWindowAttributes (xdpy, win, &wattrs);
  visual = wattrs.visual;

  xvisinfo->visual = visual;
  xvisinfo->visualid = visual->visualid;
  xvisinfo->screen = DefaultScreen (xdpy);
  xvisinfo->depth = wattrs.depth;
#ifdef __cplusplus
  xclass = xvisinfo->c_class = visual->c_class;
#else
  xclass = xvisinfo->class = visual->class;
#endif

/* on TrueColor displays wattrs.class was strangely set to GrayScale; visual->class is fine */
  if (gejcverbose)
    fprintf (stderr, "visual class: %s\n",
  (xclass == TrueColor) ? "TrueColor" : (xclass == DirectColor) ? "DirectColor" :
  (xclass == StaticGray) ? "StaticGray" : (xclass == GrayScale) ? "GrayScale" :
  (xclass == PseudoColor) ? "PseudoColor" :
  (xclass == StaticColor) ? "StaticColor" : "UNKNOWN");

  /* Mesa 2.0 accesses near all visual fields ... */
  xvisinfo->red_mask = visual->red_mask;
  xvisinfo->green_mask = visual->green_mask;
  xvisinfo->blue_mask = visual->blue_mask;
  xvisinfo->colormap_size = visual->map_entries;
  xvisinfo->bits_per_rgb = visual->bits_per_rgb;

} /* getVisualInfo */

#endif


#ifdef USEMESA

static XMesaContext firstmesacontext = 0;  /* to share display lists for texturing */


/* createMesaBackBuffer */
/* create Mesa back buffer of ximage or pixmap type */

static void createMesaBackBuffer (Window cwin, XVisualInfo* xvisinfo,
  XMesaContext *mcon, int ximage, const char* ximagestr,
# ifdef MESA_2_0
  XMesaBuffer *mbuf
# else
  int unused
# endif
)
{
# ifdef MESA_2_0
  XMesaVisual mesavis = XMesaCreateVisual (
    xdpy, xvisinfo, 1,  /* rgb mode */
    0,  /* no alpha planes (Mesa 2.0) */
    1, ximage,  /* double buffered, 0:pixmap, 1:ximage */
    1, 0, 0,  /* depth buffer, no stencil, no accum (Mesa 2.0) */
    0  /* level (Mesa 2.0) */
  );

  /* fprintf (stderr, "createMesaBackBuffer (%s)\n", ximagestr); */

  if (!mesavis)
  { fprintf (stderr, "%s. fatal: XMesaCreateVisual failed!\n", progname);
    return;
  }

  *mcon = XMesaCreateContext (mesavis, firstmesacontext);  /* share display lists */
# else
  *mcon = XMesaCreateContext (
    xdpy, xvisinfo, 1,  /* rgb mode */
    1, ximage,  /* double buffered, ximage or pixmap */
    firstmesacontext  /* share display lists */
  );
# endif

  if (!*mcon)
  { fprintf (stderr, "%s. fatal: could not create XMesaContext (%s)!\n", progname, ximagestr);
    return;
  }

  if (!firstmesacontext)
    firstmesacontext = *mcon;

# ifdef MESA_2_0
  *mbuf = XMesaCreateWindowBuffer (mesavis, cwin);
  if (gejcverbose)
    fprintf (stderr, "XMesaCreateWindowBuffer (%p, 0x%x): %p\n", mesavis, (int) cwin, *mbuf);
  if (!*mbuf)
  { fprintf (stderr, "%s. fatal: could not create window buffer (Mesa)!\n", progname);
    return;
  }
# else
  if (*mcon && !XMesaBindWindow (*mcon, cwin))
  { fprintf (stderr, "%s. fatal: binding window failed (XMesaBindWindow)!\n", progname);
    return;
  }
# endif

  if (gejcverbose)
    fprintf (stderr, "createMesaBackBuffer (%s) finished. Window: 0x%x, XMesaContext: %p\n",
      ximagestr, (int) cwin, *mcon);

} /* createMesaBackBuffer */

#endif


/* makeOpenGLcontext */
/* create an OpenGL context for the canvas */
/* this code parallels oglcontext.C/mesacontext.C of VRweb */

/* In case of OpenGL we usually prefer to open a child window atop the
 * canvas window to get a window with the desired OpenGL capabilities;
 * the canvas window opened by Java may support less features or even
 * not supporting GLX/OpenGL at all (double/depth buffering, RGB visual).
 * Event handling and resizing works, expose depends on border of parent window
 *
 * for Mesa we always use the Canvas window itself and
 * create two back buffers (pixmap and ximage) according to OGLCanvas.mesabackbuf
 *
 * always: ncontext, noglwindow
 * OpenGL: ncolormap (if INSTCOLORMAP)
 * Mesa:   nxcontext; nbuffer, nxbuffer (>= 2.0)
 */


static void makeOpenGLcontext (struct HOGLCanvas* handle, Window cwin, jn_int32 flags)
{
  ptrsizedint context;   /* ncontext */
  ptrsizedint xcontext;  /* nxcontext - Mesa only */
  Window glxwin;         /* noglwindow */
  ptrsizedint bufptr = 0L, xbufptr = 0L;  /* nbuffer, nxbuffer - Mesa 2.0 only */
  Colormap color_map = 0;  /* ncolormap, OpenGL w/ INSTCOLORMAP only */

#ifdef USEMESA
/* Mesa, always same window */
  /* yes, I know Mesa implementes the GLX routines, but they are not flexible enough when
   * we want to use the current window which I prefer.
   * Also dynamic selection of the backbuffer method is very useful in terms of speed.
   */

  static XVisualInfo xvisinfo;  /* must be static, because Mesa 2.x keeps a pointer to it */
  XMesaContext pixcon = 0;
  XMesaContext ximcon = 0;
# ifdef MESA_2_0
  XMesaBuffer pixbuf, ximbuf;
# endif

  getVisualInfo (cwin, &xvisinfo);

# ifdef MESA_2_0
#  define FORMESA20(x) x
# else
#  define FORMESA20(x) 0
# endif

  /* create ximage context for shading and/or pixmap context for wireframes */
  /* for unlocated reasons (Java JDK bug?) (unhand (handle)->mesa_backbuf) does not */
  /* contain the same value as flags argument */
  /* fprintf (stderr, "buffers to create: %d\n", (int) handle->obj->mesa_xxx_backbuf); */
  /* fprintf (stderr, "buffers to create: %d\n", (int) flags); */

  if (flags & OGLCanvas_MESABACKBUF_PIXMAP)
    createMesaBackBuffer (cwin, &xvisinfo, &pixcon, 0, "Pixmap", FORMESA20 (&pixbuf));

  if (flags & OGLCanvas_MESABACKBUF_XIMAGE)
    createMesaBackBuffer (cwin, &xvisinfo, &ximcon, 1, "XImage", FORMESA20 (&ximbuf));

  if (!pixcon && !ximcon)
  { fprintf (stderr, "%s. fatal: could create neither backbuffer for Mesa!\n", progname);
    return;
  }

  glxwin = cwin;
  context = (ptrsizedint) pixcon;
  xcontext = (ptrsizedint) ximcon;
  bufptr = (ptrsizedint) pixbuf;
  xbufptr = (ptrsizedint) ximbuf;

#else
# ifdef NEWWIN
/* OpenGL, new window */

  int init_config [32];  /* request a double buffered, z-buffered RGB visual */
  int* icfptr = init_config;
  XVisualInfo* visual;
  GLXContext glxcon;
  int cx, cy;
  unsigned int cwidth, cheight, borderwidth, cdepth;
  XSetWindowAttributes xattrs;
  unsigned long xattrmask;
  Window xw;

  xcontext = 0;  /* not used for OpenGL */

  *icfptr++ = GLX_DOUBLEBUFFER;  /* if double buffering desired */
  *icfptr++ = GLX_RGBA;
  *icfptr++ = GLX_RED_SIZE;
  *icfptr++ = 1;
  *icfptr++ = GLX_GREEN_SIZE;
  *icfptr++ = 1;
  *icfptr++ = GLX_BLUE_SIZE;
  *icfptr++ = 1;
  /* alpha bitplanes
   * *icfptr++ = GLX_ALPHA_SIZE;
   * *icfptr++ = 1;
   */
  *icfptr++ = GLX_DEPTH_SIZE;
  *icfptr++ = 1;
  *icfptr = (int) None;  /* sentinel */

  visual = glXChooseVisual (xdpy, DefaultScreen (xdpy), init_config);
  if (!visual)
  { fprintf (stderr, "%s. fatal: desired GLX visual (RGB with double- and z-buffer) not available!\n", progname);
    return;
  }

  glxcon = glXCreateContext (
    xdpy, visual,
    0,  /* have to share display lists when opening multiple windows */
    1   /* "direct rendering" flag */
  );

  if (!glxcon)
  { fprintf (stderr, "%s. fatal: could not create glx-context!\n", progname);
    return;
  }

  if (gejcverbose)
    fprintf (stderr, "visual used for OpenGL has ID 0x%x\n", (int) visual->visual->visualid);

  /* parent geometry */
  XGetGeometry (xdpy, cwin, &xw, &cx, &cy, &cwidth, &cheight, &borderwidth, &cdepth);
  if (gejcverbose)
    fprintf (stderr, "canvas (parent) geometry: %dx%d%+d%+d\n", cwidth, cheight, cx, cy);

  /* must create separate color map */
  color_map = XCreateColormap (xdpy, RootWindow (xdpy, visual->screen), visual->visual, AllocNone);

  xattrmask = CWColormap;  /* something else? */
  xattrs.colormap = color_map;
  /* Java may set a borderpixel, disliked by OpenGL */
  xattrmask |= CWBorderPixel;
  xattrs.border_pixel = 0;
  /* propagate all events ??? */
  xattrmask |= CWDontPropagate;
  xattrs.do_not_propagate_mask = 0;
  /* events */
  xattrmask |= CWEventMask;
  xattrs.event_mask = 0;
  /* copy cursor from parent */
  xattrmask |= CWCursor;
  xattrs.cursor = None;

  /* create a window */
  if (gejcverbose)
    fprintf (stderr, "creating Xwindow for OpenGL ...\n");
  xw = XCreateWindow (
    xdpy, cwin, 1, 1, /* relative to parent */
    cwidth - 2, cheight - 2, 0, /* border width */
    visual->depth, InputOutput,
    visual->visual,
    xattrmask, &xattrs
  );
  if (gejcverbose)
    fprintf (stderr, "... Xwindow for OpenGL created\n");

  /* map the window */
  XMapRaised (xdpy, xw);

  /* XInstallColormap (xdpy, color_map); */  /* see mouseInside */

  glxwin = xw;

  if (gejcverbose)
    fprintf (stderr, "makeOpenGLcontext finished. GLXwindow: 0x%x, GLXContext: %p\n", (int) glxwin, glxcon);

  context = (ptrsizedint) glxcon;

# else
/* OpenGL, same window */

  GLXContext glxcon;
  XVisualInfo xvisinfo;

  getVisualInfo (cwin, &xvisinfo);

  glxcon = glXCreateContext (
    xdpy, &xvisinfo,
    0,  /* have to share display lists when opening multiple windows */
    1   /* "direct rendering" flag */
  );

  if (!glxcon)
  { fprintf (stderr, "%s. fatal: could not create glx-context (OpenGL)!\n", progname);
    return;
  }

  glxwin = cwin;

  if (gejcverbose)
    fprintf (stderr, "makeOpenGLcontext finished. GLXwindow: 0x%x, GLXContext: %p\n", (int) glxwin, glxcon);

  context = (ptrsizedint) glxcon;

# endif
#endif  /* OpenGL */

  handle->obj->ncontext = context;
  handle->obj->nxcontext = xcontext;
  handle->obj->noglwindow = (jn_int32) glxwin;
  handle->obj->nbuffer = bufptr;
  handle->obj->nxbuffer = xbufptr;
  handle->obj->ncolormap = (jn_int32) color_map;

} /* makeOpenGLcontext */
