/*
 * Picker.java
 *
 * created: mpichler, 19970708
 *
 * changed: mpichler, 19970811
 * changed: apesen, 19970910
 * changed: kwagen, 19970917
 *
 * $Id: Picker.java,v 1.18 1997/09/29 08:19:39 apesen Exp $
 */


package iicm.vrml.vrwave;

import iicm.ge3d.GE3D;
import iicm.vrml.pw.*;
import iicm.vrml.vrwave.pwdat.*;
import iicm.utils3d.Hitpoint;
import iicm.utils3d.PickUtil;
import iicm.utils3d.Ray;
import iicm.utils3d.Mat4f;
import iicm.utils3d.Vec3f;

import java.util.Vector;


/**
 * Picker - picking traversal
 * Copyright (c) 1997 IICM
 *
 * @author Michael Pichler
 * @version 0.8, latest change:  1 Aug 97
 */


class Picker extends Traverser
{
  Scene scene_;
  Ray ray_;
  boolean twosided_;
  VHitpoint hitpoint_;
  Node hitnode_;    // node hit
  float hittimeparent_;
  boolean hitanytrf_;
  boolean dragsens_ = false;  // pick geometries, not dragsensors
  boolean keeptrf_;

  /**
   * pick the scene graph. fills out Hitpoint.
   * if flag dragsens is set, pick only dragsensors
   * if flag dragsens is not set, pick geometries
   * @return node hit object
   */

  Node pickScene (Scene s, GroupNode root, Ray ray, VHitpoint hit, boolean dragsens, boolean keeptrf)
  {
    if (root == null)
      return null;

    scene_ = s;
    ray_ = ray;
    hitpoint_ = hit;
    hitnode_ = null;
    hit.hitparent_ = null;
    hittimeparent_ = hit.hittime_;
    hitanytrf_ = false;
    dragsens_ = dragsens;
    keeptrf_ = keeptrf;

    // when there is no transform node, object and world coordinates are identical
    hit.raystartobj_ = ray.start_;
    hit.raydirobj_ = ray.direction_;
    if (keeptrf_)  // calculate tranformation matrix for sensor picking
    {
      // assert: current matrix is the one of last draw
      GE3D.getMatrix (hit.viewingmat_);
      hit.hit_trfmat_ = Mat4f.identity4d;
    }

    hit.hitpath_.removeAllElements ();  // contains parent chain from object hit up to root

    // VRML 2.0: single sided by default
    twosided_ = (s.backfaceCulling () == Scene.TRISTATE_OFF);

    tGroupNode (root);  // pick scene graph

    // calculate hitpoint
    if (hitnode_ != null)
    {
      hit.hitpoint_ = ray.at (hit.hittime_);

      // hit normal already in world coordinates (see tTransform)
      if (hit.normal_ != null)
      {
        hit.normal_.normalize ();
        if (!hitanytrf_)  // scene graph contained no Transform node
          hit.normalobj_.assign (hit.normal_);
        else
          hit.normalobj_.normalize ();
      }
      // System.err.println ("hitpoint: " + hit.hitpoint_ + " normal: " + hit.normal_);
    }
    else
    {
      hit.hitpoint_ = null;
      // System.err.println ("no hit");
    }
    // System.err.println ("parent of node hit: " + hit.hitparent_);

    return hitnode_;

  } // pickScene

  public void tGroupNode (GroupNode group)
  {
    VHitpoint hit = hitpoint_;
    float hitin = hit.hittime_;

    super.tGroupNode (group);

    float hitout = hit.hittime_;
    if (hitout != hitin)  // a hit occured inside group
    {
      if (hitout != hittimeparent_)  // deepest (innermost) group
      {
        hit.hitparent_ = group;
        // System.err.println ("hit parent " + group + " for hittime " + hit.hittime_);
        hittimeparent_ = hit.hittime_;
        hit.hitpath_.removeAllElements ();
      }
      hit.hitpath_.addElement (group);
    }
  }

  // TODO: bounding box tests at object level
  // (or at group level when explicitly given)

// void tGroup (Group g)
// {
//   System.out.println ("picking Group");
//   tGroupNode (g);
// }

  // Grouping nodes: default behaviour - traverse children
  // void tGroup (Group g)  { tGroupNode (g); }
  // void tAnchor (Anchor g)  { tGroupNode (g); }
  // void tBillboard (Billboard g)  { tGroupNode (g); }
  // void tCollision (Collision g)  { tGroupNode (g); }

  protected void tBillboard (Billboard billb)
  {
    Ray ray = ray_;
    Vec3f start = ray.start_;
    Vec3f direction = ray.direction_;
    VHitpoint hit = hitpoint_;
    BillboardData dat = (BillboardData) billb.userdata;

    // transform ray into child coordinate system
    float[] invmat = new float[16];
    if (!Mat4f.invertMatrix44 (dat.trfmatrix_, invmat))
      invmat = Mat4f.identity4d;
    ray.start_ = Mat4f.transformPoint3Mat44 (start.value_, invmat);
    ray.direction_ = Mat4f.transformVector3Mat44 (direction.value_, invmat);
    // this might be moved to native code for efficiency,
    // but bounding box tests will be more useful

    Node hitnode = hitnode_;
    float hitin = hit.hittime_;

    super.tGroupNode (billb);  // pick children

    if (hitnode != hitnode_)  // hit occured within children
    {
      // as in this.tGroupNode
      float hitout = hit.hittime_;
      if (hitout != hittimeparent_)  // deepest (innermost) group
      {
        hit.hitparent_ = billb;
        // System.err.println ("hit parent " + group + " for hittime " + hit.hittime_);
        hittimeparent_ = hit.hittime_;

        // save ray equation in object coordinate system
        hit.raystartobj_ = ray.start_;
        hit.raydirobj_ = ray.direction_;
        if (keeptrf_)  // calculate tranformation matrix for sensor picking
        {
          hit.hit_trfmat_ = Mat4f.multiplyMat43in44 (dat.trfmatrix_, hit.hit_trfmat_);
        }

        if (hit.normal_ != null)
          hit.normalobj_.assign (hit.normal_);  // normal vector in object coordinate system
        hitanytrf_ = true;
        hit.hitpath_.removeAllElements ();
      }
      hit.hitpath_.addElement (billb);

      if (hit.normal_ != null)
      {
        // transform hitpoint normal by inverse transposed 3x3 matrix
        // to check: shall the normal vector be normalized before transforming it (was not done in VRweb)?
        hit.normal_ = Mat4f.transformVector3Mat44transp (hit.normal_.value_, invmat);
      }
    }

    // back to parent coordinate system (far_ is updated during picking)
    ray_.start_ = start;
    ray_.direction_ = direction;
  } // tBillboard

  protected void tTransform (Transform trf)
  {
    Ray ray = ray_;
    Vec3f start = ray.start_;
    Vec3f direction = ray.direction_;
    VHitpoint hit = hitpoint_;

    // transform ray into child coordinate system
    float[] invmat = ((TransformData) trf.userdata).getInvMatrix ();
    ray.start_ = Mat4f.transformPoint3Mat44 (start.value_, invmat);
    ray.direction_ = Mat4f.transformVector3Mat44 (direction.value_, invmat);
    // this might be moved to native code for efficiency,
    // but bounding box tests will be more useful

    Node hitnode = hitnode_;
    float hitin = hit.hittime_;

    super.tGroupNode (trf);  // pick children

    if (hitnode != hitnode_)  // hit occured within children
    {
      // as in this.tGroupNode
      float hitout = hit.hittime_;
      if (hitout != hittimeparent_)  // deepest (innermost) group
      {
        hit.hitparent_ = trf;
        // System.err.println ("hit parent " + group + " for hittime " + hit.hittime_);
        hittimeparent_ = hit.hittime_;

        // save ray equation in object coordinate system
        hit.raystartobj_ = ray.start_;
        hit.raydirobj_ = ray.direction_;
        if (keeptrf_)  // calculate tranformation matrix for sensor picking
        {
          hit.hit_trfmat_ = Mat4f.multiplyMat43in44 (hit.hit_trfmat_, ((TransformData) trf.userdata).getMatrix ());
        }

        if (hit.normal_ != null)
          hit.normalobj_.assign (hit.normal_);  // normal vector in object coordinate system
        hitanytrf_ = true;
        hit.hitpath_.removeAllElements ();
      }
      hit.hitpath_.addElement (trf);

      if (hit.normal_ != null)
      {
        // transform hitpoint normal by inverse transposed 3x3 matrix
        // to check: shall the normal vector be normalized before transforming it (was not done in VRweb)?
        hit.normal_ = Mat4f.transformVector3Mat44transp (hit.normal_.value_, invmat);
      }
    }

    // back to parent coordinate system (far_ is updated during picking)
    ray_.start_ = start;
    ray_.direction_ = direction;
  } // Transform

  // Inline
  protected void tInline (Inline inline)
  { 
    InlineData dat = (InlineData) inline.userdata;

    if (dat.inline_ != null)  // inline fetched
    {
      tGroupNode (dat.inline_);
    }
    else  // might pick bounding box here
    {
    }
  }

  // LOD
  protected void tLOD (LOD lod)
  {
    Vector level = lod.level.getNodes ();
    // pick level shown last
    int active = ((LODdata) lod.userdata).drawnchild_;
    if (active >= 0 && active < level.size ())
      ((Node) level.elementAt (active)).traverse (this);

  } // LOD

  // Switch
  protected void tSwitch (Switch sw)
  {
    Vector choice = sw.choice.getNodes ();
    // pick child shown last
    int whichchoice = ((SwitchData) sw.userdata).drawnchild_;
    if (whichchoice >= 0 && whichchoice < choice.size ())
      ((Node) choice.elementAt (whichchoice)).traverse (this);
  }

  // Sound
  protected void tAudioClip (AudioClip n)  { }
  protected void tSound (Sound n)  { }


  // *** Lighting ***
  // lights are not picked
  protected void tDirectionalLight (DirectionalLight lgt)  { }
  protected void tPointLight (PointLight lgt)  { }
  protected void tSpotLight (SpotLight n)  { }


  // *** Scripting ***
  // script nodes are not picked
  protected void tScript (Script n)  { }

  // WorldInfo not picked
  protected void tWorldInfo (WorldInfo n)  { }


  // *** Sensor Nodes ***

  protected void tCylinderSensor (CylinderSensor cyls)  
  {
    if (!dragsens_)
      return;

    if (cyls.dragMode_ == 0)
      return;

    if (cyls.dragMode_ == CylinderSensor.CYLINDERMODE)
    {
//      if (rayhitsparaplane (ray_, hitpoint_, cyls))  
//      {
//        hitnode_ = cyls;
//        ray_.far_ = hitpoint_.hittime_;  // search next hit
//
//      }
      return;
    }

    if (cyls.dragMode_ == CylinderSensor.DISKMODE)
    {
      if (PickUtil.rayhitsdisk (ray_, 0.0f, 0.0f, true, true, hitpoint_, true))  // bottom
      {
        hitnode_ = cyls;
        ray_.far_ = hitpoint_.hittime_;  // search next hit
      }
    }
  }

  protected void tPlaneSensor (PlaneSensor pls)  
  {
    if (!dragsens_)
      return;

    if (PickUtil.rayhitsplane (ray_, hitpoint_))  
    {
      hitnode_ = pls;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }
    // System.out.println ("Pick Plane:" + pls);
  }

  protected void tProximitySensor (ProximitySensor n)  { }
  protected void tSphereSensor (SphereSensor sphs) { }  // no picking needed
  protected void tTimeSensor (TimeSensor n)  { }
  protected void tTouchSensor (TouchSensor n)  { }
  protected void tVisibilitySensor (VisibilitySensor n)  { }


  // *** Geometry Nodes ***

  // Shape

  protected void tShape (Shape shape)
  {
    if (dragsens_)  // return if only dragsensors are of interest
      return;
    Node geom = shape.geometry.getNode ();
    if (geom != null)  // pick geometry
      geom.traverse (this);
  } // Shape


  // Box

  public void tBox (Box box)
  {
    BoxData dat = (BoxData) box.userdata;

    if (PickUtil.rayhitscube (ray_, dat.min_, dat.max_, hitpoint_, false)  // from outside
      || (twosided_ && PickUtil.rayhitscube (ray_, dat.min_, dat.max_, hitpoint_, true)))  // from inside
    {
      hitnode_ = box;
      // hitpoint_.hittime_ and normal_ already set by rayhitscube (be aware when using it for bounding box tests!)
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }
    // System.err.println ("Box hit: " + (hitnode_ == box));
  } // Box


  // Cone

  protected void tCone (Cone cone)
  {
    float height = cone.height.getValue ();
    float radius = cone.bottomRadius.getValue ();
    // hitpoint_.hittime_ and normal_ set by rayhitsdisk and rayhitsconeside

    if (cone.bottom.getValue () &&
      PickUtil.rayhitsdisk (ray_, -height/2.0f, radius, twosided_, true, hitpoint_, false))  // bottom
    {
      hitnode_ = cone;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }

    if (cone.side.getValue () &&
      PickUtil.rayhitsconeside (ray_, height, radius, twosided_, hitpoint_))  // side
    {
      hitnode_ = cone;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }
  } // Cone


  // Cylinder

  protected void tCylinder (Cylinder cyl)
  {
    float height = cyl.height.getValue ();
    float radius = cyl.radius.getValue ();
    // hitpoint_.hittime_ and normal_ set by rayhitsdisk and rayhitscylinderside

    if (cyl.top.getValue () &&
      PickUtil.rayhitsdisk (ray_, height/2.0f, radius, true, twosided_, hitpoint_, false))  // top
    {
      hitnode_ = cyl;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }

    if (cyl.bottom.getValue () &&
      PickUtil.rayhitsdisk (ray_, -height/2.0f, radius, twosided_, true, hitpoint_, false))  // bottom
    {
      hitnode_ = cyl;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }

    if (cyl.side.getValue () &&
      PickUtil.rayhitscylinderside (ray_, height, radius, twosided_, hitpoint_, false))  // side
    {
      hitnode_ = cyl;
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }
  } // Cylinder


  protected void tElevationGrid (ElevationGrid n)  { }


  // Extrusion

  protected void tExtrusion (Extrusion extr)
  {
    if (extr.crossSection.getValueCount () < 1 || extr.spine.getValueCount () < 1)
      return;

    ExtrusionData dat = (ExtrusionData) extr.userdata;
    Ray ray = ray_;
    Vec3f hitnormal = hitpoint_.normal_;

    float hit = rayhitsfaceset (ray.start_.value_, ray.direction_.value_, ray.near_, ray.far_,
      dat.coords, dat.numcoordinds, dat.coordinds, dat.facenormals, twosided_,
      (hitnormal != null) ? hitnormal.value_ : null);  // updates hitpoint_.normal_ in case of hit, returns hittime

    if (hit != 0.0f)
    {
      hitnode_ = extr;
      ray.far_ = hitpoint_.hittime_ = hit;  // search next hit
    }
  } // Extrusion


  // IndexedFaceSet

  protected void tIndexedFaceSet (IndexedFaceSet faceset)
  {
    IndexedFaceSetData dat = (IndexedFaceSetData) faceset.userdata;
    MFVec3f coords = dat.coords_;
    if (coords == null || dat.fnormals_ == null)
      return;

    Ray ray = ray_;
    MFInt32 cinds = faceset.coordIndex;
    Vec3f hitnormal = hitpoint_.normal_;

    // a bounding box test would be useful here
    float hit = rayhitsfaceset (ray.start_.value_, ray.direction_.value_, ray.near_, ray.far_,
      coords.getValueData (), cinds.getValueCount (), cinds.getValueData (), dat.fnormals_.getData (), twosided_,
      (hitnormal != null) ? hitnormal.value_ : null);  // updates hitpoint_.normal_ in case of hit, returns hittime

    if (hit != 0.0f)
    {
      hitnode_ = faceset;
      ray.far_ = hitpoint_.hittime_ = hit;  // search next hit
    }
  } // IndexedFaceSet

  native float rayhitsfaceset (float[] raystart, float[] raydir, float raynear, float rayfar,
    float[] verts, int numcoordinds, int[] coordinds, float[] fnormals, boolean ccw,
    float[] hitnormal);

  protected void tIndexedLineSet (IndexedLineSet n)  { }
  protected void tPointSet (PointSet n)  { }


  // Sphere

  protected void tSphere (Sphere sphere)
  {
    float radius = sphere.radius.getValue ();
    if (PickUtil.rayhitssphere (ray_, radius, twosided_, hitpoint_))
    {
      hitnode_ = sphere;  // hitpoint_.hittime_ and normal_ set by rayhitssphere
      ray_.far_ = hitpoint_.hittime_;  // search next hit
    }
  } // Sphere

  protected void tText (Text n)  { }


  // *** Geometric Properties: not relevant for picking
  protected void tColor (Color n)  { }
  protected void tCoordinate (Coordinate n)  { }
  protected void tNormal (Normal n)  { }
  protected void tTextureCoordinate (TextureCoordinate n)  { }

  // *** Appearance Nodes: not relevant for picking
  protected void tAppearance (Appearance n)  { }
  protected void tFontStyle (FontStyle n)  { }
  protected void tMaterial (Material n)  { }
  protected void tImageTexture (ImageTexture n)  { }
  protected void tMovieTexture (MovieTexture n)  { }
  protected void tPixelTexture (PixelTexture n)  { }
  protected void tTextureTransform (TextureTransform n)
  { // will become relevant when calculating texture coordinates too
  }

  // *** Interpolator Nodes ***
  protected void tColorInterpolator (ColorInterpolator n)  { }
  protected void tCoordinateInterpolator (CoordinateInterpolator n)  { }
  protected void tNormalInterpolator (NormalInterpolator n)  { }
  protected void tOrientationInterpolator (OrientationInterpolator n)  { }
  protected void tPositionInterpolator (PositionInterpolator n)  { }
  protected void tScalarInterpolator (ScalarInterpolator n)  { }

  // *** Bindable Nodes ***
  protected void tBackground (Background n)  { }
  protected void tFog (Fog n)  { }
  protected void tNavigationInfo (NavigationInfo n)  { }
  protected void tViewpoint (Viewpoint n)  { }

  // *** instance of a PROTO node ***
  protected void tProtoInstance (ProtoInstance n)  { }
} // Picker
