/* **********************************************************
 * Copyright 1998 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

#ifndef _X86_H_
#define _X86_H_

#define INCLUDE_ALLOW_USERLEVEL
#define INCLUDE_ALLOW_VMMEXT
#define INCLUDE_ALLOW_MODULE
#define INCLUDE_ALLOW_VMMON
#define INCLUDE_ALLOW_VMNIXMOD
#define INCLUDE_ALLOW_VMK_MODULE
#define INCLUDE_ALLOW_VMKERNEL
#define INCLUDE_ALLOW_DISTRIBUTE
#define INCLUDE_ALLOW_VMCORE
#include "includeCheck.h"

#include "x86types.h"
#include "x86apic.h"
#include "x86desc.h"
#include "x86cpuid.h"
#include "vmware_pack_init.h"


#define SIZE_8BIT   1
#define SIZE_16BIT  2
#define SIZE_24BIT  3
#define SIZE_32BIT  4
#define SIZE_48BIT  6
#define SIZE_64BIT  8
#define SIZE_80BIT  10
#define SIZE_128BIT 16
#define SIZE_M512   512


#define X86_MAX_INSTR_LEN  15   /* Max byte length of an x86 instruction. */


#define NUM_IDT_VECTORS 256


/*
 * simple structures are defined but shift and masks where more
 * complicated structures use structs and bitmasks
 */


/*
 *   control registers
 */

#define CR0_PE         0x00000001
#define CR0_MP         0x00000002
#define CR0_EM         0x00000004
#define CR0_TS         0x00000008
#define CR0_ET         0x00000010
#define CR0_NE         0x00000020
#define CR0_WP         0x00010000
#define CR0_AM         0x00040000
#define CR0_NW         0x20000000
#define CR0_CD         0x40000000
#define CR0_PG         0x80000000
#define CR0_RESERVED   ((UReg)CONST64U(0xffffffff1ffaffc0))

#define CR3_PWT        0x00000008
#define CR3_PCD        0x00000010
#define CR3_PDB_SHIFT  12
#define CR3_PDB_MASK   0xfffff000
#define CR3_IGNORE     0xFFF
#define PAE_CR3_IGNORE 0x1F

#define CR4_VME        0x00000001
#define CR4_PVI        0x00000002
#define CR4_TSD        0x00000004
#define CR4_DE         0x00000008
#define CR4_PSE        0x00000010
#define CR4_PAE        0x00000020
#define CR4_MCE        0x00000040
#define CR4_PGE        0x00000080
#define CR4_PCE        0x00000100

#define CR4_OSFXSR     0x00000200 // CPU/OS supports SIMD insts
#define CR4_OSXMMEXCPT 0x00000400 // #XF exception enable PIII only
#define CR4_VMXE       0x00002000

#define CR4_RESERVED   ((UReg)CONST64U(0xfffffffffffff800))
#define CR4_NOT_IMP    0x00000000

#define CR4_RESERVED_WO_SSE   (CR4_OSXMMEXCPT)
#define CR4_NOT_IMP_WO_SSE    (CR4_OSXMMEXCPT)

#define CR4_RESERVED_WO_VME   (CR4_VME | CR4_PVI)

#define CR8_RESERVED   0xfffffffffffffff0ULL

#define EFLAGS_IOPL_SHIFT 12
#define FLAG_DEF \
FLG(CF, 0),                        /* User */ \
FLG(SET, 1),                                  \
FLG(PF, 2),                        /* User */ \
FLG(AF, 4),                        /* User */ \
FLG(ZF, 6),                        /* User */ \
FLG(SF, 7),                        /* User */ \
FLG(TF, 8),                        /* Priv */ \
FLG(IF, 9),                        /* Priv */ \
FLG(DF, 10),                       /* User */ \
FLG(OF, 11),                       /* User */ \
FLG(NT, 14),                       /* Priv */ \
FLG(RF, 16),                       /* Priv */ \
FLG(VM, 17),                       /* Priv */ \
FLG(AC, 18),                       /* Priv */ \
FLG(VIF, 19),                      /* Priv */ \
FLG(VIP, 20),                      /* Priv */ \
FLG(ID, 21),                       /* Priv */

typedef enum x86_FLAGS {
   EFLAGS_NONE         = 0,
#define FLG(nm,num) EFLAGS_ ## nm = (1 << num)
   FLAG_DEF
#undef FLG
   EFLAGS_IOPL         = 3 << EFLAGS_IOPL_SHIFT,        /* Priv */
   EFLAGS_USER         = (EFLAGS_CF | EFLAGS_PF | EFLAGS_AF | EFLAGS_ZF |
                          EFLAGS_SF | EFLAGS_DF | EFLAGS_OF),
   EFLAGS_PRIV         = (EFLAGS_TF  | EFLAGS_IF  | EFLAGS_IOPL | EFLAGS_NT  |
                          EFLAGS_RF  | EFLAGS_VM  | EFLAGS_AC   | EFLAGS_VIF |
                          EFLAGS_VIP | EFLAGS_ID),
   EFLAGS_ALL          = EFLAGS_USER | EFLAGS_PRIV,

   EFLAGS_REAL_32      = (EFLAGS_ALL & ~(EFLAGS_VIP | EFLAGS_VIF | EFLAGS_VM)),
   EFLAGS_V8086_32     = (EFLAGS_ALL & ~(EFLAGS_VIP | EFLAGS_VIF |
                                         EFLAGS_VM  | EFLAGS_IOPL)),
   EFLAGS_ALL_16       = EFLAGS_ALL & 0xffff,
   EFLAGS_REAL_16      = EFLAGS_REAL_32 & 0xffff,
   EFLAGS_V8086_16     = EFLAGS_V8086_32 & 0xffff,
   EFLAGS_CLEAR_ON_EXC = (EFLAGS_TF | EFLAGS_VM | EFLAGS_RF | EFLAGS_NT),
   EFLAGS_PRIV_DE      = (EFLAGS_NT | EFLAGS_TF | EFLAGS_ID | EFLAGS_AC | 
                          EFLAGS_VM),
   EFLAGS__4           = 0x7fffffff,   /* ensure 4 byte encoding */
} x86_FLAGS;

typedef enum x86_64_FLAGS {
#define FLG(nm,num) RFLAGS_ ## nm = ( 1 << num )
   FLAG_DEF
#undef FLG
   RFLAGS_IOPL         = 3 << EFLAGS_IOPL_SHIFT,        /* Priv */
   RFLAGS_USER         = EFLAGS_USER | 0, /* any 64-bit specific flags here */
   RFLAGS_PRIV         = EFLAGS_PRIV | 0, /* any 64-bit specific flags here */
   RFLAGS_ALL          = RFLAGS_USER | RFLAGS_PRIV
} x86_64_FLAGS;

#define DR6_B0         0x00000001
#define DR6_B1         0x00000002
#define DR6_B2         0x00000004
#define DR6_B3         0x00000008
#define DR6_B0123      (DR6_B0 | DR6_B1 | DR6_B2 | DR6_B3)
#define DR6_BD         0x00002000
#define DR6_BS         0x00004000
#define DR6_BT         0x00008000
#define DR6_ONES       0xffff0ff0
#define DR6_RESERVED_MASK 0xffff1ff0

#define DR7_L0         0x00000001
#define DR7_G0         0x00000002
#define DR7_L1         0x00000004
#define DR7_G1         0x00000008
#define DR7_L2         0x00000010
#define DR7_G2         0x00000020
#define DR7_L3         0x00000040
#define DR7_G3         0x00000080
#define DR7_ENABLED    0x000000ff

#define DR7_LE         0x00000100
#define DR7_GE         0x00000200
#define DR7_GD         0x00002000
#define DR7_ONES       0x00000400
#define DR7_DEFAULT    (DR7_LE | DR7_GE | DR7_ONES)

#define DR7_RW(_r,_n)  (((_r) >> (16+(_n)*4)) & 0x3)
#define DR7_L(_r,_n)   (((_r) >> ((_n)*2)) & 1)
#define DR7_G(_r,_n)   (((_r) >> (1 + (_n)*2)) & 1)
#define DR7_LEN(_r,_n) (((_r) >> (18+(_n)*4)) & 0x3)

#define DR7_RW_INST    0x0
#define DR7_RW_WRITES  0x1
#define DR7_RW_IO      0x2
#define DR7_RW_ACCESS  0x3

#define MSR_TSC               0x00000010
#define MSR_PLATFORM_ID       0x00000017
#define MSR_APIC_BASE         0x0000001b
#define MSR_FEATCTL           0x0000003a
#define MSR_BIOS_SIGN_ID      0x0000008b
#define MSR_PERFCTR0          0x000000c1
#define MSR_PERFCTR1          0x000000c2
#define MSR_MTRR_CAP          0x000000fe
#define MSR_L2CFG             0x0000011e
#define MSR_SYSENTER_CS       0x00000174
#define MSR_SYSENTER_ESP      0x00000175
#define MSR_SYSENTER_EIP      0x00000176
#define MSR_MCG_CAP           0x00000179
#define MSR_MCG_STATUS        0x0000017a
#define MSR_MCG_CTL           0x0000017b
#define MSR_EVNTSEL0          0x00000186
#define MSR_EVNTSEL1          0x00000187
#define MSR_DEBUGCTL          0x000001d9
#define MSR_EFER              0xc0000080
#define MSR_FSBASE            0xc0000100
#define MSR_GSBASE            0xc0000101
#define MSR_KERNELGSBASE      0xc0000102
#define MSR_TSC_AUX           0xc0000103

#define MSR_MTRR_BASE0        0x00000200
#define MSR_MTRR_MASK0        0x00000201
#define MSR_MTRR_BASE1        0x00000202
#define MSR_MTRR_MASK1        0x00000203
#define MSR_MTRR_BASE2        0x00000204
#define MSR_MTRR_MASK2        0x00000205
#define MSR_MTRR_BASE3        0x00000206
#define MSR_MTRR_MASK3        0x00000207
#define MSR_MTRR_BASE4        0x00000208
#define MSR_MTRR_MASK4        0x00000209
#define MSR_MTRR_BASE5        0x0000020a
#define MSR_MTRR_MASK5        0x0000020b
#define MSR_MTRR_BASE6        0x0000020c
#define MSR_MTRR_MASK6        0x0000020d
#define MSR_MTRR_BASE7        0x0000020e
#define MSR_MTRR_MASK7        0x0000020f
#define MSR_MTRR_FIX64K_00000 0x00000250
#define MSR_MTRR_FIX16K_80000 0x00000258
#define MSR_MTRR_FIX16K_A0000 0x00000259
#define MSR_MTRR_FIX4K_C0000  0x00000268
#define MSR_MTRR_FIX4K_C8000  0x00000269
#define MSR_MTRR_FIX4K_D0000  0x0000026a
#define MSR_MTRR_FIX4K_D8000  0x0000026b
#define MSR_MTRR_FIX4K_E0000  0x0000026c
#define MSR_MTRR_FIX4K_E8000  0x0000026d
#define MSR_MTRR_FIX4K_F0000  0x0000026e
#define MSR_MTRR_FIX4K_F8000  0x0000026f
#define MSR_MTRR_DEF_TYPE     0x000002ff

#define MSR_MC0_CTL          0x00000400
#define MSR_MC0_STATUS       0x00000401
#define MSR_MC0_ADDR         0x00000402
#define MSR_MC0_MISC         0x00000403

#define MSR_LASTBRANCHFROMIP 0x000001db // Intel P6 Family
#define MSR_LASTBRANCHTOIP   0x000001dc // Intel P6 Family
#define MSR_LASTINTFROMIP    0x000001dd // Intel P6 Family
#define MSR_LASTINTTOIP      0x000001de // Intel P6 Family

#define MSR_LER_FROM_LIP     0x000001d7 // Intel Pentium4 Family
#define MSR_LER_TO_LIP       0x000001d8 // Intel Pentium4 Family
#define MSR_LASTBRANCH_TOS   0x000001da // Intel Pentium4 Family
#define MSR_LASTBRANCH_0     0x000001db // Intel Pentium4 Family
#define MSR_LASTBRANCH_1     0x000001dc // Intel Pentium4 Family
#define MSR_LASTBRANCH_2     0x000001dd // Intel Pentium4 Family
#define MSR_LASTBRANCH_3     0x000001de // Intel Pentium4 Family

/* DebugCtlMSTR bits */
#define MSR_DEBUGCTL_LBR     0x00000001

/* Feature control bits */
#define MSR_FEATCTL_LOCK     0x00000001
#define MSR_FEATCTL_SMXE     0x00000002
#define MSR_FEATCTL_VMXE     0x00000004

/* MSR_EFER bits. */
#define MSR_EFER_SCE         0x0000000000000001ULL  /* Sys call ext'ns:  r/w */
#define MSR_EFER_RAZ         0x00000000000000feULL  /* Read as zero          */
#define MSR_EFER_LME         0x0000000000000100ULL  /* Long mode enable: r/w */
#define MSR_EFER_LMA         0x0000000000000400ULL  /* Long mode active: r/o */
#define MSR_EFER_NXE         0x0000000000000800ULL  /* No-exec enable:   r/w */
#define MSR_EFER_TRUNCE      0x0000000000002000ULL  /* trun. seg. enable:r/w */
#define MSR_EFER_FFXSR       0x0000000000004000ULL  /* Fast FXSAVE:      r/w */
#define MSR_EFER_MBZ         0xffffffffffff9200ULL  /* Must be zero (resrvd) */

/* This ifndef is necessary because this is defined by some kernel headers. */
#ifndef MSR_K7_HWCR
#define MSR_K7_HWCR          0xc0010015 // AMD Athlon Family
#endif
#define MSR_K7_HWCR_SSEDIS   0x00008000 // Disable SSE bit
#define MSR_K7_HWCR_TLBFFDIS 0x00000040 // Disable TLB Flush Filter

/* Syscall/Sysret related MSRs (x86_64) */
#define MSR_STAR             0xc0000081 // Also present on Athlons.
#define MSR_LSTAR            0xc0000082
#define MSR_CSTAR            0xc0000083
#define MSR_SFMASK           0xc0000084

/*
 * MTRR bit description
 */
#define MTRR_CAP_WC           0x400
#define MTRR_CAP_FIX          0x100
#define MTRR_CAP_VCNT_MASK    0xff

#define MTRR_DEF_ENABLE       0x800
#define MTRR_DEF_FIXED_ENABLE 0x400
#define MTRR_DEF_TYPE_MASK    0xff

#define MTRR_BASE_TYPE_MASK   0xff
#define MTRR_BASE_ADDR_MASK   (0xffffff000LL)

#define MTRR_MASK_VALID       0x800
#define MTRR_MASK_ADDR_MASK   (0xffffff000LL)

#define MTRR_TYPE_UC          0
#define MTRR_TYPE_WC          1
#define MTRR_TYPE_WT          4
#define MTRR_TYPE_WP          5
#define MTRR_TYPE_WB          6

typedef uint32 CReg;


/*
 *   segment selectors
 */

#define SELECTOR_GDT             0
#define SELECTOR_LDT             1
#define SELECTOR_RPL_SHIFT       0
#define SELECTOR_RPL_MASK        0x03u
#define SELECTOR_TI_SHIFT        2
#define SELECTOR_TI_MASK         0x4
#define SELECTOR_INDEX_SHIFT     3
#define SELECTOR_INDEX_MASK      0xfff8

#define SELECTOR_RPL(_sel)       (((Selector)(_sel)) & SELECTOR_RPL_MASK)
#define SELECTOR_TABLE(_sel)     ((((Selector)(_sel)) & SELECTOR_TI_MASK) >> SELECTOR_TI_SHIFT)
#define SELECTOR_INDEX(_sel)     (((Selector)(_sel)) >> SELECTOR_INDEX_SHIFT)
#define SELECTOR_CLEAR_RPL(_sel) (((Selector)(_sel)) & ~SELECTOR_RPL_MASK)
#define NULL_SELECTOR(_sel)      (!SELECTOR_CLEAR_RPL(_sel))

/*
 *   tasks
 */

typedef
#include "vmware_pack_begin.h"
struct Task64 {
   uint32     reserved0;
   uint64     rsp[3];   // Stacks for CPL 0-2.
   uint64     ist[8];   // ist[0] is reserved.
   uint64     reserved1;
   uint16     reserved2;
   uint16     IOMapBase;
}
#include "vmware_pack_end.h"
Task64;


typedef
#include "vmware_pack_begin.h"
struct Task {

   uint16     prevTask,  __prevTasku;
   uint32     esp0;
   uint16     ss0,  __ss0u;
   uint32     esp1;
   uint16     ss1,  __ss1u;
   uint32     esp2;
   uint16     ss2,  __ss2u;
   uint32     cr3;
   uint32     eip;
   uint32     eflags;
   uint32     eax;
   uint32     ecx;
   uint32     edx;
   uint32     ebx;
   uint32     esp;
   uint32     ebp;
   uint32     esi;
   uint32     edi;
   uint16     es,  __esu;
   uint16     cs,  __csu;
   uint16     ss,  __ssu;
   uint16     ds,  __dsu;
   uint16     fs,  __fsu;
   uint16     gs,  __gsu;
   uint16     ldt,  __ldtu;
   uint16     trap;
   uint16     IOMapBase;
}
#include "vmware_pack_end.h"
Task;

typedef
#include "vmware_pack_begin.h"
struct {
   uint16     prevTask;
   uint16     sp0;  // static.  Unmarked fields are dynamic
   uint16     ss0;  // static
   uint16     sp1;  // static
   uint16     ss1;  // static
   uint16     sp2;  // static
   uint16     ss2;  // static 
   uint16     ip;
   uint16     flags;
   uint16     ax;
   uint16     cx;
   uint16     dx;
   uint16     bx;
   uint16     sp;
   uint16     bp;
   uint16     si;
   uint16     di;
   uint16     es;
   uint16     cs;
   uint16     ss;
   uint16     ds;
   uint16     ldt;  // static
}
#include "vmware_pack_end.h"
Task16;


/*
 *   far pointers
 */

typedef
#include "vmware_pack_begin.h"
struct {
   uint32 va;
   Selector seg;
}
#include "vmware_pack_end.h"
FarPtr;

typedef
#include "vmware_pack_begin.h"
struct FarPtr16 {
   uint16   offset;
   uint16   selector;
}
#include "vmware_pack_end.h"
FarPtr16;

typedef
#include "vmware_pack_begin.h"
struct FarPtr32 {
   uint32   offset;
   uint16   selector;
}
#include "vmware_pack_end.h"
FarPtr32;

typedef
#include "vmware_pack_begin.h"
struct FarPtr64 {
   uint64   offset;
   uint16   selector;
}
#include "vmware_pack_end.h"
FarPtr64;

/*
 * General-purpose registers and segment registers 
 *
 * REG_NULL indicates the absence of a register in certain APIs. 
 * REG_ZERO denotes a special register that always contains zero.
 * In other words: 
 *    - VREG[REG_NULL] is an error
 *    - VREG[REG_ZERO] evaluates to zero.
 */
typedef enum RegisterName {
   REG_NULL = -1, // -1 so that NUM_REGS32 doesn't include REG_NULL

   REG_RAX, REG_EAX = REG_RAX,
   REG_RCX, REG_ECX = REG_RCX,
   REG_RDX, REG_EDX = REG_RDX,
   REG_RBX, REG_EBX = REG_RBX,
   REG_RSP, REG_ESP = REG_RSP,
   REG_RBP, REG_EBP = REG_RBP,
   REG_RSI, REG_ESI = REG_RSI,
   REG_RDI, REG_EDI = REG_RDI,
   REG_R8, REG_R9, REG_R10, REG_R11, REG_R12, REG_R13, REG_R14, REG_R15,

   REG_ZERO = 16,  /* Not a real register, not counted in "NUM_REGS". */
   REG_IMM  = 17,  /* Not a real register, not counted in "NUM_REGS". */
   NUM_REGS = 16,
   NUM_REGS32 = 8
} RegisterName;


/* Return TRUE iff "regName" is a valid register number. */
static INLINE Bool
x86_IsGoodRegister(RegisterName regName)
{
#ifdef VMM32
   return (REG_NULL < regName && regName < NUM_REGS32);
#else
   return (REG_NULL < regName && regName < NUM_REGS);
#endif
}

typedef enum RegisterName8 {
   REG8_NULL = -1,
   REG8_AL,
   REG8_CL,
   REG8_DL,
   REG8_BL,
   REG8_AH,
   REG8_CH,
   REG8_DH,
   REG8_BH,
   NUM_REG8S
} RegisterName8;

/* Return TRUE iff "regName8" is a valid 8 bit register number. */
static INLINE Bool 
x86_IsGoodRegister8(RegisterName8 regName8) 
{ 
   return (REG8_NULL < regName8 && regName8 < NUM_REG8S); 
}

/* Mapping: EAX -> AL, ECX -> CL, EDX -> DL, EBX -> BL. */
static INLINE RegisterName8
x86_RegToReg8L(RegisterName reg)
{
   ASSERT(reg == REG_EAX || reg == REG_ECX ||
          reg == REG_EDX || reg == REG_EBX);

   return (RegisterName8)reg;
}

/* Mapping: EAX -> AH, ECX -> CH, EDX -> DH, EBX -> BH. */
static INLINE RegisterName8
x86_RegToReg8H(RegisterName reg)
{
   ASSERT(reg == REG_EAX || reg == REG_ECX ||
          reg == REG_EDX || reg == REG_EBX);

   return (RegisterName8)(reg + 4);
}

typedef enum SegmentName {
   SEG_NULL = -1, // -1 so that NUM_SEGS doesn't include SEG_NULL
   SEG_ES,
   SEG_CS,
   SEG_SS,
   SEG_DS,
   SEG_FS,
   SEG_GS,
   SEG_LDTR,
   SEG_TR,
   NUM_SEGS
} SegmentName;

#define NUM_NORMAL_SEGS  6

static INLINE Bool 
x86_IsGoodSegmentName(SegmentName segName)
{ 
   return SEG_NULL < segName && segName < NUM_SEGS; 
}


typedef enum ExcSrc {
   EXCSRC_INVALID       = 0,
   EXCSRC_FAULT         = 1,
   EXCSRC_IRQ           = 2,
   EXCSRC_INTN          = 3
} ExcSrc;

typedef enum ExcClass {
   EXCCLASS_BENIGN       = 1,
   EXCCLASS_CONTRIBUTORY = 2,
   EXCCLASS_PAGEFAULT    = 3
} ExcClass;


/*
 * X86-defined stack layouts for interrupts, exceptions, irets, calls, etc.
 */

/*
 * Layout of the 64-bit stack frame on exception.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame64 {
   uint64       rip;
   uint16       cs, __sel[3];
   uint64       rflags;
   uint64       rsp;
   uint16       ss, __ssel[3];
}
#include "vmware_pack_end.h"
x86ExcFrame64;

typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame64WithErrorCode {
   uint32       errorCode, __errorCode;
   uint64       rip;
   uint16       cs, __sel[3];
   uint64       rflags;
   uint64       rsp;
   uint16       ss, __ssel[3];
}
#include "vmware_pack_end.h"
x86ExcFrame64WithErrorCode;

/*
 * Layout of the 32-bit stack frame on exception.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame32 {
   uint32         eip;
   union {
      struct {
         uint16   sel, __sel;
      }           cs16;
      uint32      cs32;
   } u;
   uint32         eflags;
}
#include "vmware_pack_end.h"
x86ExcFrame32;

/*
 * Layout of the 32-bit stack frame with ss:esp and no error code.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame32WithStack {
   uint32      eip;
   uint16      cs, __csu;
   uint32      eflags;
   uint32      esp;
   uint16      ss, __ssu;
}
#include "vmware_pack_end.h"
x86ExcFrame32WithStack;

/*
 * Layout of the 32-bit stack frame on inter-level transfer.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame32IL {
   uint32      errorCode;
   uint32      eip;
   uint16      cs, __csu;
   uint32      eflags;
   uint32      esp;
   uint16      ss, __ssu;
}
#include "vmware_pack_end.h"
x86ExcFrame32IL;


/*
 * Layout of the 16-bit stack frame on exception.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame16 {
   uint16   eip;
   uint16   cs;
   uint16   eflags;
}
#include "vmware_pack_end.h"
x86ExcFrame16;

/*
 * Layout of the 16-bit stack frame which incudes ss:sp.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrame16WithStack {
   uint16   ip;
   uint16   cs;
   uint16   flags;
   uint16   sp;
   uint16   ss;
}
#include "vmware_pack_end.h"
x86ExcFrame16WithStack;

/*
 * Layout of the 32-bit stack frame on exception
 * from V8086 mode. It is also a superset
 * of inter-level exception stack frame, which
 * in turn is superset of intra-level exception
 * stack frame.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrameV8086 {
   uint32         eip;
   union {
      struct {
         uint16   sel, __sel;
      }           cs16;
      uint32      cs32;
   } u;
   uint32         eflags;
   uint32         esp;
   uint16         ss, __ss;
   uint16         es, __es;
   uint16         ds, __ds;
   uint16         fs, __fs;
   uint16         gs, __gs;
}
#include "vmware_pack_end.h"
x86ExcFrameV8086;

/*
 * Layout of the 32-bit stack frame on exception
 * from V8086 mode with errorCode. It is
 * superset of SegmentExcFrameV8086.
 */
typedef
#include "vmware_pack_begin.h"
struct x86ExcFrameV8086WithErrorCode {
   uint32         errorCode;
   uint32         eip;
   union {
      struct {
         uint16   sel, __sel;
      }           cs16;
      uint32      cs32;
   } u;
   uint32         eflags;
   uint32         esp;
   uint16         ss, __ss;
   uint16         es, __es;
   uint16         ds, __ds;
   uint16         fs, __fs;
   uint16         gs, __gs;
}
#include "vmware_pack_end.h"
x86ExcFrameV8086WithErrorCode;

/*
 * Layout of the stack on a 32 bit far call.
 */
typedef
#include "vmware_pack_begin.h"
struct x86CallStack32 {
   uint32   eip;
   uint16   cs, __cs;
}
#include "vmware_pack_end.h"
x86CallStack32;

/*
 * Layout of the stack on a 16 bit far call.
 */
typedef
#include "vmware_pack_begin.h"
struct x86CallStack16 {
   uint16   ip;
   uint16   cs;
}
#include "vmware_pack_end.h"
x86CallStack16;

/*
 * Layout of the stack on a 32 bit far call.
 */
typedef
#include "vmware_pack_begin.h"
struct x86CallGateStack32 {
   uint32   eip;
   uint16   cs, __cs;
   uint32   esp;
   uint16   ss, __ss;
}
#include "vmware_pack_end.h"
x86CallGateStack32;

/*
 * Layout of the stack on a 16 bit far call.
 */
typedef
#include "vmware_pack_begin.h"
struct x86CallGateStack16 {
   uint16   ip;
   uint16   cs;
   uint16   sp;
   uint16   ss;
}
#include "vmware_pack_end.h"
x86CallGateStack16;


/*
 *   page tables
 */

#define PTE_P           0x00000001
#define PTE_RW          0x00000002
#define PTE_US          0x00000004
#define PTE_PWT         0x00000008
#define PTE_PCD         0x00000010 // careful: MMU_PTE_EX has same value.
#define PTE_A           0x00000020
#define PTE_D           0x00000040
#define PTE_PS          0x00000080
#define PTE_G           0x00000100

#define PTE_PROT_FLAGS  (PTE_P|PTE_RW|PTE_US)
#define PTE_FLAGS       (PTE_PROT_FLAGS|PTE_G)

#define PTE_PAGE_TABLE  (PTE_P|PTE_RW|PTE_US|PTE_A|PTE_D) 

/*
 * This is usable for 4M or 2M pde's (both PAE and non-PAE). Use the PTE
 * macros for large pages.
 */ 
#define PDE_PAGE_TABLE  (PTE_P | PTE_RW | PTE_US | PTE_A) 

#define PTE_PRESENT(_pte)   (((_pte) & PTE_P) != 0)
#define PTE_WRITEABLE(_pte) (((_pte) & PTE_RW) != 0)
#define PTE_ACCESS(_pte)    (((_pte) & PTE_A) != 0)
#define PTE_DIRTY(_pte)     (((_pte) & PTE_D) != 0)
#define PTE_USER(_pte)      (((_pte) & PTE_US) != 0)
#define PTE_GLOBAL(_pte)    (((_pte) & PTE_G) != 0)
#define PTE_LARGEPAGE(_pte) (((_pte) & PTE_PS) != 0)
#define PTE_UNCACHED(_pte)  (((_pte) & PTE_PCD) != 0)

#define PTE_AVAIL_MASK       0xe00
#define PTE_AVAIL_SHIFT      9
#define PTE_PFN_MASK         0xfffff000
#define PAE_PTE_PFN_MASK     CONST64U(0xffffff000)
#define LM_PTE_PFN_MASK      CONST64U(0xffffffffff000)
#define PTE_PFN_SHIFT        12
#define PAE_PTE_PFN_SHIFT    12
#define LM_PTE_PFN_SHIFT     12
#define PTE_2_PFN(_pte)      (((_pte) & PTE_PFN_MASK) >> PTE_PFN_SHIFT)
#define PAE_PTE_2_PFN(_pte)  (((_pte) & PAE_PTE_PFN_MASK) >> PAE_PTE_PFN_SHIFT)
#define LM_PTE_2_PFN(_pte)   (((_pte) & LM_PTE_PFN_MASK) >> LM_PTE_PFN_SHIFT)
#define PTE_AVAIL(_pte)      (((_pte) & PTE_AVAIL_MASK) >> PTE_AVAIL_SHIFT)

#define PDE_PFN_MASK         0xffc00000
#define PAE_PDE_LARGE_PFN_MASK     0xfffe00000LL
#define LM_PDE_PFN_MASK      0xfffffffe00000LL
#define PDE_PFN_SHIFT        PTE_PFN_SHIFT

/* Use these only for 4M or 2M pde's, not the 4K ones. */

#define PDPTE_PFN_MASK       0xffffff000LL
#define PDPTE_PFN_SHIFT      12
#define PDPTE_2_PFN(_p)      (((_p) & PDPTE_PFN_MASK) >> PDPTE_PFN_SHIFT)
#define PDPTE_INVALID        0x6 /* Reserved bits */

#define NUM_PAE_PDIRS        4
#define PDES_PER_PDIR        (PAGE_SIZE/sizeof(VM_PDE))
#define PTES_PER_PDIR        PDES_PER_PDIR
#define PAE_PDES_PER_PDIR    (PAGE_SIZE/sizeof(VM_PAE_PDE))
#define PTES_PER_PGTBL       (PAGE_SIZE/sizeof(VM_PTE))
#define PAE_PTES_PER_PGTBL   (PAGE_SIZE/sizeof(VM_PAE_PTE))
#define L1ES_PER_PGTBL       (PAGE_SIZE/sizeof(VM_L1E))

#define PGOFF_MASK            0x3ff
#define PAE_PGOFF_MASK        0x1ff
#define PDOFF_MASK            0x3ff
#define PAE_PDOFF_MASK        0x1ff
#define PDPTOFF_MASK          0x3
#define PAE_RANGE_MASK        0x7ff
#define LM_L4OFF_MASK         LM_L1OFF_MASK
#define LM_L3OFF_MASK         LM_L1OFF_MASK
#define LM_L2OFF_MASK         LM_L1OFF_MASK
#define LM_L1OFF_MASK         0x1ff

#define PDE_4M_RSVD_MASK      (1 << 21)
#define WIDE_PDE_2M_RSVD_MASK 0x1fe000

#define PFN_2_PDPTOFF(_a)     (((_a) >> 18) & PDPTOFF_MASK)
#define PFN_2_PDOFF(_a)       (((_a) >> 10) & PDOFF_MASK)
#define PAE_PFN_2_PDOFF(_a)   (((_a) >> 9) & PAE_PDOFF_MASK)
#define PFN_2_PGOFF(_a)       ((_a) & PGOFF_MASK)
#define PAE_PFN_2_PGOFF(_a)   ((_a) & PAE_PGOFF_MASK)
#define PFN_2_RANGE(_pfn)     PFN_2_PDOFF(_pfn)
#define PAE_PFN_2_RANGE(_pfn) (((_pfn) >> 9) & PAE_RANGE_MASK)

#define RANGE_2_LA(_r)        ((_r) << 22)
#define PAE_RANGE_2_LA(_r)    ((_r) << 21)
#define RANGE_2_PDPTOFF(_r)   (((_r) >> 9) & 0x3)
#define RANGE_2_PDOFF(_r)     ((_r))
#define PAE_RANGE_2_PDOFF(_r) ((_r) & 0x1ff)

#define LA_2_PDPTOFF(_a)      (((_a) >> 30) & PDPTOFF_MASK)
#define LA_2_PDOFF(_a)        (((_a) >> 22) & PDOFF_MASK)
#define PAE_LA_2_PDOFF(_a)    (((_a) >> 21) & PAE_PDOFF_MASK)
#define LA_2_PGOFF(_a)        (PFN_2_PGOFF(VA_2_VPN(_a)))
#define PAE_LA_2_PGOFF(_a)    (PAE_PFN_2_PGOFF(VA_2_VPN(_a)))
#define LA_2_RANGE(_a)        LA_2_PDOFF((_a))
#define PAE_LA_2_RANGE(_a)    (((_a) >> 21) & PAE_RANGE_MASK)
#define PAE_LA_2_PDPTOFF(_a)  (((_a) >> 30) & 0x3)

#define LM_LA_2_L4OFF(_a)    (((_a) >> 39) & LM_L4OFF_MASK)
#define LM_LA_2_L3OFF(_a)    (((_a) >> 30) & LM_L3OFF_MASK)
#define LM_LA_2_L2OFF(_a)    (((_a) >> 21) & LM_L2OFF_MASK)
#define LM_LA_2_L1OFF(_a)    (((_a) >> 12) & LM_L1OFF_MASK)

#define LM_LPN_2_L4OFF(_a)   (((_a) >> (39 - LM_PTE_PFN_SHIFT)) & LM_L4OFF_MASK)
#define LM_LPN_2_L3OFF(_a)   (((_a) >> (30 - LM_PTE_PFN_SHIFT)) & LM_L3OFF_MASK)
#define LM_LPN_2_L2OFF(_a)   (((_a) >> (21 - LM_PTE_PFN_SHIFT)) & LM_L2OFF_MASK)
#define LM_LPN_2_L1OFF(_a)   (((_a) >> (12 - LM_PTE_PFN_SHIFT)) & LM_L1OFF_MASK)

/* This following macro only work for flat model were VA==LA */
#define VA_2_PDOFF(_a)        ((_a) >> 22)
#define VA_2_PGOFF(_a)        (PFN_2_PGOFF(VA_2_VPN(_a)))

#define PTE_2_PA(_a)          (PPN_2_PA(PTE_2_PFN(_a)))
#define PAE_PTE_2_PA(_a)      (PPN_2_PA(PAE_PTE_2_PFN(_a)))

#define PDE_FLAGS(_pde)       ((_pde) & 0xfff)  
#define PAE_PDE_FLAGS(_pde)   ((_pde) & 0xfff)  
/*
 *   exception error codes
 */

#define EXC_DE            0
#define EXC_DB            1
#define EXC_NMI           2
#define EXC_BP            3
#define EXC_OF            4
#define EXC_BR            5
#define EXC_UD            6
#define EXC_NM            7
#define EXC_DF            8
#define EXC_TS           10
#define EXC_NP           11
#define EXC_SS           12
#define EXC_GP           13
#define EXC_PF           14
#define EXC_MF           16
#define EXC_AC           17
#define EXC_MC           18
#define EXC_XF           19  // SIMD exception

/* 
 * Pseudo-exception vectors used internally. Negative numbers are
 * used for internal monitor conditions, such as no fault, or an
 * indication that the instruction should be re-run with simulation.
 * 
 * Some of these are used to force transfer from direct exec to BT.
 * EXC_EIP_CHG is used to indicate that %eip has changed in a nonstandard
 * manner. Examples include a <rep> prefixed instruction that didn't exhaust
 * the rep count, control flow transfer, and INVLPG that may have changed
 * the mapping of the next instruction.
 * EXC_CANNOT_INLINE is used by the inliner at translation time only.
 */
#define EXC_NONE                -1
#define EXC_TRANSLATOR_OVERFLOW -2
#define EXC_BT_SIMULATE         -3
#define EXC_BT_RELOCATE         -4
#define EXC_EIP_CHG             -5
#define EXC_CANNOT_INLINE       -6
#define EXC_DEFERRED_PF         -7
#define EXC_LAST_PSEUDO_FAULT   EXC_DEFERRED_PF
#define EXC_SMI                 -8    /* Not a fault. */

/* Error code flags */
#define PF_P            0x1
#define PF_RW           0x2
#define PF_US           0x4
#define PF_RSVD         0x8
#define PF_ID           0x10

#define ERRCODE_EXT     0x1
#define ERRCODE_IDT     0x2

/*
 *  instruction decoding
 */

#define MODRM_MOD(_m)  (((_m) >> 6) & 0x3)
#define MODRM_REG(_m)  (((_m) >> 3) & 0x7)
#define MODRM_RM(_m)   (((_m) >> 0) & 0x7)

/*
 * Debug registers
 */

#define DEBUG_STATUS_B0   (1<<0)
#define DEBUG_STATUS_B1   (1<<1)
#define DEBUG_STATUS_B2   (1<<2)
#define DEBUG_STATUS_B3   (1<<3)
#define DEBUG_STATUS_DB   (1<<13)
#define DEBUG_STATUS_BS   (1<<14)
#define DEBUG_STATUS_BT   (1<<15)

typedef struct DebugControlRegister {

   int l0:1;
   int g0:1;
   int l1:1;
   int g1:1;
   int l2:1;
   int g2:1;
   int l3:1;
   int g3:1;
   
   int le:1;
   int ge:1;
   int oo1:3;
   
   int gd:1;
   int oo:2;
   
   int rw0:2;
   int len0:2;
   int rw1:2;
   int len1:2;
   int rw2:2;
   int len2:2;
   int rw3:2;
   int len3:2;
   
} DebugControlRegister;

/*
 * System Call Information for each VCPU.
 */
typedef struct SystemCallRegisters {
   Selector     sysenterCS;
   uint16       _pad[3];
   uint64       sysenterRIP;
   uint64       sysenterRSP;
   uint64       star;
   uint64       lstar;
   uint64       cstar;
   uint64       sfmask;
} SystemCallRegisters;

typedef struct SystemCallState {
   SystemCallRegisters scs32;   // Cannot use 'vmm32' and 'vmm64' as they
   SystemCallRegisters scs64;   // are defined as constants '0' or '1'.
   Bool                sce;
   uint8               _pad0[3];
   uint32              _pad1;
} SystemCallState;

/*
 * Macros for LM page table used by x86_64.
 */
#define LM_OFF_MASK       0x1ff

#define PFN_2_L4OFF(_a)   (((_a) >> 27) & LM_OFF_MASK)
#define PFN_2_L3OFF(_a)   (((_a) >> 18) & LM_OFF_MASK)
#define PFN_2_L2OFF(_a)   (((_a) >> 9) & LM_OFF_MASK)
#define PFN_2_L1OFF(_a)    ((_a) & LM_OFF_MASK)

#define LM_PTE_SHIFT          12
#define LM_A1_SHIFT           52
#define LM_A2_SHIFT           9

#define LM_A1_MASK            CONST64(0x7ff)
#define LM_A2_MASK            CONST64(0x7)
#define LM_FLAGS_MASK         CONST64(0x80000000000001ff)
#define LM_PDE_MASK           CONST64(0xffffffffffffff7f)
#define LM_CR3_FLAGS_MASK     CONST64(0x18)
#define PTE_NX                CONST64(0x8000000000000000)

#define LM_MAKE_CR3(_mpfn, _flags) \
                   (((uint64)(_mpfn) << LM_PTE_SHIFT) | \
                   ((_flags) & LM_CR3_FLAGS_MASK))

#define LM_MAKE_PTE(_mpfn, _a1, _a2, _flags) \
                   (((uint64)(_mpfn) << LM_PTE_SHIFT) | \
                   ((((uint64)_a1) & LM_A1_MASK) << LM_A1_SHIFT)\
                   | (((_a2) & LM_A2_MASK) << LM_A2_SHIFT) \
                   | ((uint64)(_flags) & LM_FLAGS_MASK))

#define LM_MAKE_L4E LM_MAKE_PTE
#define LM_MAKE_L3E LM_MAKE_PTE
#define LM_MAKE_L2E(_mpfn, _a1, _a2, _flags) \
                   (LM_MAKE_PTE(_mpfn, _a1, _a2, _flags) & \
                   LM_PDE_MASK)

                 
#define MAKE_SELECTOR_UNCHECKED(_index, _ti, _RPL)  \
        ( ((_index) << SELECTOR_INDEX_SHIFT)    \
        | ((_ti   ) << SELECTOR_TI_SHIFT)       \
        | ((_RPL  ) << SELECTOR_RPL_SHIFT) )

static INLINE Selector
MAKE_SELECTOR(unsigned index, unsigned ti, unsigned rpl)
{
   ASSERT(index <= (SELECTOR_INDEX_MASK >> SELECTOR_INDEX_SHIFT) &&
          ti    <= (SELECTOR_TI_MASK >> SELECTOR_TI_SHIFT)       &&
          rpl   <= (SELECTOR_RPL_MASK >> SELECTOR_RPL_SHIFT));
   return MAKE_SELECTOR_UNCHECKED(index, ti, rpl);
}

#define MAKE_PTE(_mpfn, _avail, _flags) \
                        (((_mpfn) << PTE_PFN_SHIFT) \
                         | (((_avail) << PTE_AVAIL_SHIFT) & PTE_AVAIL_MASK) \
                         | (_flags))

#define PAE_MAKE_PTE(_mpfn, _avail, _flags) \
     ((((VM_PAE_PTE) (_mpfn)) << PAE_PTE_PFN_SHIFT) \
      | (((_avail) << PTE_AVAIL_SHIFT) & PTE_AVAIL_MASK) \
      | (_flags))

#define MAKE_PDE(_mpfn, _avail, _flags) MAKE_PTE(_mpfn, _avail, _flags)
#define PAE_MAKE_PDE(_mpfn, _avail, _flags) PAE_MAKE_PTE(_mpfn, _avail, _flags)
#define MAKE_PDPTE(_mpfn, _avail, _flags) PAE_MAKE_PTE(_mpfn, _avail, _flags)
#define PDPTE_FLAGS (PTE_P)
     


/*
 * SMM State Save Map 
 *
 *   We standardize on a lightly modified version of the opteron SMM
 *   structure. The modification is intended to ease work with AMD's SVM
 *   hardware.
 *   
 *   Note that this is different from the state save map exhibited on
 *   real, 32-bit hardware. That's ok. SMM code is under the control of
 *   the BIOS, which we can bend to our devious will.
 */

typedef enum {
   SMMVERSION_DEFAULT = 0,
   SMMVERSION_LEGACY
} SMMVersion;

/*
 * SMMSegState - Store the "hidden" state of a segment descriptor in 
 * the SMM save state map. NOTE: Format is somewhat defined by the 
 * Intel hardware.
 */
typedef struct SMMSegState {
   uint16 sel;
   uint16 attrib;
   uint32 limit;   /* Limit from descriptor */
   uint64 base;
} SMMSegState;

/*
 * SMM32SegState - legacy SMM format. Needs to stick around so we can
 * restore old checkpoints. Barfola.
 */
typedef struct SMM32SegState {
   uint32 limit;
   uint32 base;
   uint32 rights;
} SMM32SegState;

/*
 * Format the the SMM save state map.
 * NOTE: This too is defined by the Intel hardware.
 */

typedef struct SMMStateSaveMap {
   SMMSegState hiddenSegPart[6];
   SMMSegState gdtr;
   SMMSegState ldtr;
   SMMSegState idtr;
   SMMSegState hiddenTRpart;
   union {
      struct {
         uint32 selState;       // We steal a enough of this reserved
                                // space to stash away the selector state. 
         uint8  intrLinesMask;  // Save IntrLine_DB for RSM.
      } s;
      uint8  reserved0[40];
   } u;
   Bool ioInstRestart;
   Bool haltRestart;
   uint8 svmCpl;             // reserved in SMM; cpl for SVM
   uint8 reserved1[5];
   uint64 efer;
   uint8 reserved2[36];
   uint32 revId;             // reserved in SVM
   uint32 smbase;            // reserved in SVM
   uint8 reserved3[68];
   Reg64 cr4reg;
   Reg64 cr3reg;
   Reg64 cr0reg;
   Reg64 dr7reg;
   Reg64 dr6reg;
   Reg64 rflags;
   Reg64 rip;
   /* This array is indexed in reverse. REG_RAX is the 15'th element, and
    * REG_R15 is the the 0'th. To get at reg foo, use REG_R15 - REG_FOO
    */
   Reg64 gpRegs[NUM_REGS];
} SMMStateSaveMap;


/*
 * SMM32StateSaveMap - legacy SMM format. See "Barfola" above.
 */
typedef struct SMM32StateSaveMap {
   union {
      uint32 selState;       // We steal a enough of this (hopefully) unused
                             // space to stash away the selector state. 
      uint8  reserved0[248];
   } u;
   uint32 smbase;
   uint32 revId; 
   uint16 ioInstRestart;
   uint16 haltRestart;
   uint8  reserved01[12];
   uint32 ioInstEIP; 
   uint8  reserved1[18];
   uint32 cr4reg;
   uint8  reserved2[4];
   SMM32SegState hiddenSegPart[7];
   SMM32SegState gdtr;
   SMM32SegState idtr;
   SMM32SegState hiddenTRpart;
   uint32    segRegs[8];
   Reg32     dr7reg;
   Reg32     dr6reg;
   Reg32     gpRegs[8];
   Reg32     eip;
   x86_FLAGS eflags;
   Reg32     cr3reg;
   Reg32     cr0reg;
} SMM32StateSaveMap;

#define SMM_SAVEMAP_SIZE   512
#define SMRAM_DEFAULT_BASE 0x30000
#define SMRAM_ENTRY_POINT 0x8000
#define SMRAM_SAVE_STATE_OFFSET 0xfe00

#define SMM_RIGHTS_BITS   0x00f0ff00

/*
 * FPU FSAVE/FXSAVE area format. 
 *
 * FSAVEState  - Memory format of the FSAVE/FNSAVE instruction. 108 bytes
 * FXSAVEState - Memory format of the FXSAVE instruction. 512 bytes
 *               Note: FXSAVEState must be 128bit aligned. 
 *
 * Taken from linux headers
 */

typedef
#include "vmware_pack_begin.h"
struct FSAVEState {
   uint32   cwd;
   uint32   swd;
   uint32   twd;
   uint32   fip;
   uint32   fcs;
   uint32   foo;
   uint32   fos;
   uint32   st_space[20];   /* 8*10 bytes for each FP-reg = 80 bytes */
}
#include "vmware_pack_end.h"
FSAVEState;

typedef
#include "vmware_pack_begin.h"
struct FXSAVEState {
   uint16   fxcwd;
   uint16   fxswd;
   uint16   fxtwd;
   uint16   fxfopcode;
   uint32   fxfip;
   uint16   fxfcs;
   uint16   __reserved_00;
   uint32   fxfoo;
   uint16   fxfos;
   uint16   __reserved_01;
   uint32   mxcsr;
   uint32   __reserved_02;
   uint32   st_space[32];     /* 8*16 bytes for each FP/MMX-reg = 128 bytes */
   uint32   xmm_space[32];    /* 8*16 bytes for each XMM-reg = 128 bytes */
   uint32   __reserved_03 [14*4]; /* 14 16byte lines for remainder */
}
#include "vmware_pack_end.h"
FXSAVEState;

typedef
#include "vmware_pack_begin.h"
struct FXSAVEState64 {
   uint16   cwd;
   uint16   swd;
   uint16   twd;
   uint16   fop;
   uint64   rip;
   uint64   rdp;
   uint32   mxcsr;
   uint32   mxcsr_mask;
   uint32   st_space[32];	/* 8*16 bytes for each FP-reg = 128 bytes */
   uint32   xmm_space[64];	/* 16*16 bytes for each XMM-reg = 256 bytes */
   uint32   padding[24];
}
#include "vmware_pack_end.h"
FXSAVEState64;

/*
 *----------------------------------------------------------------------
 *
 * x86_ErrorCode2Offset --
 *
 *     Transforms an error code into a byte offset by masking off
 *     the top 16 bits and the bottom three bits.  Note that
 *     some exceptions don't push any error codes, some always push
 *     zero, and the page fault exception uses its own unique error
 *     code format that does not make reference to a selector/offset.
 *
 * Results:
 *     Returns the descriptor table offset referenced by the error code.
 *
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

static INLINE int
x86_ErrorCode2Offset(uint32 errorCode)
{
   return errorCode & ~0xffff0007;
}


/*
 *----------------------------------------------------------------------
 *
 * x86_ErrorCode2RPL0Selector --
 *
 *     Returns the RPL0 version of the selector referenced by an error
 *     code.  Note that it's impossible to reconstruct the RPL of the
 *     original selector that caused the error code to be generated,
 *     because the RPL bits of the selector are used as flag bits
 *     in the error code.  It is not legal to call this function on
 *     an error code with the IDT bit set, since such error codes
 *     refer to IDT offsets, not GDT/LDT selectors.
 *
 * Results:
 *     Returns the descriptor table offset referenced by the error code.
 *
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Selector
x86_ErrorCode2RPL0Selector(uint32 errorCode)
{
   ASSERT((errorCode & ERRCODE_IDT) == 0);
   return ((Selector) errorCode & ~0xffff0003);
}


/*
 *----------------------------------------------------------------------
 *
 * x86_ExcUsesErrorCode --
 *
 *     Determines whether the specified fault number pushes an error code
 *     on the stack when the vector is invoked.
 *
 * Results:
 *     TRUE if the exception number uses an error code, and FALSE otherwise.
 *
 * Side effects:
 *     None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Bool
x86_ExcUsesErrorCode(int exc)
{
   const uint32 bits = (1 << EXC_DF) | (1 << EXC_TS) | (1 << EXC_NP) |
                       (1 << EXC_SS) | (1 << EXC_GP) | (1 << EXC_PF) |
                       (1 << EXC_AC);
   return (bits & (1 << exc)) != 0;
}


/*
 *----------------------------------------------------------------------
 *
 * x86_ExcUsesSelectorErrorCode -- 
 *
 *      Determines if a fault number is one that can push an error code
 *      that refers to a selector.  Note that some faults will sometimes
 *      explicitly push an error code of 0 for some conditions while
 *      pushing "normal" error codes for other fault conditions.
 *
 * Results:
 *      TRUE if this fault number may push an error code that
 *      refers to a segment and FALSE otherwise.
 *
 * Side effects:
 *      None.
 *
 *----------------------------------------------------------------------
 */

static INLINE Bool
x86_ExcUsesSelectorErrorCode(int exc)
{
   return (exc >= EXC_TS && exc <= EXC_GP);
}


/* 
 *---------------------------------------------------------------------------
 * IsGoodMPN --
 *    Return TRUE if "mpn" looks plausible. We could make this
 *    test stricter in the non-PAE case. We could even take into
 *    account the amount of memory on the host.
 *---------------------------------------------------------------------------
 */

static INLINE Bool
IsGoodMPN(MPN mpn)
{
   return mpn <= MAX_MPN;
}


/*
 * x86-64 architecture requires implementations supporting less than
 * full 64-bit VAs to ensure that all virtual addresses are in canonical
 * form. An address is in canonical form if the address bits from the
 * most significant implemented bit up to bit 63 are all ones or all
 * zeros. If this is not the case, the processor generates #GP/#SS. Our
 * VCPU implements 48 bits of virtual address space.
 */

#define VA64_IMPL_BITS             48
#define VA64_CANONICAL_MASK        ~((CONST64U(1) << (VA64_IMPL_BITS - 1)) - 1)
#define VA64_CANONICAL_HOLE_START   (CONST64U(1) << (VA64_IMPL_BITS - 1))
#define VA64_CANONICAL_HOLE_LEN  VA64_CANONICAL_MASK - VA64_CANONICAL_HOLE_START

static INLINE Bool
x86_IsCanonical(VA64 va)
{
   return (va & VA64_CANONICAL_MASK) == 0 || 
          (va & VA64_CANONICAL_MASK) == VA64_CANONICAL_MASK;
}

static INLINE Bool
x86_IsCanonicalRange(VA64 va, unsigned size)
{
   /*
    * The check is simple as long as the size is less
    * than the number of implemented bits.
    *
    * The only case we don't handle is one where the VA goes from a
    * high canonical address and wraps to a non-canonical address
    * (e.g. 0x00008000_00000000) or higher.  Our test would falsely
    * consider this canonical.
    */
   ASSERT(sizeof(size) * 8 < VA64_IMPL_BITS);

   /*
    * VA64_CANONICAL_MASK is the lowest canonical address with the
    * upper bits all set.
    *
    * VA64_CANONICAL_HOLE_START is one higher than the highest valid
    * canonical address with the upper bits all cleared.  Note that we
    * access up to (va + size - 1), not (va + size), so <= is correct.
    */
   return va >= VA64_CANONICAL_MASK ||
          va + size <= VA64_CANONICAL_HOLE_START;
}

#endif /* _X86_H_ */
