package org.herac.tuxguitar.io.importer;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.sound.midi.MetaMessage;
import javax.sound.midi.MidiEvent;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.Sequence;
import javax.sound.midi.ShortMessage;
import javax.sound.midi.Track;

import org.herac.tuxguitar.player.base.MidiControllers;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Measure;
import org.herac.tuxguitar.song.models.MeasureHeader;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.NoteEffect;
import org.herac.tuxguitar.song.models.RGBColor;
import org.herac.tuxguitar.song.models.Song;
import org.herac.tuxguitar.song.models.SongChannel;
import org.herac.tuxguitar.song.models.SongTrack;
import org.herac.tuxguitar.song.models.Tempo;
import org.herac.tuxguitar.song.models.TimeSignature;

public class MidiImporter implements SongImporter{

	private static final long MIN_START = 1000;
	
	private static final Duration MIN_DURATION = new Duration(Duration.SIXTY_FOURTH);
	
	private float divisionType;
	private int resolution;
	private boolean moveTick;
	private List headers;
	private List tracks;
	private List tempNotes;
	private List tempChannels;
	private List trackTunningHelpers;
	
	public MidiImporter(){		
	}
	
	public Song importSong(String fileName) throws IOException, Exception, Error{				
		Sequence sequence = MidiSystem.getSequence(new File(fileName));			
		initFields(sequence);
		
		Track[] tracks = sequence.getTracks();
		for(int i = 0; i < tracks.length; i++){				
			Track track = tracks[i];	
			int trackNumber = getNextTrackNumber();
			int events = track.size();
			for(int j = 0;j < events;j ++){
				MidiEvent event = track.get(j);		
				parseMessage(trackNumber,event.getTick(),event.getMessage().getMessage());
			}			
		}
		checkAll();
		
		return new SongAdjuster(new Song(this.tracks,this.headers)).adjustSong();
		//return new Song(this.tracks,this.headers);
	}
		
	private void initFields(Sequence sequence){
		this.divisionType = sequence.getDivisionType();
		this.resolution = sequence.getResolution();
		this.headers = new ArrayList();
		this.tracks = new ArrayList();
		this.tempNotes = new ArrayList();
		this.tempChannels = new ArrayList();
		this.trackTunningHelpers = new ArrayList();
	}
	
	private int getNextTrackNumber(){
		return (tracks.size() + 1);
	}
	
	private void parseMessage(int trackNumber,long tick,byte[] data){		
		int length = data.length;	
		tick = parseTick(tick + resolution);
		
		//NOTE ON
		if((((length > 0)?(data[0] & 0xFF):0) & 0xF0) == ShortMessage.NOTE_ON){
			parseNoteOn(trackNumber,tick,data);
		}
		//NOTE OFF		
		else if((((length > 0)?(data[0] & 0xFF):0) & 0xF0) == ShortMessage.NOTE_OFF){
			parseNoteOff(trackNumber,tick,data);
		}
		//PROGRAM CHANGE		
		else if((((length > 0)?(data[0] & 0xFF):0) & 0xF0) == ShortMessage.PROGRAM_CHANGE){
			parseProgramChange(data);
		}
		//CONTROL CHANGE		
		else if((((length > 0)?(data[0] & 0xFF):0) & 0xF0) == ShortMessage.CONTROL_CHANGE){
			parseControlChange(data);
		}				
		//TIME SIGNATURE		
		else if(((length > 0)?(data[0] & 0xFF):0) == MetaMessage.META && (data[1] == 0x58)){
			parseTimeSignature(tick,data);
		}		
		//TEMPO		
		else if(((length > 0)?(data[0] & 0xFF):0) == MetaMessage.META && (data[1] == 0x51)){
			parseTempo(tick,data);
		}				
	}

	private long parseTick(long tick){
		return Math.abs(Duration.QUARTER_TIME * tick / this.resolution);
	}
	
	private void parseNoteOn(int track,long tick,byte[] data){				
		int length = data.length;		
		int channel = (length > 0)?((data[0] & 0xFF) & 0x0F):0;
		int value = (length > 1)?(data[1] & 0xFF):0;
		int velocity = (length > 2)?(data[2] & 0xFF):0;
		if(velocity == 0){
			parseNoteOff(track,tick,data);
		}else if(value > 0){			
			makeTempNotesBefore(tick,track);		
			getTempChannel(channel).track = track;
			getTrackTunningHelper(track).checkValue(value);
			tempNotes.add(new TempNote(track,channel,value,tick));
		}
	}
	
	private void parseNoteOff(int track,long tick,byte[] data){		
		int length = data.length;
		
		int channel = (length > 0)?((data[0] & 0xFF) & 0x0F):0;
		int value = (length > 1)?(data[1] & 0xFF):0;
		int velocity = (length > 2)?(data[2] & 0xFF):0;
		
		makeNote(tick,track,channel,value);
	}	
	
	private void parseProgramChange(byte[] data){		
		int length = data.length;		
		int channel = (length > 0)?((data[0] & 0xFF) & 0x0F):-1;
		int instrument = (length > 1)?(data[1] & 0xFF):-1;
		if(channel != -1 && instrument != -1){
			getTempChannel(channel).instrument = instrument;
		}
	}		

	private void parseControlChange(byte[] data){				
		int length = data.length;		
		int channel = (length > 0)?((data[0] & 0xFF) & 0x0F):-1;
		int control = (length > 1)?(data[1] & 0xFF):-1;
		int value = (length > 2)?(data[2] & 0xFF):-1;
		if(channel != -1 && control != -1 && value != -1){
			if(control == MidiControllers.VOLUME){
				getTempChannel(channel).volume = value;
			}
			else if(control == MidiControllers.BALANCE){
				getTempChannel(channel).balance = value;
			}			
		}
	}		

	
	private void parseTimeSignature(long tick,byte[] data){
		TimeSignature timeSignature = new TimeSignature(data[3],new Duration(Duration.QUARTER));
		if (data[4] == 0) {
			timeSignature.getDenominator().setValue(Duration.WHOLE);			
        } else if (data[4] == 1) {
        	timeSignature.getDenominator().setValue(Duration.HALF);
        } else if (data[4] == 2) {
        	timeSignature.getDenominator().setValue(Duration.QUARTER);
        } else if (data[4] == 3) {
        	timeSignature.getDenominator().setValue(Duration.EIGHTH);
        } else if (data[4] == 4) {
        	timeSignature.getDenominator().setValue(Duration.SIXTEENTH);
        } else if (data[4] == 5) {
        	timeSignature.getDenominator().setValue(Duration.THIRTY_SECOND);
        }						 
		//timeSignature.getDenominator().setValue((96 / data[5]));		
		getHeader(tick).setTimeSignature(timeSignature);
	}		
	
	private void parseTempo(long tick,byte[] data){				
		Tempo tempo = Tempo.fromUSQ((data[5] & 0x00ff) | ((data[4] & 0x00ff) << 8) | ((data[3] & 0x00ff) << 16));		
		getHeader(tick).setTempo(tempo);
	}	
	
	private SongTrack getTrack(int number){		
		Iterator it = tracks.iterator();
		while(it.hasNext()){
			SongTrack track = (SongTrack)it.next();
			if(track.getNumber() == number){
				return track;
			}
		}		
		SongChannel channel = new SongChannel((short)-1,(short)-1,(short)0);
		SongTrack track = new SongTrack(number,"",channel,new ArrayList(),new ArrayList(),0,(RGBColor)RGBColor.RED.clone());
		//SongTrack track = new SongTrack(number,"",channel,new ArrayList(),SongManager.createDefaultInstrumentStrings(),0,(RGBColor)RGBColor.RED.clone());
		tracks.add(track);
		return track;	
	}		
	
	private MeasureHeader getHeader(long tick){
		tick = (tick >= MIN_START)?tick:MIN_START;
		
		Iterator it = headers.iterator();
		while(it.hasNext()){
			MeasureHeader header = (MeasureHeader)it.next();
			if(tick >= header.getStart() && tick < header.getStart() + header.getLength()){
				return header;
			}
		}		
		MeasureHeader last = getLastHeader();		
		int number = (last != null)?last.getNumber() + 1:1;
		long start = (last != null)?(last.getStart() + last.getLength()):1000;
		int tripletFeel = MeasureHeader.TRIPLET_FEEL_NONE;
		TimeSignature timeSignature = (last != null)?(TimeSignature)last.getTimeSignature().clone():new TimeSignature(4,new Duration(Duration.QUARTER));
		Tempo tempo = (last != null)?(Tempo)last.getTempo().clone():new Tempo(120);						
		MeasureHeader header = new MeasureHeader(number,start,timeSignature,tempo,null,tripletFeel,false,0);
		
		this.headers.add(header);
		
		if(header.getStart() + header.getLength() <= tick){		
			return header;
		}
		return getHeader(tick);
	}
	
	private MeasureHeader getLastHeader(){
		if(!this.headers.isEmpty()){
			return (MeasureHeader)this.headers.get(this.headers.size() - 1);
		}		
		return null;
	}

	private Measure getMeasure(SongTrack track,long tick){	
		tick = (tick >= MIN_START)?tick:MIN_START;		
		Iterator it = track.getMeasures().iterator();
		while(it.hasNext()){
			Measure measure = (Measure)it.next();
			if(tick >= measure.getStart() && tick < measure.getStart() + measure.getLength()){
				return measure;
			}
		}				
		getHeader(tick);
		for(int i = 0;i < headers.size();i++){
			boolean exist = false;
			MeasureHeader header = (MeasureHeader)headers.get(i);			
			for(int j = 0;j < track.getMeasures().size();j++){
				Measure measure = (Measure)track.getMeasures().get(j);				
				if(measure.getHeader().equals(header)){
					exist = true;
				}
			}
			if(!exist){
				track.getMeasures().add(new Measure(header,new ArrayList(),new ArrayList(),1,0));		
			}
		}
		return getMeasure(track,tick);
	}
	
	private TempNote getTempNote(int track,int channel,int value,boolean purge){
		for(int i = 0;i < tempNotes.size();i ++){
			TempNote note = (TempNote)tempNotes.get(i);
			if(note.track == track && note.channel == channel && note.value == value){
				if(purge){
					tempNotes.remove(i);
				}
				return note;
			}
		}
		return null;
	}
	
	private TrackTunningHelper getTrackTunningHelper(int track){
		Iterator it = this.trackTunningHelpers.iterator();
		while(it.hasNext()){
			TrackTunningHelper helper = (TrackTunningHelper)it.next();
			if(helper.getTrack() == track){
				return helper;
			}
		}
		TrackTunningHelper helper = new TrackTunningHelper(track);		
		this.trackTunningHelpers.add(helper);
		
		return helper;		
	}
	
	private void makeTempNotesBefore(long tick,int track){
		boolean check = true;
		while(check){
			check = false;
			for(int i = 0;i < tempNotes.size();i ++){
				TempNote note = (TempNote)tempNotes.get(i);
				if(note.tick < tick && note.track == track){
					//TODO
					tick = note.tick + 5000;
					makeNote(tick,track,note.channel,note.value);
					check = true;
					break;
				}
			}
		}
	}
	
	private void makeNote(long tick,int track,int channel,int value){				
		TempNote tempNote = getTempNote(track,channel,value,true);					
		if(tempNote != null){
			int nString = 0;
			int nValue = tempNote.value;						
			int nVelocity = 64;			
			long nStart = tempNote.tick;	
			Duration nDuration = Duration.fromTime(tick - tempNote.tick,MIN_DURATION);
			NoteEffect effect = new NoteEffect();

			Measure measure = getMeasure(getTrack(track),tempNote.tick);
			Note note = new Note(nValue,nStart,nDuration,nVelocity,nString,false,effect);
			measure.addNote(note);			
		}
	}

	
	public TempChannel getTempChannel(int channel){
		Iterator it = tempChannels.iterator();
		while(it.hasNext()){
			TempChannel tempChannel = (TempChannel)it.next();
			if(tempChannel.channel == channel){
				return tempChannel;
			}
		}	
		TempChannel tempChannel = new TempChannel(channel);
		tempChannels.add(tempChannel);
		
		return tempChannel;
	}	
	
	
	
	
	private void checkAll()throws Exception{
		checkTracks();
		
		int headerCount = headers.size();
		for(int i = 0;i < tracks.size();i ++){
			SongTrack track = (SongTrack)tracks.get(i);	
				
			while(track.getMeasures().size() < headerCount){				
				long start = Duration.QUARTER_TIME;
				Measure lastMeasure = (Measure)((!track.getMeasures().isEmpty())?track.getMeasures().get(track.getMeasures().size() - 1) :null);
				if(lastMeasure != null){
					start = (lastMeasure.getStart() + lastMeasure.getLength());
				}
				track.getMeasures().add(new Measure(getHeader(start),new ArrayList(),new ArrayList(),1,0));
			}
		}
		
		if(headers.isEmpty() || tracks.isEmpty()){
			throw new Exception("Empty Song");
		}
	}
	
	private void checkTracks(){
		Iterator it = tracks.iterator();
		while(it.hasNext()){
			SongTrack track = (SongTrack)it.next();
			int channel1 = 0;
			int channel2 = 0;
			Iterator tcIt = tempChannels.iterator();
			while(tcIt.hasNext()){
				TempChannel tempChannel = (TempChannel)tcIt.next();
				if(tempChannel.track == track.getNumber()){
					if(track.getChannel().getChannel() < 0){
						track.getChannel().setChannel((short)tempChannel.channel);
						track.getChannel().setInstrument((short)tempChannel.instrument);
						track.getChannel().setVolume((short)tempChannel.volume);
						track.getChannel().setBalance((short)tempChannel.balance);
					}else if(track.getChannel().getEffectChannel() < 0){
						track.getChannel().setEffectChannel((short)tempChannel.channel);
					}
				}
			}
			if(track.getChannel().getChannel() < 0){
				track.getChannel().setChannel((short)(SongManager.MAX_CHANNELS - 1));
				track.getChannel().setInstrument((short)0);
				track.getChannel().setVolume((short)127);
				track.getChannel().setBalance((short)64);				
			}
			if(track.getChannel().getEffectChannel() < 0){
				track.getChannel().setEffectChannel(track.getChannel().getChannel());
			}			
			
			if(!track.isPercussionTrack()){
				track.setStrings(getTrackTunningHelper((int)track.getNumber()).getStrings());
			}else{
				track.setStrings(SongManager.createPercusionStrings(6));
			}
		}
	}
	
	
	private class TempNote{						
		private int track;
		private int channel;
		private int value;
		private long tick;
		
		public TempNote(int track, int channel, int value,long tick) {
			this.track = track;
			this.channel = channel;
			this.value = value;
			this.tick = tick;
		}					
	}
	
	private class TempChannel{
		private int channel;						
		private int instrument;
		private int volume;
		private int balance;
		private int track;
		
		public TempChannel(int channel) {
			this.channel = channel;
			this.instrument = 0;
			this.volume = 127;
			this.balance = 64;
			this.track = -1;
		}				
	}
	
	
	private class TrackTunningHelper{
		private int track;
		private int maxValue;
		private int minValue;
		
		public TrackTunningHelper(int track){		
			this.track = track;
			this.maxValue = -1;
			this.minValue = -1;
		}
		
		public void checkValue(int value){
			if(this.minValue < 0 || value < this.minValue){
				this.minValue = value;
			}
			if(this.maxValue < 0 || value > this.maxValue){
				this.maxValue = value;
			}			
		}
		

	    private List getStrings() {
	        List strings = new ArrayList();

	        int maxFret = 24;
	        
	        if(minValue >= 40 && maxValue <= 64 + maxFret){
            	strings.add(new InstrumentString(1, 64));
            	strings.add(new InstrumentString(2, 59));
            	strings.add(new InstrumentString(3, 55));
            	strings.add(new InstrumentString(4, 50));
            	strings.add(new InstrumentString(5, 45));
            	strings.add(new InstrumentString(6, 40));	        		        	
	        }
	        else if(minValue >= 38 && maxValue <= 64 + maxFret){
            	strings.add(new InstrumentString(1, 64));
            	strings.add(new InstrumentString(2, 59));
            	strings.add(new InstrumentString(3, 55));
            	strings.add(new InstrumentString(4, 50));
            	strings.add(new InstrumentString(5, 45));
            	strings.add(new InstrumentString(6, 38));	        		        	
	        }	        
	        else if(minValue >= 35 && maxValue <= 64 + maxFret){
        		strings.add(new InstrumentString(1, 64));
        		strings.add(new InstrumentString(2, 59));
        		strings.add(new InstrumentString(3, 55));
        		strings.add(new InstrumentString(4, 50));
        		strings.add(new InstrumentString(5, 45));
        		strings.add(new InstrumentString(6, 40));
        		strings.add(new InstrumentString(7, 35));	        		        	
	        }
	        else if(minValue >= 28 && maxValue <= 43 + maxFret){
            	strings.add(new InstrumentString(1, 43));
            	strings.add(new InstrumentString(2, 38));
            	strings.add(new InstrumentString(3, 33));
            	strings.add(new InstrumentString(4, 28));	        	
	        }
	        else if(minValue >= 23 && maxValue <= 43 + maxFret){
            	strings.add(new InstrumentString(1, 43));
            	strings.add(new InstrumentString(2, 38));
            	strings.add(new InstrumentString(3, 33));
            	strings.add(new InstrumentString(4, 28));
            	strings.add(new InstrumentString(5, 23));
	        }else{
		        int stringCount = 6;	        	
	        	int stringSpacing = ((maxValue - (maxFret - 4) - minValue) / stringCount);
	        	if(stringSpacing > 5){
			        stringCount = 7;	        	
		        	stringSpacing = ((maxValue - (maxFret - 4) - minValue) / stringCount);	        		
	        	}
	        	
	        	int maxStringValue = (minValue + (stringCount * stringSpacing));
	        	while(strings.size() < stringCount){	     
	        		maxStringValue -= stringSpacing;
	        		strings.add(new InstrumentString(strings.size() + 1,maxStringValue));
	        	}
	        }

	        return strings;
	    }		

		public int getMaxValue() {
			return maxValue;
		}

		public int getMinValue() {
			return minValue;
		}

		public int getTrack() {
			return track;
		}
		
	}
	
	private class SongAdjuster{				
		private Song song;
		private long minDurationTime;
		
		public SongAdjuster(Song song){
			this.song = song;
			this.minDurationTime = 60;
		}
		
		public Song adjustSong(){
			Iterator it = song.getTracks().iterator();
		
			while(it.hasNext()){
				SongTrack track = (SongTrack)it.next();
				adjustTrack(track);			
			}
			return song;
		}
		
		private void adjustTrack(SongTrack track){
			Iterator it = track.getMeasures().iterator();
			while(it.hasNext()){
				Measure measure = (Measure)it.next();
				adjustMeasure(measure);
				adjustStrings(track,measure);
			}
		}

		/*
		private void adjustMeasure(Measure measure){						

			for(int i = 0; i < measure.getNotes().size();i++){
				Note note = (Note)measure.getNotes().get(i);
				long start = note.getStart();
				long time = note.getDuration().getTime();				
				if(i == 0 && measure.getStart() + minDurationTime > start){
					note.setStart(measure.getStart());
				}				
				int nextIndex = (i + 1);
				for(int j = nextIndex; j < measure.getNotes().size();j++){
					Note next = (Note)measure.getNotes().get(j);
					long nextTime = next.getDuration().getTime();
					long diff = 0;
					if(start + minDurationTime > next.getStart()){
						start = (next.getStart() > start)?next.getStart():start;
						next.setStart(note.getStart());
						if(time > nextTime){
							next.setDuration((Duration)note.getDuration().clone());
							time = next.getDuration().getTime();
						}else if(time < nextTime){
							note.setDuration((Duration)next.getDuration().clone());
						}
					}else{	
						if(note.getStart() + time + minDurationTime + diff > next.getStart()){
							diff = (next.getStart() - (note.getStart() + time));
							next.setStart(note.getStart() + time);						
						}else{
							break;
						}
					}
				}
			}
		}
		*/
		private void adjustMeasure(Measure measure){						
			List beatNotes = new ArrayList();
			long nextStart = measure.getStart();			
			
			for(int i = 0; i < measure.getNotes().size();i++){
				Note note = (Note)measure.getNotes().get(i);
				long start = note.getStart();
				int nextIndex = (i + 1);
				beatNotes.add(note);
				
				if((nextStart < start && nextStart + minDurationTime > start) || (nextStart > start && nextStart - minDurationTime < start)){
					note.setStart(nextStart);
				}

				long end = measure.getStart() + measure.getLength();
				for(int j = nextIndex; j < measure.getNotes().size();j++){
					Note next = (Note)measure.getNotes().get(j);
					long nextTime = next.getDuration().getTime();
					long diff = 0;
					if(start + minDurationTime > next.getStart()){
						start = (next.getStart() > start)?next.getStart():start;
						beatNotes.add(next);	
						i = j;
					}else{
						end= next.getStart();
						break;
					}
				}
				nextStart = nextStart + adjustBeat(beatNotes,note.getStart(),end);
				beatNotes.clear();				
			}
		}		
		
		private long adjustBeat(List beatNotes,long start,long nextBeatStart){
			Duration lengthDuration = Duration.fromTime(nextBeatStart - start,MIN_DURATION);			
			long length = lengthDuration.getTime();
			
			Duration maxDuration = null;
			for(int i = 0; i < beatNotes.size();i++){
				Note note = (Note)beatNotes.get(i);
				note.setStart(start);								
				if(maxDuration == null || note.getDuration().getTime() > maxDuration.getTime()){
					maxDuration = note.getDuration();
				}
			}
			if(maxDuration != null){	
				if(maxDuration.getTime() > length){
					maxDuration = lengthDuration;
				}				
				for(int i = 0; i < beatNotes.size();i++){
					Note note = (Note)beatNotes.get(i);
					note.setStart(start);
					note.setDuration((Duration)maxDuration.clone());
				}			
			}
			
			return length;
		}
		
		
		private void adjustStrings(SongTrack track,Measure measure){
			List tempStrings = new ArrayList();
			
			//ajusto las cuerdas
			long start = 0;		
			Iterator it = measure.getNotes().iterator();
			while(it.hasNext()){
				Note note = (Note)it.next();

				//checkeo las cuerdas
				if(note.getStart() != start){
					tempStrings.clear();
					tempStrings.addAll(track.getStrings());
				}
				int string = getStringForValue(tempStrings,note.getValue());
				for(int j = 0;j < tempStrings.size();j ++){
					InstrumentString tempString = (InstrumentString)tempStrings.get(j);	
					if(tempString.getNumber() == string){						
						note.setValue(note.getValue() - tempString.getValue());
						note.setString(tempString.getNumber());
						tempStrings.remove(j);
						break;
					}
				}				
				if(note.getString() < 1){
					string = getStringForValue(track.getStrings(),note.getValue());
					note.setValue(note.getValue() - ((InstrumentString)track.getStrings().get(string - 1)).getValue());
					note.setString(string);
				}				
				start = note.getStart();
			}
		}		
		
		
		
		private int getStringForValue(List strings,int value){
			int minFret = -1;
			int stringForValue = 0;
			for(int i = 0;i < strings.size();i++){
				InstrumentString string = (InstrumentString)strings.get(i);						
				int fret = value - string.getValue();			
				if(minFret < 0 || (fret >= 0 && fret < minFret)){
					stringForValue = string.getNumber();
					minFret = fret;				
				}
			}						
			return stringForValue;
		}	
		
	}
}
