-- `Topal': GPG/Pine integration
--
-- Copyright (C) 2001  Phillip J. Brooke
--
--     This program is free software; you can redistribute it and/or modify
--     it under the terms of the GNU General Public License as published by
--     the Free Software Foundation; either version 2 of the License, or
--     (at your option) any later version.
--
--     This program is distributed in the hope that it will be useful,
--     but WITHOUT ANY WARRANTY; without even the implied warranty of
--     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--     GNU General Public License for more details.
--
--     You should have received a copy of the GNU General Public License
--     along with this program; if not, write to the Free Software
--     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

with Ada.Text_IO;

package body Expanding_Array is

   procedure Create (Left         : in out Big_Array;
                     Initial_Size : in     Positive) is
      A : Big_Array;
   begin
      A.Actual := new Actual_Array(1..Initial_Size);
      A.Array_Size := Initial_Size;
      Left := A;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Create");
         raise;
   end Create;

   procedure Copy (Left  : in out Big_Array;
                   Right : in     Big_Array) is
      A : Big_Array;
   begin
      -- Now, deep copy the array.
      A.Actual := new Actual_Array(1..Size(Right));
      A.Array_Size := Size(Right);
      A.Actual.all(1..A.Array_Size) := Right.Actual.all(1..Right.Array_Size);
      Left := A;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Copy");
         raise;
   end Copy;

   function Size (A : Big_Array) return Positive is
   begin
      return A.Array_Size;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Size");
         raise;
   end Size;

   function Last (A : Big_Array) return Positive is
   begin
      return Size(A);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Last");
         raise;
   end Last;

   procedure Expand(A : in out Big_Array;
                    S : in     Positive) is
      New_Actual : Actual_Array_Pointer;
   begin
      New_Actual := new Actual_Array(1..A.Array_Size + S);
      New_Actual.all(1..A.Array_Size) := A.Actual.all(1..A.Array_Size);
      A.Actual := New_Actual;
      A.Array_Size := A.Array_Size + S;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Expand");
         raise;
   end Expand;

   function First (A : Big_Array) return Natural is
   begin
      return A.Actual.all'First;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.First");
         raise;
   end First;

   procedure Set(A     : in out Big_Array;
                 Index : in     Positive;
                 Value : in     Item) is
   begin
      -- Is the array large enough?
      while Last(A) < Index loop
         -- Double the size....
         Expand(A, A.Array_Size);
      end loop;
      -- Set the value.
      A.Actual(Index) := Value;
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Set");
         raise;
   end Set;

   function Value(A     : Big_Array;
                  Index : Positive) return Item is
   begin
      return A.Actual(Index);
   exception
      when others =>
         Ada.Text_IO.Put_Line(Ada.Text_IO.Standard_Error,
                              "Exception raised in Expanding_Array.Value");
         raise;
   end Value;


end Expanding_Array;
