package tijmp.ui;

import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

/** A class that displays a class fields, the list of super classes.
 */
public class ClassInspector extends JPanel {

    private JList interfaces;
    private JList staticFields;
    private JList instanceFields;

    public ClassInspector (Class<?> clz) {
	JComboBox cb = getClassCombo (clz);
	interfaces = getInterfaces (clz);
	staticFields = getFields (clz, new StaticFieldFilter ());
	instanceFields = getFields (clz, new InstanceFieldFilter ());
	
	GridBagLayout gb = new GridBagLayout ();
	GridBagConstraints gc = new GridBagConstraints ();
	setLayout (gb);

	gc.insets = new Insets (2, 2, 2, 2);
	gc.gridx = 1;
	gc.gridy = 1;
	gc.fill = GridBagConstraints.HORIZONTAL;
	gc.anchor = GridBagConstraints.NORTHWEST;
	add (new JLabel ("Class"), gc);
	gc.gridy++;
	add (new JLabel ("Interfaces"), gc);
	gc.gridy++;
	add (new JLabel ("Static fields"), gc);
	gc.gridy++;
	add (new JLabel ("Instance fields"), gc);

	gc.gridx = 2;
	gc.gridy = 1;
	gc.weightx = 1;
	gc.fill = GridBagConstraints.HORIZONTAL;
	add (cb, gc);
	gc.gridy++;
	gc.weighty = 1;
	gc.fill = GridBagConstraints.BOTH;
	add (new JScrollPane (interfaces), gc);
	gc.gridy++;
	add (new JScrollPane (staticFields), gc);
	gc.gridy++;
	gc.weighty = 3;
	add (new JScrollPane (instanceFields), gc);
    }

    private static interface FieldFilter {
	boolean accept (Field f);
    }

    private static class InstanceFieldFilter implements FieldFilter {
	public boolean accept (Field f) {
	    return !Modifier.isStatic (f.getModifiers ());
	}
    }

    private static class StaticFieldFilter implements FieldFilter {
	public boolean accept (Field f) {
	    return Modifier.isStatic (f.getModifiers ());
	}
    }
    
    private JList getFields (Class<?> clz, FieldFilter ff) {
	Field[] fields = clz.getDeclaredFields ();
	if (fields.length == 0)
	    return new JList ();
	List<Field> ls = new ArrayList<Field> (fields.length);
	for (Field f : fields) {
	    if (ff.accept (f))
		ls.add (f);
	}
	fields = new Field[ls.size ()];
	ls.toArray (fields);
	JList jl = new JList (fields);
	jl.setCellRenderer (new FieldRenderer ());
	return jl;
    }

    private JList getInterfaces (Class<?> clz) {
	Class<?>[] classes = clz.getInterfaces ();
	if (classes.length == 0)
	    return new JList ();
	JList ls = new JList (classes);
	ls.setCellRenderer (new ListClassRenderer ());
	return ls;
    }

    private JComboBox getClassCombo (Class<?> sc) {
	List<Class<?>> classes = new ArrayList<Class<?>> ();
	while (sc != null) {
	    classes.add (0, sc);
	    sc = sc.getSuperclass ();
	}
	JComboBox cb = new JComboBox (classes.toArray ());
	cb.setSelectedIndex (classes.size () - 1);
	cb.setRenderer (new ListClassRenderer ());
	cb.addActionListener (new SetClass ());
	return cb;
    }

    private class SetClass implements ActionListener {
	public void actionPerformed (ActionEvent e) {
	    JComboBox cb = (JComboBox)e.getSource ();
	    Class clz = (Class)cb.getSelectedItem ();
	    JList ni = getInterfaces (clz);
	    interfaces.setModel (ni.getModel ());
	    JList nsf = getFields (clz, new StaticFieldFilter ());
	    staticFields.setModel (nsf.getModel ());
	    JList nif = getFields (clz, new InstanceFieldFilter ());
	    instanceFields.setModel (nif.getModel ());
	}
    }
}
