/*
 * Copyright (c) 2005-2009 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.skin;

import java.awt.*;
import java.util.Map;

import javax.swing.SwingUtilities;

import org.jvnet.substance.api.*;
import org.jvnet.substance.painter.border.*;
import org.jvnet.substance.painter.decoration.*;
import org.jvnet.substance.painter.gradient.StandardGradientPainter;
import org.jvnet.substance.painter.highlight.ClassicHighlightPainter;
import org.jvnet.substance.shaper.ClassicButtonShaper;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceSizeUtils;

/**
 * <code>Twilight</code> skin. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 5.2
 */
public class TwilightSkin extends SubstanceSkin {
	/**
	 * Display name for <code>this</code> skin.
	 */
	public static final String NAME = "Twilight";

	/**
	 * Creates a new <code>Twilight</code> skin.
	 */
	public TwilightSkin() {
		Map<String, SubstanceColorScheme> schemes = SubstanceSkin
				.getColorSchemes(TwilightSkin.class
						.getClassLoader()
						.getResource(
								"org/jvnet/substance/skin/twilight.colorschemes"));
		SubstanceColorScheme activeScheme = schemes.get("Twilight Active");
		SubstanceColorScheme defaultScheme = schemes.get("Twilight Default");

		SubstanceColorSchemeBundle defaultSchemeBundle = new SubstanceColorSchemeBundle(
				activeScheme, defaultScheme, defaultScheme);
		defaultSchemeBundle.registerColorScheme(defaultScheme, 0.5f,
				ComponentState.DISABLED_UNSELECTED);

		// borders
		SubstanceColorScheme borderDisabledSelectedScheme = schemes
				.get("Twilight Disabled Selected Border");
		SubstanceColorScheme borderScheme = schemes.get("Twilight Border");
		defaultSchemeBundle.registerColorScheme(borderDisabledSelectedScheme,
				ColorSchemeAssociationKind.BORDER,
				ComponentState.DISABLED_SELECTED);
		defaultSchemeBundle.registerColorScheme(borderScheme,
				ColorSchemeAssociationKind.BORDER);

		// marks
		SubstanceColorScheme markActiveScheme = schemes
				.get("Twilight Mark Active");
		defaultSchemeBundle.registerColorScheme(markActiveScheme,
				ColorSchemeAssociationKind.MARK, ComponentState
						.getActiveStates());

		// separators
		SubstanceColorScheme separatorScheme = schemes
				.get("Twilight Separator");
		defaultSchemeBundle.registerColorScheme(separatorScheme,
				ColorSchemeAssociationKind.SEPARATOR);

		SubstanceColorScheme watermarkScheme = schemes
				.get("Twilight Watermark");

		this.registerDecorationAreaSchemeBundle(defaultSchemeBundle,
				watermarkScheme, DecorationAreaType.NONE);

		SubstanceColorSchemeBundle decorationsSchemeBundle = new SubstanceColorSchemeBundle(
				activeScheme, defaultScheme, defaultScheme);
		decorationsSchemeBundle.registerColorScheme(defaultScheme, 0.5f,
				ComponentState.DISABLED_UNSELECTED);

		// borders
		decorationsSchemeBundle.registerColorScheme(
				borderDisabledSelectedScheme,
				ColorSchemeAssociationKind.BORDER,
				ComponentState.DISABLED_SELECTED);
		decorationsSchemeBundle.registerColorScheme(borderScheme,
				ColorSchemeAssociationKind.BORDER);

		// marks
		decorationsSchemeBundle.registerColorScheme(markActiveScheme,
				ColorSchemeAssociationKind.MARK, ComponentState
						.getActiveStates());

		// separators
		SubstanceColorScheme separatorDecorationsScheme = schemes
				.get("Twilight Decorations Separator");
		decorationsSchemeBundle.registerColorScheme(separatorDecorationsScheme,
				ColorSchemeAssociationKind.SEPARATOR);

		SubstanceColorScheme decorationsWatermarkScheme = schemes
				.get("Twilight Decorations Watermark");

		this.registerDecorationAreaSchemeBundle(decorationsSchemeBundle,
				decorationsWatermarkScheme, DecorationAreaType.TOOLBAR,
				DecorationAreaType.GENERAL, DecorationAreaType.FOOTER);

		SubstanceColorSchemeBundle headerSchemeBundle = new SubstanceColorSchemeBundle(
				activeScheme, defaultScheme, defaultScheme);
		headerSchemeBundle.registerColorScheme(defaultScheme, 0.5f,
				ComponentState.DISABLED_UNSELECTED);

		// borders
		headerSchemeBundle.registerColorScheme(borderDisabledSelectedScheme,
				ColorSchemeAssociationKind.BORDER,
				ComponentState.DISABLED_SELECTED);
		headerSchemeBundle.registerColorScheme(borderScheme,
				ColorSchemeAssociationKind.BORDER);
		// marks
		headerSchemeBundle.registerColorScheme(markActiveScheme,
				ColorSchemeAssociationKind.MARK, ComponentState
						.getActiveStates());

		headerSchemeBundle.registerHighlightColorScheme(activeScheme, 0.7f,
				ComponentState.ROLLOVER_UNSELECTED,
				ComponentState.ROLLOVER_ARMED, ComponentState.ARMED);
		headerSchemeBundle.registerHighlightColorScheme(activeScheme, 0.8f,
				ComponentState.SELECTED);
		headerSchemeBundle.registerHighlightColorScheme(activeScheme, 1.0f,
				ComponentState.ROLLOVER_SELECTED);

		SubstanceColorScheme headerWatermarkScheme = schemes
				.get("Twilight Header Watermark");

		this.registerDecorationAreaSchemeBundle(headerSchemeBundle,
				headerWatermarkScheme, DecorationAreaType.PRIMARY_TITLE_PANE,
				DecorationAreaType.SECONDARY_TITLE_PANE,
				DecorationAreaType.HEADER);

		setSelectedTabFadeStart(0.2);
		setSelectedTabFadeEnd(0.9);

		this.buttonShaper = new ClassicButtonShaper();
		this.watermark = null;
		this.gradientPainter = new StandardGradientPainter();
		this.decorationPainter = new TwilightDecorationPainter();
		this.highlightPainter = new ClassicHighlightPainter();
		this.borderPainter = new CompositeBorderPainter("Twilight",
				new ClassicBorderPainter(), new DelegateBorderPainter(
						"Twilight Inner", new ClassicBorderPainter(),
						0x40FFFFFF, 0x20FFFFFF, 0x00FFFFFF,
						new ColorSchemeTransform() {
							@Override
							public SubstanceColorScheme transform(
									SubstanceColorScheme scheme) {
								return scheme.tint(0.3);
							}
						}));
	}

	public String getDisplayName() {
		return NAME;
	}

	/**
	 * Custom decoration painter for the Twilight skins.
	 * 
	 * @author Kirill Grouchnikov
	 */
	private class TwilightDecorationPainter extends MatteDecorationPainter {
		@Override
		public String getDisplayName() {
			return "Twilight";
		}

		@Override
		public void paintDecorationArea(Graphics2D graphics, Component comp,
				DecorationAreaType decorationAreaType, int width, int height,
				SubstanceSkin skin) {
			Component c = comp;
			Component topMostWithSameDecorationAreaType = c;
			while (c != null) {
				if (SubstanceDecorationUtilities.getImmediateDecorationType(c) == decorationAreaType) {
					topMostWithSameDecorationAreaType = c;
				}
				c = c.getParent();
			}
			int topHeight = topMostWithSameDecorationAreaType.getHeight();

			Point inTopMost = SwingUtilities.convertPoint(comp,
					new Point(0, 0), topMostWithSameDecorationAreaType);
			int dy = inTopMost.y;

			SubstanceColorScheme fillScheme = skin
					.getBackgroundColorScheme(decorationAreaType);
			Graphics2D fillGraphics = (Graphics2D) graphics.create();
			fillGraphics.translate(0, -dy);

			fillGraphics.setColor(fillScheme.getMidColor());
			fillGraphics.fillRect(0, 0, width, topHeight);

			int shadowHeight = 4;
			GradientPaint fillPaint = new GradientPaint(0, topHeight
					- shadowHeight, SubstanceColorUtilities.getAlphaColor(
					fillScheme.getUltraDarkColor().darker(), 0), 0, topHeight,
					SubstanceColorUtilities.getAlphaColor(fillScheme
							.getUltraDarkColor().darker(), 128));
			fillGraphics.setPaint(fillPaint);
			fillGraphics.fillRect(0, topHeight - shadowHeight, width,
					shadowHeight);
			fillGraphics.dispose();

			if (decorationAreaType == DecorationAreaType.TOOLBAR) {
				int fontSize = SubstanceSizeUtils.getComponentFontSize(c);
				float borderStrokeWidth = SubstanceSizeUtils
						.getBorderStrokeWidth(fontSize);
				graphics.setStroke(new BasicStroke(borderStrokeWidth));

				SubstanceColorScheme colorScheme = skin.getColorScheme(comp,
						ComponentState.DEFAULT);
				graphics.setColor(SubstanceColorUtilities.getAlphaColor(
						colorScheme.getForegroundColor(), 32));
				graphics.translate(0, -dy - 1);
				graphics.drawLine(0, (int) borderStrokeWidth, width,
						(int) borderStrokeWidth);
				graphics.setColor(colorScheme.getUltraDarkColor());
				graphics.drawLine(0, 1
						+ topMostWithSameDecorationAreaType.getHeight()
						- (int) borderStrokeWidth, width, 1
						+ topMostWithSameDecorationAreaType.getHeight()
						- (int) borderStrokeWidth);
			}
			if (decorationAreaType == DecorationAreaType.FOOTER) {
				graphics.translate(0, -dy);

				int fontSize = SubstanceSizeUtils.getComponentFontSize(c);
				float borderStrokeWidth = SubstanceSizeUtils
						.getBorderStrokeWidth(fontSize);
				graphics.setStroke(new BasicStroke(borderStrokeWidth));

				SubstanceColorScheme colorScheme = skin.getColorScheme(comp,
						ComponentState.DEFAULT);
				graphics.setColor(colorScheme.getUltraDarkColor());
				graphics.drawLine(0, (int) borderStrokeWidth, width,
						(int) borderStrokeWidth);
				graphics.setColor(SubstanceColorUtilities.getAlphaColor(
						colorScheme.getForegroundColor(), 32));
				graphics.drawLine(0, (int) (2 * borderStrokeWidth), width,
						(int) (2 * borderStrokeWidth));
			}
		}
	}
}
