/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.utils.icon;

import java.awt.*;
import java.awt.image.BufferedImage;

import javax.swing.*;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeState;
import org.jvnet.substance.api.ComponentState;
import org.jvnet.substance.api.SubstanceColorScheme;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.icon.TransitionAwareIcon.Delegate;

public class ArrowButtonTransitionAwareIcon implements Icon {
	/**
	 * Icon cache to speed up the subsequent icon painting. The basic assumption
	 * is that the {@link #delegate} returns an icon that paints the same for
	 * the same parameters.
	 */
	private static LazyResettableHashMap<Icon> iconMap = new LazyResettableHashMap<Icon>(
			"ButtonArrowTransitionAwareIcon");

	private AbstractButton button;

	private int orientation;

	private int iconWidth;

	private int iconHeight;

	/**
	 * Delegate to compute the actual icons.
	 */
	protected Delegate delegate;

	public ArrowButtonTransitionAwareIcon(final AbstractButton button,
			final int orientation) {
		this.button = button;
		this.orientation = orientation;
		this.delegate = new TransitionAwareIcon.Delegate() {
			@Override
			public Icon getColorSchemeIcon(SubstanceColorScheme scheme) {
				int fontSize = SubstanceSizeUtils.getComponentFontSize(button);
				return SubstanceImageCreator.getArrowIcon(fontSize,
						orientation, scheme);
			}
		};

		this.iconWidth = this.delegate.getColorSchemeIcon(
				SubstanceColorSchemeUtilities.getColorScheme(this.button,
						ComponentState.DEFAULT)).getIconWidth();
		this.iconHeight = this.delegate.getColorSchemeIcon(
				SubstanceColorSchemeUtilities.getColorScheme(this.button,
						ComponentState.DEFAULT)).getIconHeight();
	}

	@Override
	public void paintIcon(Component c, Graphics g, int x, int y) {
		this.getIconToPaint((AbstractButton) c).paintIcon(c, g, x, y);
	}

	private Icon getIconToPaint(AbstractButton button) {
		ButtonModel model = button.getModel();
		ComponentState currState = ComponentState.getState(model, button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);
		if (!currState.isKindActive(FadeKind.ENABLE))
			prevState = currState;
		float cyclePos = currState.getCyclePosition();
		SubstanceColorScheme currScheme = SubstanceColorSchemeUtilities
				.getColorScheme(button, currState);

		SubstanceColorScheme prevScheme = currScheme;

		FadeState fadeState = SubstanceFadeUtilities.getFadeState(button,
				FadeKind.ROLLOVER, FadeKind.SELECTION, FadeKind.PRESS,
				FadeKind.ARM);
		if (fadeState != null) {
			prevScheme = SubstanceColorSchemeUtilities.getColorScheme(button,
					prevState);
			cyclePos = fadeState.getFadePosition();
			if (!fadeState.isFadingIn())
				cyclePos = 1.0f - cyclePos;
		}
		float currAlpha = SubstanceColorSchemeUtilities.getAlpha(button, currState);
		float prevAlpha = SubstanceColorSchemeUtilities.getAlpha(button, prevState);

		HashMapKey key = SubstanceCoreUtilities.getHashKey(this.orientation,
				SubstanceSizeUtils.getComponentFontSize(this.button),
				currScheme.getDisplayName(), prevScheme.getDisplayName(),
				currAlpha, prevAlpha, cyclePos);
		if (!iconMap.containsKey(key)) {
			Icon icon = this.delegate.getColorSchemeIcon(currScheme);
			Icon prevIcon = this.delegate.getColorSchemeIcon(prevScheme);

			BufferedImage temp = SubstanceCoreUtilities.getBlankImage(icon
					.getIconWidth(), icon.getIconHeight());
			Graphics2D g2d = temp.createGraphics();

			if (currScheme == prevScheme) {
				// same theme - can paint just the current icon, no matter
				// what the cycle position is.
				g2d.setComposite(AlphaComposite.getInstance(
						AlphaComposite.SRC_OVER, currAlpha));
				icon.paintIcon(button, g2d, 0, 0);
			} else {
				// make optimizations for limit values of the cycle position.
				if (cyclePos < 1.0f) {
					g2d.setComposite(AlphaComposite.SrcOver.derive(prevAlpha));
					prevIcon.paintIcon(button, g2d, 0, 0);
				}
				if (cyclePos > 0.0f) {
					g2d.setComposite(AlphaComposite.SrcOver.derive(currAlpha * cyclePos));
					icon.paintIcon(button, g2d, 0, 0);
				}
			}

			iconMap.put(key, new ImageIcon(temp));
			g2d.dispose();
		}

		return iconMap.get(key);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.Icon#getIconHeight()
	 */
	public int getIconHeight() {
		return this.iconHeight;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.Icon#getIconWidth()
	 */
	public int getIconWidth() {
		return this.iconWidth;
	}
}
