/*
 * Copyright (c) 2005-2008 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.painter.highlight;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;
import java.util.EnumSet;
import java.util.Set;

import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.api.SubstanceColorScheme;
import org.jvnet.substance.api.SubstanceConstants;
import org.jvnet.substance.api.SubstanceConstants.Side;
import org.jvnet.substance.painter.border.SubstanceBorderPainter;
import org.jvnet.substance.painter.gradient.ClassicGradientPainter;
import org.jvnet.substance.utils.*;

/**
 * Highlight painter that paints a classic gradient. This class is part of
 * officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.3
 */
public class ClassicHighlightPainter implements SubstanceHighlightPainter {
	/**
	 * The display name for the highlight painters of this class.
	 */
	public static final String DISPLAY_NAME = "Classic";

	/**
	 * Cache for small objects.
	 */
	protected static LazyResettableHashMap<BufferedImage> smallImageCache = new LazyResettableHashMap<BufferedImage>(
			"ClassicHighlightPainter");

	/**
	 * Single gradient painter instance.
	 */
	protected ClassicGradientPainter painter;

	/**
	 * Creates new classic title painter.
	 */
	public ClassicHighlightPainter() {
		this.painter = new ClassicGradientPainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @seeorg.jvnet.substance.painter.highlight.SubstanceHighlightPainter#
	 * paintHighlight(java.awt.Graphics2D, java.awt.Component, int, int, float,
	 * java.util.Set, org.jvnet.substance.api.SubstanceColorScheme,
	 * org.jvnet.substance.api.SubstanceColorScheme, float)
	 */
	public void paintHighlight(Graphics2D graphics, Component comp, int width,
			int height, float borderAlpha, Set<Side> openSides,
			SubstanceColorScheme colorScheme1,
			SubstanceColorScheme colorScheme2, float cyclePos) {

		SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
				.getBorderPainter(comp);

		if (openSides == null) {
			openSides = EnumSet.noneOf(Side.class);
		}
		if (width * height < 100000) {
			String openKey = "";
			for (Side oSide : openSides) {
				openKey += oSide.name() + "-";
			}

			HashMapKey key = SubstanceCoreUtilities.getHashKey(width, height,
					colorScheme1.getDisplayName(), colorScheme2
							.getDisplayName(), cyclePos, borderAlpha, openKey,
					borderPainter.getDisplayName());
			BufferedImage result = smallImageCache.get(key);
			if (result == null) {
				result = SubstanceCoreUtilities.getBlankImage(width, height);
				this.internalPaint((Graphics2D) result.getGraphics(), comp,
						width, height, colorScheme1, colorScheme2, cyclePos,
						borderPainter, borderAlpha, openSides);
				smallImageCache.put(key, result);
			}
			graphics.drawImage(result, 0, 0, null);
			return;
		}

		this.internalPaint(graphics, comp, width, height, colorScheme1,
				colorScheme2, cyclePos, borderPainter, borderAlpha, openSides);
	}

	/**
	 * Paints the highlight.
	 * 
	 * @param graphics
	 *            Graphics context.
	 * @param comp
	 *            Component.
	 * @param width
	 *            Width.
	 * @param height
	 *            Height.
	 * @param borderPainter
	 *            Border painter.
	 * @param borderAlpha
	 *            Border alpha factor.
	 * @param openSides
	 *            The sides specified in this set will not be painted. Can be
	 *            <code>null</code> or empty.
	 * @param colorScheme1
	 *            The first color scheme.
	 * @param colorScheme2
	 *            The second color scheme.
	 * @param cyclePos
	 *            Cycle position. Is used for rollover and selection animations.
	 *            Must be in 0..10 range.
	 */
	protected void internalPaint(Graphics2D graphics, Component comp,
			int width, int height, SubstanceColorScheme colorScheme1,
			SubstanceColorScheme colorScheme2, float cyclePos,
			SubstanceBorderPainter borderPainter, float borderAlpha,
			Set<SubstanceConstants.Side> openSides) {

		graphics.translate(-3, -3);
		this.painter.paintContourBackground(graphics, comp, width + 6,
				height + 6, new Rectangle(width + 6, height + 6), false,
				colorScheme1, colorScheme2, cyclePos, false, true);
		graphics.translate(3, 3);
		if (borderAlpha > 0.0f) {
			int openDelta = 3 + (int) (Math.ceil(3.0 * SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(comp))));
			int deltaLeft = openSides.contains(Side.LEFT) ? openDelta : 0;
			int deltaRight = openSides.contains(Side.RIGHT) ? openDelta : 0;
			int deltaTop = openSides.contains(Side.TOP) ? openDelta : 0;
			int deltaBottom = openSides.contains(Side.BOTTOM) ? openDelta : 0;

			int borderDelta = (int) Math.floor(SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(comp)) / 2.0);
			GeneralPath contour = SubstanceOutlineUtilities.getBaseOutline(
					width + deltaLeft + deltaRight, height + deltaTop
							+ deltaBottom, 0.0f, null, borderDelta);
			// (null, new Insets(
			// borderDelta, borderDelta, borderDelta, borderDelta), width
			// + deltaLeft + deltaRight, height + deltaTop + deltaBottom);

			graphics.translate(-deltaLeft, -deltaTop);

			if (borderAlpha > 0.0f) {
				Graphics2D g2d = (Graphics2D) graphics.create();
				g2d.setComposite(TransitionLayout.getAlphaComposite(null,
						borderAlpha, graphics));
				int borderThickness = (int) SubstanceSizeUtils
						.getBorderStrokeWidth(SubstanceSizeUtils
								.getComponentFontSize(comp));
				GeneralPath contourInner = SubstanceOutlineUtilities
						.getBaseOutline(width + deltaLeft + deltaRight, height
								+ deltaTop + deltaBottom, 0.0f, null,
								borderDelta + borderThickness);
				borderPainter.paintBorder(g2d, comp, width + deltaLeft
						+ deltaRight, height + deltaTop + deltaBottom, contour,
						contourInner, colorScheme1, colorScheme2, cyclePos,
						true);
				g2d.dispose();
			}
		}
	}

	/**
	 * Returns the memory usage string.
	 * 
	 * @return Memory usage string.
	 */
	public static String getMemoryUsage() {
		StringBuffer sb = new StringBuffer();
		sb.append("ClassicHighlightPainter: \n");
		sb.append("\t" + smallImageCache.size() + " regular");
		// + pairwiseBackgrounds.size() + " pairwise");
		return sb.toString();
	}
}
