/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.skin;

import java.awt.Color;

import org.jvnet.substance.border.ClassicInnerBorderPainter;
import org.jvnet.substance.button.ClassicButtonShaper;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.theme.SubstanceComplexTheme;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;
import org.jvnet.substance.title.MarbleNoiseHeaderPainter;
import org.jvnet.substance.utils.ComponentState;
import org.jvnet.substance.utils.SubstanceColorUtilities;
import org.jvnet.substance.utils.SubstanceConstants.ColorShiftKind;
import org.jvnet.substance.watermark.SubstanceNullWatermark;

/**
 * <code>Autumn</code> skin. This class is part of officially supported API.
 * 
 * @author Kirill Grouchnikov
 * @since version 4.0
 */
public class AutumnSkin extends SubstanceAbstractSkin {
	/**
	 * Display name for <code>this</code> skin.
	 */
	public static String NAME = "Autumn";

	/**
	 * Color scheme for active visual state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class ActiveScheme implements ColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#FDDDAC");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#FCEF9F");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#FCD592");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#F9BE84");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#F8B87A");

		/**
		 * The main ultra-dark color.
		 */
		// private static final Color mainUltraDarkColor =
		// Color.decode("#5B0E00");
		private static final Color mainUltraDarkColor = Color.decode("#AC623B");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#AC623B");

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return ActiveScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return ActiveScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return ActiveScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return ActiveScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return ActiveScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return ActiveScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return ActiveScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Color scheme for default visual state.
	 * 
	 * @author Kirill Grouchnikov
	 */
	protected static class DefaultScheme implements ColorScheme {
		/**
		 * The main ultra-light color.
		 */
		private static final Color mainUltraLightColor = Color
				.decode("#FFF2DF");

		/**
		 * The main extra-light color.
		 */
		private static final Color mainExtraLightColor = Color
				.decode("#FFE3C5");

		/**
		 * The main light color.
		 */
		private static final Color mainLightColor = Color.decode("#FDD1A4");

		/**
		 * The main medium color.
		 */
		private static final Color mainMidColor = Color.decode("#FBCD9C");

		/**
		 * The main dark color.
		 */
		private static final Color mainDarkColor = Color.decode("#FCC896");

		/**
		 * The main ultra-dark color.
		 */
		// private static final Color mainUltraDarkColor =
		// Color.decode("#872A00");
		private static final Color mainUltraDarkColor = Color.decode("#AC623B");

		/**
		 * The foreground color.
		 */
		private static final Color foregroundColor = Color.decode("#AC623B");

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
		 */
		public Color getForegroundColor() {
			return DefaultScheme.foregroundColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
		 */
		public Color getUltraLightColor() {
			return DefaultScheme.mainUltraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
		 */
		public Color getExtraLightColor() {
			return DefaultScheme.mainExtraLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
		 */
		public Color getLightColor() {
			return DefaultScheme.mainLightColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
		 */
		public Color getMidColor() {
			return DefaultScheme.mainMidColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
		 */
		public Color getDarkColor() {
			return DefaultScheme.mainDarkColor;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
		 */
		public Color getUltraDarkColor() {
			return DefaultScheme.mainUltraDarkColor;
		}
	}

	/**
	 * Creates a new <code>Autumn</code> skin.
	 */
	public AutumnSkin() {
		SubstanceTheme activeTheme = new SubstanceTheme(new ActiveScheme(),
				"Autumn Active", ThemeKind.COLD);
		SubstanceTheme defaultTheme = new SubstanceTheme(new DefaultScheme(),
				"Autumn Default", ThemeKind.COLD);

		// final ColorScheme disabledScheme = new
		// BlendBiColorScheme(defaultTheme
		// .getColorScheme(), defaultTheme.getDisabledTheme()
		// .getColorScheme(), 0.6) {
		// Color foreColor = new Color(128, 131, 136);
		//
		// @Override
		// public Color getForegroundColor() {
		// return foreColor;
		// }
		// };
		SubstanceTheme disabledTheme = defaultTheme;
		// new SubstanceTheme(disabledScheme,
		// "Disabled Autumn", ThemeKind.COLD);

		SubstanceTheme activeTitleTheme = defaultTheme;

		SubstanceComplexTheme theme = new SubstanceComplexTheme(NAME,
				ThemeKind.COLD, activeTheme, defaultTheme, disabledTheme,
				activeTitleTheme) {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.jvnet.substance.theme.SubstanceComplexTheme#getLightBackgroundColor()
			 */
			@Override
			public Color getLightBackgroundColor() {
				return SubstanceColorUtilities
						.getInterpolatedColor(defaultTheme.getColorScheme()
								.getUltraLightColor(), defaultTheme
								.getColorScheme().getExtraLightColor(), 0.5);

			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see org.jvnet.substance.theme.SubstanceTheme#getDisabledForegroundColor()
			 */
			@Override
			public Color getDisabledForegroundColor() {
				return SubstanceColorUtilities.getInterpolatedColor(
						defaultTheme.getColorScheme().getForegroundColor(),
						defaultTheme.getColorScheme().getLightColor(), 0.6f);
			}
		};
		theme
				.setNonActivePainter(new SimplisticSoftBorderReverseGradientPainter());
		theme.setSelectedTabFadeStart(0.1);
		theme.setSelectedTabFadeEnd(0.3);
		theme.registerComponentStateTheme(disabledTheme, 0.6f, false,
				ComponentState.DISABLED_UNSELECTED,
				ComponentState.DISABLED_SELECTED);

		this.theme = theme;
		// ((SubstanceComplexTheme) this.theme)
		// .setCellRendererBackgroundTheme(new SubstanceBlendBiTheme(
		// new SubstanceTerracottaTheme(),
		// new SubstanceSunGlareTheme(), 0.5).tint(0.2));
		// theme.addPaintAsActive(JScrollBar.class);
		shaper = new ClassicButtonShaper();
		watermark = new SubstanceNullWatermark();
		gradientPainter = new MatteGradientPainter();

		this.borderPainter = new ClassicInnerBorderPainter(0.8f,
				ColorShiftKind.TINT);

		MarbleNoiseHeaderPainter titlePainter = new MarbleNoiseHeaderPainter(
				false, false);
		titlePainter.setPaintingSeparators(true);
		titlePainter.setPaintingToolbarDropShadows(true);
		titlePainter.setTextureAlpha(0.7f);
		this.titlePainter = titlePainter;

		tabBackgroundComposite = new AlphaControlBackgroundComposite(0.75f);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.skin.SubstanceSkin#getDisplayName()
	 */
	public String getDisplayName() {
		return NAME;
	}
}
