/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.color;

import java.awt.Color;

import org.jvnet.substance.utils.SubstanceColorUtilities;

/**
 * Base class for color schemes simulating color-blind users.
 * 
 * @author Kirill Grouchnikov
 */
public class ColorBlindColorScheme extends BaseColorScheme {
	/**
	 * Matrix for converting RGB to LMS.
	 */
	public double[][] _rgbToLms = { { 0.05059983, 0.08585369, 0.00952420 },
			{ 0.01893033, 0.08925308, 0.01370054 },
			{ 0.00292202, 0.00975732, 0.07145979 } };

	/**
	 * Matrix for converting LMS to RGB.
	 */
	public double[][] _lmsToRgb = { { 30.830854, -29.832659, 1.610474 },
			{ -6.481468, 17.715578, -2.532642 },
			{ -0.375690, -1.199062, 14.273846 } };

	/**
	 * The main ultra-light color.
	 */
	private Color mainUltraLightColor;

	/**
	 * The main extra-light color.
	 */
	private Color mainExtraLightColor;

	/**
	 * The main light color.
	 */
	private Color mainLightColor;

	/**
	 * The main medium color.
	 */
	private Color mainMidColor;

	/**
	 * The main dark color.
	 */
	private Color mainDarkColor;

	/**
	 * The main ultra-dark color.
	 */
	private Color mainUltraDarkColor;

	/**
	 * The foreground color.
	 */
	private Color foregroundColor;

	/**
	 * The original color scheme.
	 */
	private ColorScheme origScheme;

	/**
	 * Blindness kind.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public enum BlindnessKind {
		/**
		 * Protanopia color blindness.
		 */
		PROTANOPIA,

		/**
		 * Deuteranopia color blindness.
		 */
		DEUTERANOPIA,

		/**
		 * Tritanopia color blindness.
		 */
		TRITANOPIA
	}

	/**
	 * Creates a new color scheme that simulates color-blindness.
	 * 
	 * @param origScheme
	 *            Original color scheme.
	 * @param kind
	 *            Color-blindness kind.
	 */
	public ColorBlindColorScheme(ColorScheme origScheme, BlindnessKind kind) {
		this.origScheme = origScheme;
		this.foregroundColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getForegroundColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainUltraDarkColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getUltraDarkColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainDarkColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getDarkColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainMidColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getMidColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainLightColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getLightColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainExtraLightColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getExtraLightColor(), _rgbToLms, kind, _lmsToRgb);
		this.mainUltraLightColor = SubstanceColorUtilities.getColorBlindColor(
				origScheme.getUltraLightColor(), _rgbToLms, kind, _lmsToRgb);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getForegroundColor()
	 */
	public Color getForegroundColor() {
		return this.foregroundColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getUltraLightColor()
	 */
	public Color getUltraLightColor() {
		return this.mainUltraLightColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getExtraLightColor()
	 */
	public Color getExtraLightColor() {
		return this.mainExtraLightColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getLightColor()
	 */
	public Color getLightColor() {
		return this.mainLightColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getMidColor()
	 */
	public Color getMidColor() {
		return this.mainMidColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getDarkColor()
	 */
	public Color getDarkColor() {
		return this.mainDarkColor;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.color.ColorScheme#getUltraDarkColor()
	 */
	public Color getUltraDarkColor() {
		return this.mainUltraDarkColor;
	}

	/**
	 * Returns the original color scheme.
	 * 
	 * @return The original color scheme.
	 */
	public ColorScheme getOrigScheme() {
		return this.origScheme;
	}
}
