/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashSet;
import java.util.Set;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicSpinnerUI;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.SubstanceConstants.Side;
import org.jvnet.substance.utils.icon.TransitionAwareIcon;

/**
 * UI for spinners in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceSpinnerUI extends BasicSpinnerUI {
	// /**
	// * Focus listener.
	// */
	// protected FocusBorderListener substanceFocusListener;

	/**
	 * Tracks changes to editor, removing the border as necessary.
	 */
	protected PropertyChangeListener substancePropertyChangeListener;

	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	/**
	 * The next (increment) button.
	 */
	protected SubstanceSpinnerButton nextButton;

	/**
	 * The previous (decrement) button.
	 */
	protected SubstanceSpinnerButton prevButton;

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return new SubstanceSpinnerUI();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicSpinnerUI#createNextButton()
	 */
	@Override
	protected Component createNextButton() {
		this.nextButton = new SubstanceSpinnerButton(this.spinner,
				SwingConstants.NORTH);
		this.nextButton.setFont(this.spinner.getFont());

		Icon icon = new TransitionAwareIcon(this.nextButton,
				new TransitionAwareIcon.Delegate() {
					public Icon getThemeIcon(SubstanceTheme theme) {
						return SubstanceImageCreator.getArrowIcon(spinner
								.getFont().getSize(), -2, -1, 0,
								SwingConstants.NORTH, theme);
					}
				});
		//
		// SubstanceImageCreator.getArrowIcon(SubstanceSizeUtils
		// .getArrowIconWidth() - 2, SubstanceSizeUtils
		// .getArrowIconHeight() - 1, SubstanceSizeUtils
		// .getArrowStrokeWidth(), SwingConstants.NORTH,
		// SubstanceCoreUtilities.getActiveTheme(this.spinner,
		// true));
		this.nextButton.setIcon(this.spinner, icon);

		int spinnerButtonSize = SubstanceSizeUtils
				.getSpinnerButtonWidth(spinner.getFont().getSize());
		this.nextButton.setPreferredSize(new Dimension(spinnerButtonSize,
				spinnerButtonSize));
		this.nextButton.setMinimumSize(new Dimension(5, 5));

		Set<Side> openSides = new HashSet<Side>();
		openSides.add(Side.BOTTOM);
		this.nextButton.putClientProperty(
				SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, openSides);

		this.installNextButtonListeners(this.nextButton);
		return this.nextButton;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicSpinnerUI#createPreviousButton()
	 */
	@Override
	protected Component createPreviousButton() {
		this.prevButton = new SubstanceSpinnerButton(this.spinner,
				SwingConstants.SOUTH);
		this.prevButton.setFont(this.spinner.getFont());

		Icon icon = new TransitionAwareIcon(this.prevButton,
				new TransitionAwareIcon.Delegate() {
					public Icon getThemeIcon(SubstanceTheme theme) {
						// System.out.println(spinner.getFont().getSize());
						return SubstanceImageCreator.getArrowIcon(spinner
								.getFont().getSize(), -2, -1, 0,
								SwingConstants.SOUTH, theme);
					}
				});
		this.prevButton.setIcon(this.spinner, icon);

		int spinnerButtonSize = SubstanceSizeUtils
				.getSpinnerButtonWidth(this.prevButton.getFont().getSize());
		this.prevButton.setPreferredSize(new Dimension(spinnerButtonSize,
				spinnerButtonSize));
		this.prevButton.setMinimumSize(new Dimension(5, 5));

		Set<Side> openSides = new HashSet<Side>();
		openSides.add(Side.TOP);
		this.prevButton.putClientProperty(
				SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY, openSides);

		this.installPreviousButtonListeners(this.prevButton);
		return this.prevButton;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicSpinnerUI#installDefaults()
	 */
	@Override
	protected void installDefaults() {
		super.installDefaults();
		JComponent editor = this.spinner.getEditor();
		if ((editor != null) && (editor instanceof JSpinner.DefaultEditor)) {
			JTextField tf = ((JSpinner.DefaultEditor) editor).getTextField();
			if (tf != null) {
				int fontSize = this.spinner.getFont().getSize();
				Insets ins = SubstanceSizeUtils
						.getSpinnerTextBorderInsets(fontSize);
				tf.setBorder(new EmptyBorder(ins.top, ins.left, ins.bottom,
						ins.right));
				tf.setFont(spinner.getFont());
			}
		}

		Border b = this.spinner.getBorder();
		if (b == null || b instanceof UIResource) {
			this.spinner.setBorder(// new
					// BorderUIResource.CompoundBorderUIResource(
					new SubstanceBorder(SubstanceSizeUtils
							.getSpinnerBorderInsets(this.spinner.getFont()
									.getSize())));
			// ,
			// new BasicBorders.MarginBorder()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicSpinnerUI#installListeners()
	 */
	@Override
	protected void installListeners() {
		super.installListeners();
		// this.substanceFocusListener = new FocusBorderListener(this.spinner);
		// this.spinner.addFocusListener(this.substanceFocusListener);

		this.substancePropertyChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if ("editor".equals(evt.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							if (substanceFadeStateListener != null) {
								substanceFadeStateListener
										.unregisterListeners();
							}
							if (spinner == null)
								return;
							JComponent editor = spinner.getEditor();
							if ((editor != null)
									&& (editor instanceof JSpinner.DefaultEditor)) {
								JTextField tf = ((JSpinner.DefaultEditor) editor)
										.getTextField();
								if (tf != null) {
									Insets ins = SubstanceSizeUtils
											.getSpinnerTextBorderInsets(spinner
													.getFont().getSize());
									tf.setBorder(new EmptyBorder(ins.top,
											ins.left, ins.bottom, ins.right));
									substanceFadeStateListener = new FadeStateListener(
											tf, null, new FadeTrackerAdapter() {
												@Override
												public void fadeEnded(
														FadeKind fadeKind) {
													if (spinner != null)
														spinner.repaint();
												}

												@Override
												public void fadePerformed(
														FadeKind fadeKind,
														float fadeCycle10) {
													if (spinner != null)
														spinner.repaint();
												}
											});
									substanceFadeStateListener
											.registerListeners();
								}
							}
						}
					});
				}

				if ("font".equals(evt.getPropertyName())) {
					SwingUtilities.invokeLater(new Runnable() {
						public void run() {
							spinner.updateUI();
						}
					});
				}
			}
		};
		this.spinner
				.addPropertyChangeListener(this.substancePropertyChangeListener);

		JComponent editor = spinner.getEditor();
		if ((editor != null) && (editor instanceof JSpinner.DefaultEditor)) {
			JTextField tf = ((JSpinner.DefaultEditor) editor).getTextField();
			this.substanceFadeStateListener = new FadeStateListener(tf, null,
					new FadeTrackerAdapter() {
						@Override
						public void fadeEnded(FadeKind fadeKind) {
							if (spinner != null)
								spinner.repaint();
						}

						@Override
						public void fadePerformed(FadeKind fadeKind,
								float fadeCycle10) {
							if (spinner != null)
								spinner.repaint();
						}
					});
			this.substanceFadeStateListener.registerListeners();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicSpinnerUI#uninstallListeners()
	 */
	@Override
	protected void uninstallListeners() {
		this.substanceFadeStateListener.unregisterListeners();
		this.substanceFadeStateListener = null;

		// this.substanceFocusListener.cancelAnimations();
		// this.spinner.removeFocusListener(this.substanceFocusListener);
		// this.substanceFocusListener = null;

		this.spinner
				.removePropertyChangeListener(this.substancePropertyChangeListener);
		this.substancePropertyChangeListener = null;

		super.uninstallListeners();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#paint(java.awt.Graphics,
	 *      javax.swing.JComponent)
	 */
	@Override
	public void paint(Graphics g, JComponent c) {
		super.paint(g, c);
		if (hasFocus(this.spinner)
				|| FadeTracker.getInstance().isTracked(this.spinner,
						FadeKind.FOCUS)) {
			this.paintFocus(g, this.spinner.getEditor().getBounds());
		}
	}

	/**
	 * Checks if a component or any of its children have focus.
	 * 
	 * @param comp
	 *            Component.
	 * @return <code>true</code> if the component of any of its children have
	 *         focus, <code>false</code> otherwise.
	 */
	private static boolean hasFocus(Component comp) {
		if (comp.hasFocus())
			return true;
		if (comp instanceof Container) {
			Container cont = (Container) comp;
			for (int i = 0; i < cont.getComponentCount(); i++) {
				Component child = cont.getComponent(i);
				if (hasFocus(child))
					return true;
			}
		}
		return false;
	}

	/**
	 * Paints the focus indication.
	 * 
	 * @param g
	 *            Graphics.
	 * @param bounds
	 *            Bounds for text.
	 */
	protected void paintFocus(Graphics g, Rectangle bounds) {
		// FadeTracker fadeTracker = FadeTracker.getInstance();
		JComponent editor = spinner.getEditor();
		if ((editor != null) && (editor instanceof JSpinner.DefaultEditor)) {
			JTextField tf = ((JSpinner.DefaultEditor) editor).getTextField();
			if (tf != null) {
				SubstanceCoreUtilities.paintFocus(g, this.spinner, tf, null,
						bounds, 0.4f, 2 + SubstanceSizeUtils
								.getExtraPadding(spinner.getFont().getSize()));
				//				
				// FocusKind focusKind = SubstanceCoreUtilities
				// .getFocusKind(this.spinner);
				// if ((focusKind == FocusKind.NONE)
				// && (!fadeTracker.isTracked(tf, FadeKind.FOCUS)))
				// return;
				//
				// Graphics2D graphics = (Graphics2D) g.create();
				// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				// RenderingHints.VALUE_ANTIALIAS_ON);
				//
				// float alpha = 1.0f;
				// if (fadeTracker.isTracked(tf, FadeKind.FOCUS)) {
				// alpha = fadeTracker.getFade10(tf, FadeKind.FOCUS) / 10.f;
				// }
				// graphics.setComposite(TransitionLayout.getAlphaComposite(
				// this.spinner, alpha));
				//
				// Color color = SubstanceColorUtilities
				// .getFocusColor(this.spinner);
				// graphics.setColor(color);
				// focusKind.paintFocus(this.spinner, tf, graphics, bounds);
				// graphics.dispose();
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#getPreferredSize(javax.swing.JComponent)
	 */
	@Override
	public Dimension getPreferredSize(JComponent c) {
		Dimension nextD = this.nextButton.getPreferredSize();
		Dimension previousD = this.prevButton.getPreferredSize();
		Dimension editorD = spinner.getEditor().getPreferredSize();

		Dimension size = new Dimension(editorD.width, editorD.height);
		size.width += Math.max(nextD.width, previousD.width);
		Insets insets = this.spinner.getInsets();
		size.width += insets.left + insets.right;
		size.height += insets.top + insets.bottom;
		return size;
	}
}
