/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashMap;
import java.util.Map;

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicButtonListener;
import javax.swing.plaf.basic.BasicRadioButtonUI;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.*;

/**
 * UI for radio buttons in <b>Substance </b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceRadioButtonUI extends BasicRadioButtonUI {
	// /**
	// * Default radio button dimension.
	// */
	// private static final int DIMENSION = 11;

	/**
	 * Background delegate.
	 */
	private SubstanceFillBackgroundDelegate bgDelegate;

	/**
	 * Property change listener. Listens on changes to
	 * {@link AbstractButton#MODEL_CHANGED_PROPERTY} property.
	 */
	protected PropertyChangeListener substancePropertyListener;

	/**
	 * Associated toggle button.
	 */
	protected JToggleButton button;

	/**
	 * Icons for all component states
	 */
	private static Map<String, Icon> icons = new HashMap<String, Icon>();

	/**
	 * Listener for fade animations.
	 */
	protected FadeStateListener substanceFadeStateListener;

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(org.jvnet.substance.theme.SubstanceTheme)
	 */
	public static synchronized void reset() {
		SubstanceRadioButtonUI.icons.clear();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#installListeners(javax.swing.AbstractButton)
	 */
	@Override
	protected void installListeners(final AbstractButton b) {
		super.installListeners(b);

		substanceFadeStateListener = new FadeStateListener(b, b.getModel(),
				SubstanceCoreUtilities.getFadeCallback(b));
		substanceFadeStateListener.registerListeners();

		substancePropertyListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if (AbstractButton.MODEL_CHANGED_PROPERTY.equals(evt
						.getPropertyName())) {
					if (substanceFadeStateListener != null)
						substanceFadeStateListener.unregisterListeners();
					substanceFadeStateListener = new FadeStateListener(b, b
							.getModel(), SubstanceCoreUtilities
							.getFadeCallback(b));
					substanceFadeStateListener.registerListeners();
				}
				if ("opaque".equals(evt.getPropertyName())) {
					if (!Boolean.TRUE.equals(b
							.getClientProperty(SubstanceButtonUI.LOCK_OPACITY))) {
						b.putClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL,
								evt.getNewValue());
						// System.out
						// .println("PCL: "
						// + b.getText()
						// + "->"
						// + b
						// .getClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL));
					}
				}
			}
		};
		b.addPropertyChangeListener(substancePropertyListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicRadioButtonUI#installDefaults(javax.swing.AbstractButton)
	 */
	@Override
	protected void installDefaults(AbstractButton b) {
		super.installDefaults(b);
		Border border = b.getBorder();
		if (border == null || border instanceof UIResource) {
			b.setBorder(SubstanceSizeUtils
					.getRadioButtonBorder(SubstanceSizeUtils
							.getComponentFontSize(b)));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicRadioButtonUI#uninstallDefaults(javax.swing.AbstractButton)
	 */
	@Override
	protected void uninstallDefaults(AbstractButton b) {
		super.uninstallDefaults(b);

		// b.setOpaque((Boolean) b
		// .getClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL));
		// b.putClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL, null);
	}

	@Override
	public void installUI(JComponent c) {
		// c.putClientProperty(SubstanceButtonUI.LOCK_OPACITY, null);
		super.installUI(c);
	}

	@Override
	public void uninstallUI(JComponent c) {
		// c.setOpaque((Boolean) c
		// .getClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL));
		// c.putClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL, null);
		super.uninstallUI(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#uninstallListeners(javax.swing.AbstractButton)
	 */
	@Override
	protected void uninstallListeners(AbstractButton b) {
		substanceFadeStateListener.unregisterListeners();
		substanceFadeStateListener = null;

		b.removePropertyChangeListener(substancePropertyListener);
		substancePropertyListener = null;

		super.uninstallListeners(b);
	}

	/**
	 * Returns the icon that matches the current and previous states of the
	 * radio button.
	 * 
	 * @param button
	 *            Button (should be {@link JRadioButton}).
	 * @param currState
	 *            Current state of the checkbox.
	 * @param prevState
	 *            Previous state of the checkbox.
	 * @return Matching icon.
	 */
	private static synchronized Icon getIcon(JToggleButton button,
			ComponentState currState, ComponentState prevState) {
		// check if fading
		// FadeTracker fadeTracker = FadeTracker.getInstance();
		float visibility = currState.isKindActive(FadeKind.SELECTION) ? 10 : 0;

		SubstanceTheme theme = SubstanceCoreUtilities.getTheme(button,
				currState, true, true);
		SubstanceTheme theme2 = SubstanceCoreUtilities.getTheme(button,
				prevState, true, true);
		float cyclePos = 0;

		FadeState fadeState = SubstanceFadeUtilities.getFadeState(button,
				FadeKind.SELECTION, FadeKind.ROLLOVER, FadeKind.PRESS);
		if (fadeState != null) {
			cyclePos = fadeState.getFadePosition();
			if (fadeState.isFadingIn())
				cyclePos = 10 - cyclePos;
			if (fadeState.fadeKind == FadeKind.SELECTION) {
				visibility = fadeState.getFadePosition();
			}
		}

		int checkMarkSize = // SubstanceSizeUtils
		// .getRadioButtonMarkSize(SubstanceSizeUtils.getControlFontSize());
		// if (!(button.getFont() instanceof UIResource))
		// checkMarkSize =
		SubstanceSizeUtils.getRadioButtonMarkSize(button.getFont().getSize());
		String key = checkMarkSize + ":" + currState.name() + ":"
				+ currState.name() + ":" + theme.getDisplayName() + ":"
				+ theme2.getDisplayName() + ":" + cyclePos + ":" + visibility;

		Icon result = SubstanceRadioButtonUI.icons.get(key);
		if (result != null)
			return result;
		result = new ImageIcon(SubstanceImageCreator.getRadioButton(button,
				checkMarkSize, currState, 0, theme, theme2, cyclePos,
				visibility / 10.f));
		SubstanceRadioButtonUI.icons.put(key, result);
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent b) {
		return new SubstanceRadioButtonUI((JToggleButton) b);
	}

	/**
	 * Simple constructor.
	 * 
	 * @param button
	 *            Associated radio button.
	 */
	public SubstanceRadioButtonUI(JToggleButton button) {
		bgDelegate = new SubstanceFillBackgroundDelegate();
		this.button = button;
		button.setRolloverEnabled(true);
		// button.setOpaque(false);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#createButtonListener(javax.swing.AbstractButton)
	 */
	@Override
	protected BasicButtonListener createButtonListener(AbstractButton b) {
		return new RolloverButtonListener(b);
		// return RolloverButtonListener.getListener(b);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicRadioButtonUI#getDefaultIcon()
	 */
	@Override
	public Icon getDefaultIcon() {
		ButtonModel model = button.getModel();
		ComponentState currState = ComponentState.getState(model, button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);
		return SubstanceRadioButtonUI.getIcon(button, currState, prevState);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#paint(java.awt.Graphics,
	 *      javax.swing.JComponent)
	 */
	@Override
	public void update(Graphics g, JComponent c) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel))
			return;
		// important to synchronize on the button as we are
		// about to fiddle with its opaqueness
		// synchronized (c) {
		// // this.bgDelegate.update(g, c);
		// // remove opacity
		boolean isOpaque = c.isOpaque();
		c.putClientProperty(SubstanceButtonUI.LOCK_OPACITY, Boolean.TRUE);
		c.setOpaque(false);

		// boolean hasCustomBackground = !(c.getBackground() instanceof
		// UIResource);
		// if (hasCustomBackground
		// && (Boolean.TRUE.equals(c
		// .getClientProperty(SubstanceButtonUI.OPACITY_ORIGINAL)))) {
		if (isOpaque || TransitionLayout.isOpaque(c)) {
			bgDelegate.update(g, c);
		}
		// } else {
		// bgDelegate.updateIfOpaque(g, c);
		// }
		super.paint(g, c);

		c.setOpaque(isOpaque);

		c.putClientProperty(SubstanceButtonUI.LOCK_OPACITY, null);
		// // restore opacity
		// c.setOpaque(isOpaque);
		// }

		// Some ugly hack to allow fade-out of focus ring. The code
		// in BasicRadioButtonUI doesn't call paintFocus() at all
		// when the component is not focus owner.
		AbstractButton b = (AbstractButton) c;
		if (!b.isFocusPainted())
			return;

		FontMetrics fm = c.getFontMetrics(c.getFont());

		Insets i = c.getInsets();
		Dimension size = new Dimension();
		Rectangle viewRect = new Rectangle();
		Rectangle iconRect = new Rectangle();
		Rectangle textRect = new Rectangle();

		size = b.getSize(size);
		viewRect.x = i.left;
		viewRect.y = i.top;
		viewRect.width = size.width - (i.right + viewRect.x);
		viewRect.height = size.height - (i.bottom + viewRect.y);
		iconRect.x = iconRect.y = iconRect.width = iconRect.height = 0;
		textRect.x = textRect.y = textRect.width = textRect.height = 0;

		Icon altIcon = b.getIcon();

		String text = SwingUtilities.layoutCompoundLabel(c, fm, b.getText(),
				altIcon != null ? altIcon : getDefaultIcon(), b
						.getVerticalAlignment(), b.getHorizontalAlignment(), b
						.getVerticalTextPosition(), b
						.getHorizontalTextPosition(), viewRect, iconRect,
				textRect, b.getText() == null ? 0 : b.getIconTextGap());

		if ((text != null) && (textRect.width > 0) && (textRect.height > 0)) {
			if (!(b.hasFocus() && b.isFocusPainted())) {
				if (FadeTracker.getInstance().isTracked(c, FadeKind.FOCUS))
					this.paintFocus(g, textRect, size);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicRadioButtonUI#paintFocus(java.awt.Graphics,
	 *      java.awt.Rectangle, java.awt.Dimension)
	 */
	@Override
	protected void paintFocus(Graphics g, Rectangle t, Dimension d) {
		// System.out.println(button.getText() + " -> focus");
		SubstanceCoreUtilities.paintFocus(g, button, button, null, t, 1.0f, 1);
		// FadeTracker fadeTracker = FadeTracker.getInstance();
		// FocusKind focusKind =
		// SubstanceCoreUtilities.getFocusKind(this.button);
		// if ((focusKind == FocusKind.NONE)
		// && (!fadeTracker.isTracked(this.button, FadeKind.FOCUS)))
		// return;
		// Graphics2D graphics = (Graphics2D) g.create();
		// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
		// RenderingHints.VALUE_ANTIALIAS_ON);
		//
		// float alpha = 1.0f;
		// if (fadeTracker.isTracked(this.button, FadeKind.FOCUS)) {
		// alpha = fadeTracker.getFade10(this.button, FadeKind.FOCUS) / 10.f;
		// }
		// // System.out.println(button.getText() + " -> " + alpha);
		// graphics.setComposite(TransitionLayout.getAlphaComposite(this.button,
		// alpha));
		//
		// Color color = SubstanceColorUtilities.getFocusColor(this.button);
		// graphics.setColor(color);
		// focusKind.paintFocus(this.button, graphics, t);
		// graphics.dispose();
	}

	/**
	 * Returns memory usage string.
	 * 
	 * @return Memory usage string.
	 */
	public static String getMemoryUsage() {
		StringBuffer sb = new StringBuffer();
		sb.append("SubstanceRadioButtonUI: \n");
		sb.append("\t" + SubstanceRadioButtonUI.icons.size() + " icons");
		return sb.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicButtonUI#paintText(java.awt.Graphics,
	 *      javax.swing.JComponent, java.awt.Rectangle, java.lang.String)
	 */
	@Override
	protected void paintText(Graphics g, JComponent c, Rectangle textRect,
			String text) {
		SubstanceCoreUtilities.paintText(g, (AbstractButton) c, textRect, text,
				getTextShiftOffset(), getPropertyPrefix());
	}
}
