/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.awt.image.BufferedImage;
import java.util.HashMap;
import java.util.Map;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicProgressBarUI;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeTracker;
import org.jvnet.substance.border.InnerDelegateBorderPainter;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.BaseButtonShaper;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.painter.SubstanceGradientPainter;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;
import org.jvnet.substance.utils.SubstanceSizeUtils;

/**
 * UI for progress bars in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceProgressBarUI extends BasicProgressBarUI {
	/**
	 * Hash for computed stripe images.
	 */
	private static Map<String, BufferedImage> stripeMap = new HashMap<String, BufferedImage>();

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(SubstanceTheme)
	 */
	public static synchronized void reset() {
		SubstanceProgressBarUI.stripeMap.clear();
	}

	/**
	 * The current state of the indeterminate animation's cycle. 0, the initial
	 * value, means paint the first frame. When the progress bar is
	 * indeterminate and showing, the default animation thread updates this
	 * variable by invoking incrementAnimationIndex() every repaintInterval
	 * milliseconds.
	 */
	private float animationIndex;

	/**
	 * Value change listener on the associated progress bar.
	 */
	protected ChangeListener substanceValueChangeListener;

	/**
	 * The speed factor for the indeterminate progress bars.
	 */
	protected float speed;

	/**
	 * Fade kind for the progress bar value change.
	 */
	public static final FadeKind PROGRESS_BAR_VALUE_CHANGED = new FadeKind(
			"substancelaf.progressBarValueChanged");

	// static {
	// FadeConfigurationManager.getInstance().allowFades(PROGRESS_BAR_VALUE_CHANGED);
	// }

	/**
	 * Property name for storing the <code>from</code> value on progress bar
	 * value animation. Is for internal use only.
	 */
	private static final String PROGRESS_BAR_FROM = "substancelaf.internal.from";

	/**
	 * Property name for storing the <code>to</code> value on progress bar
	 * value animation. Is for internal use only.
	 */
	private static final String PROGRESS_BAR_TO = "substancelaf.internal.to";

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return new SubstanceProgressBarUI();
	}

	@Override
	protected void installDefaults() {
		super.installDefaults();

		progressBar.putClientProperty(PROGRESS_BAR_TO, progressBar.getValue());
		LookAndFeel.installProperty(progressBar, "opaque", Boolean.FALSE);

		this.speed = (20.0f * UIManager.getInt("ProgressBar.repaintInterval"))
				/ UIManager.getInt("ProgressBar.cycleTime");
	}

	@Override
	protected void installListeners() {
		super.installListeners();

		final BoundedRangeModel model = progressBar.getModel();
		if (model instanceof DefaultBoundedRangeModel) {
			substanceValueChangeListener = new ChangeListener() {
				public void stateChanged(ChangeEvent e) {
					int oldValue = (Integer) progressBar
							.getClientProperty(PROGRESS_BAR_TO);
					int currValue = model.getValue();
					int span = model.getMaximum() - model.getMinimum();
					Insets b = progressBar.getInsets(); // area for border
					int barRectWidth = progressBar.getWidth()
							- (b.right + b.left);
					int barRectHeight = progressBar.getHeight()
							- (b.top + b.bottom);
					int totalPixels = (progressBar.getOrientation() == JProgressBar.HORIZONTAL) ? barRectWidth
							: barRectHeight;
					// fix for defect 223 (min and max on the model are the
					// same).
					int pixelDelta = (span <= 0) ? 0 : (currValue - oldValue)
							* totalPixels / span;

					FadeTracker fadeTracker = FadeTracker.getInstance();
					if (!fadeTracker.isTracked(progressBar,
							PROGRESS_BAR_VALUE_CHANGED)) {
						progressBar.putClientProperty(PROGRESS_BAR_FROM,
								progressBar.getClientProperty(PROGRESS_BAR_TO));
					}
					progressBar.putClientProperty(PROGRESS_BAR_TO, progressBar
							.getValue());
					if (Math.abs(pixelDelta) > 5) {
						FadeTracker.getInstance().trackFadeIn(
								PROGRESS_BAR_VALUE_CHANGED, progressBar, false,
								null);
					}
				}
			};
			((DefaultBoundedRangeModel) model)
					.addChangeListener(substanceValueChangeListener);
		}
	}

	@Override
	protected void uninstallListeners() {
		BoundedRangeModel model = progressBar.getModel();
		if (model instanceof DefaultBoundedRangeModel) {
			((DefaultBoundedRangeModel) model)
					.removeChangeListener(substanceValueChangeListener);
		}

		super.uninstallListeners();
	}

	/**
	 * Retrieves stripe image.
	 * 
	 * @param baseSize
	 *            Stripe base in pixels.
	 * @param isRotated
	 *            if <code>true</code>, the resulting stripe image will be
	 *            rotated.
	 * @param colorScheme
	 *            Color scheme to paint the stripe image.
	 * @return Stripe image.
	 */
	private static synchronized BufferedImage getStripe(int baseSize,
			boolean isRotated, ColorScheme colorScheme) {
		String key = "" + baseSize + ":" + isRotated + ":"
				+ SubstanceCoreUtilities.getSchemeId(colorScheme);
		BufferedImage result = SubstanceProgressBarUI.stripeMap.get(key);
		if (result == null) {
			result = SubstanceImageCreator.getStripe(baseSize, colorScheme
					.getUltraLightColor());
			if (isRotated) {
				result = SubstanceImageCreator.getRotated(result, 1);
			}
			SubstanceProgressBarUI.stripeMap.put(key, result);
		}
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#paintDeterminate(java.awt.Graphics,
	 *      javax.swing.JComponent)
	 */
	@Override
	public void paintDeterminate(Graphics g, JComponent c) {
		if (!(g instanceof Graphics2D)) {
			return;
		}

		Insets insets = progressBar.getInsets();
		insets.top /= 2;
		insets.left /= 2;
		insets.bottom /= 2;
		insets.right /= 2;
		int barRectWidth = progressBar.getWidth()
				- (insets.right + insets.left);
		int barRectHeight = progressBar.getHeight()
				- (insets.top + insets.bottom);

		// amount of progress to draw
		int amountFull = getAmountFull(insets, barRectWidth, barRectHeight);

		Graphics2D graphics = (Graphics2D) g;

		SubstanceTheme bgTheme = SubstanceCoreUtilities.getTheme(progressBar,
				true);
		if (progressBar.isEnabled())
			bgTheme = bgTheme.getDefaultTheme();
		else
			bgTheme = bgTheme.getDisabledTheme();
		SubstanceGradientPainter gp = SubstanceCoreUtilities
				.getGradientPainter(this.progressBar);
		// background
		if (progressBar.getOrientation() == SwingConstants.HORIZONTAL) {
			Shape contour = BaseButtonShaper.getBaseOutline(barRectWidth + 1,
					barRectHeight, 0, null);
			BufferedImage bg = gp.getContourBackground(barRectWidth + 1,
					barRectHeight, contour, false, bgTheme.getColorScheme(),
					bgTheme.getColorScheme(), 0, true, false);
			graphics.drawImage(bg, insets.left, insets.top, null);
			// SubstanceImageCreator.paintRectangularStripedBackground(graphics,
			// b.left, b.top, barRectWidth, barRectHeight + 1, bgTheme
			// .getColorScheme(), null, 0, false);
		} else { // VERTICAL
			Shape contour = BaseButtonShaper.getBaseOutline(barRectHeight + 1,
					barRectWidth, 0, null);
			BufferedImage bg = gp.getContourBackground(barRectHeight + 1,
					barRectWidth, contour, false, bgTheme.getColorScheme(),
					bgTheme.getColorScheme(), 0, true, false);
			graphics.drawImage(SubstanceImageCreator.getRotated(bg, 3),
					insets.left, insets.top, null);
			//
			// SubstanceImageCreator.paintRectangularStripedBackground(graphics,
			// b.left, b.top, barRectWidth + 1, barRectHeight, bgTheme
			// .getColorScheme(), null, 0, true);
		}

		if (amountFull > 0) {
			float borderStrokeWidth = SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(this.progressBar));
			SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
					.getBorderPainter(null, null);
			int borderDelta = (borderPainter instanceof InnerDelegateBorderPainter) ? (int) (Math
					.ceil(2.0 * borderStrokeWidth)) / 2
					: (int) (borderStrokeWidth / 2.0);

			ColorScheme fillColorScheme = progressBar.isEnabled() ? SubstanceCoreUtilities
					.getActiveScheme(progressBar)
					: SubstanceCoreUtilities.getDisabledScheme(progressBar);
			if (progressBar.getOrientation() == SwingConstants.HORIZONTAL) {
				int barWidth = amountFull - 2 * borderDelta;
				int barHeight = barRectHeight - 2 * borderDelta;
				if ((barWidth > 0) && (barHeight > 0)) {
					if (c.getComponentOrientation().isLeftToRight()) {
						SubstanceImageCreator.paintRectangularBackground(g,
								insets.left + borderDelta, insets.top
										+ borderDelta, barWidth, barHeight,
								fillColorScheme, 0.6f, false);
					} else {
						// fix for RTL determinate horizontal progress bar in
						// 2.3
						SubstanceImageCreator.paintRectangularBackground(g,
								insets.left + barRectWidth - amountFull - 2
										* borderDelta,
								insets.top + borderDelta, barWidth, barHeight,
								fillColorScheme, 0.6f, false);
					}
				}
			} else { // VERTICAL
				int barWidth = progressBar.getHeight() - insets.bottom
						- amountFull - 2 * borderDelta;
				int barHeight = barRectWidth - 2 * borderDelta;
				if ((barWidth > 0) && (barHeight > 0)) {
					// fix for issue 95. Vertical bar is growing from the bottom
					SubstanceImageCreator.paintRectangularBackground(g,
							insets.left + borderDelta, barWidth + borderDelta,
							barHeight, amountFull, fillColorScheme, 0.6f, true);
				}
			}
		}

		// Deal with possible text painting
		if (progressBar.isStringPainted()) {
			this.paintString(g, insets.left, insets.top, barRectWidth,
					barRectHeight, amountFull, insets);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#getSelectionBackground()
	 */
	@Override
	protected Color getSelectionBackground() {
		SubstanceTheme bgTheme = SubstanceCoreUtilities.getTheme(progressBar,
				true);
		if (progressBar.isEnabled())
			bgTheme = bgTheme.getDefaultTheme();
		else
			bgTheme = bgTheme.getDisabledTheme();
		return bgTheme.getForegroundColor();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#getSelectionForeground()
	 */
	@Override
	protected Color getSelectionForeground() {
		ColorScheme fillColorScheme = progressBar.isEnabled() ? SubstanceCoreUtilities
				.getActiveScheme(progressBar)
				: SubstanceCoreUtilities.getDefaultScheme(progressBar);
		return fillColorScheme.getForegroundColor();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#paintIndeterminate(java.awt.Graphics,
	 *      javax.swing.JComponent)
	 */
	@Override
	public void paintIndeterminate(Graphics g, JComponent c) {
		if (!(g instanceof Graphics2D)) {
			return;
		}

		Insets b = progressBar.getInsets(); // area for border
		int barRectWidth = progressBar.getWidth() - (b.right + b.left);
		int barRectHeight = progressBar.getHeight() - (b.top + b.bottom);

		int valComplete = (int) animationIndex;// int)
		// (progressBar.getPercentComplete()
		// * barRectWidth);
		Graphics2D graphics = (Graphics2D) g;

		// background
		if (progressBar.getOrientation() == SwingConstants.HORIZONTAL) {
			// SubstanceImageCreator.paintRectangularStripedBackground(graphics,
			// b.left, b.top, barRectWidth, barRectHeight + 1,
			// SubstanceCoreUtilities.getDisabledScheme(progressBar),
			// null, 0, false);
		} else { // VERTICAL
		// SubstanceImageCreator.paintRectangularStripedBackground(graphics,
		// b.left, b.top, barRectWidth + 1, barRectHeight,
		// SubstanceCoreUtilities.getDisabledScheme(progressBar),
		// null, 0, 0.6f, true);
		}

		ColorScheme fillColorScheme = progressBar.isEnabled() ? SubstanceCoreUtilities
				.getActiveScheme(progressBar)
				: SubstanceCoreUtilities.getDefaultScheme(progressBar);
		if (progressBar.getOrientation() == SwingConstants.HORIZONTAL) {
			SubstanceImageCreator.paintRectangularStripedBackground(graphics,
					b.left, b.top, barRectWidth, barRectHeight,
					fillColorScheme, SubstanceProgressBarUI.getStripe(
							barRectHeight, false, fillColorScheme),
					valComplete, 0.6f, false);
		} else { // VERTICAL
			// fix for issue 95. Vertical progress bar rises from the bottom.
			SubstanceImageCreator.paintRectangularStripedBackground(graphics,
					b.left, b.top, barRectWidth, barRectHeight,
					fillColorScheme, SubstanceProgressBarUI.getStripe(
							barRectWidth, true, fillColorScheme), 2
							* barRectWidth - valComplete, 0.6f, true);
		}

		// Deal with possible text painting
		if (progressBar.isStringPainted()) {
			this.paintString(g, b.left, b.top, barRectWidth, barRectHeight,
					barRectWidth, b);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#getBox(java.awt.Rectangle)
	 */
	@Override
	protected Rectangle getBox(Rectangle r) {
		Insets b = progressBar.getInsets(); // area for border
		int barRectWidth = progressBar.getWidth() - (b.right + b.left);
		int barRectHeight = progressBar.getHeight() - (b.top + b.bottom);
		return new Rectangle(b.left, b.top, barRectWidth, barRectHeight);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#incrementAnimationIndex()
	 */
	@Override
	protected void incrementAnimationIndex() {
		float newValue = animationIndex + this.speed;

		Insets b = progressBar.getInsets(); // area for border
		int barRectHeight = progressBar.getHeight() - (b.top + b.bottom);
		int barRectWidth = progressBar.getWidth() - (b.right + b.left);
		int threshold = 0;
		if (progressBar.getOrientation() == SwingConstants.HORIZONTAL) {
			threshold = 2 * barRectHeight + 1;
		} else {
			threshold = 2 * barRectWidth + 1;
		}
		animationIndex = newValue % threshold;
		progressBar.repaint();
	}

	/**
	 * Returns the memory usage string.
	 * 
	 * @return The memory usage string.
	 */
	public static String getMemoryUsage() {
		StringBuffer sb = new StringBuffer();
		sb.append("SubstanceProgressBarUI: \n");
		sb.append("\t" + SubstanceProgressBarUI.stripeMap.size() + " stripes");
		return sb.toString();
	}

	@Override
	protected int getAmountFull(Insets b, int width, int height) {
		int amountFull = 0;
		BoundedRangeModel model = progressBar.getModel();

		long span = model.getMaximum() - model.getMinimum();
		double currentValue = model.getValue();
		FadeTracker fadeTracker = FadeTracker.getInstance();
		if (fadeTracker.isTracked(progressBar, PROGRESS_BAR_VALUE_CHANGED)) {
			double fade10 = fadeTracker.getFade10(progressBar,
					PROGRESS_BAR_VALUE_CHANGED);
			int from = (Integer) progressBar
					.getClientProperty(PROGRESS_BAR_FROM);
			int to = (Integer) progressBar.getClientProperty(PROGRESS_BAR_TO);
			currentValue = from + (int) (fade10 * (to - from) / 10.0);
		}

		double percentComplete = (currentValue - model.getMinimum()) / span;

		if ((model.getMaximum() - model.getMinimum()) != 0) {
			if (progressBar.getOrientation() == JProgressBar.HORIZONTAL) {
				amountFull = (int) Math.round(width * percentComplete);
			} else {
				amountFull = (int) Math.round(height * percentComplete);
			}
		}
		return amountFull;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#getPreferredInnerHorizontal()
	 */
	@Override
	protected Dimension getPreferredInnerHorizontal() {
		int size = SubstanceSizeUtils.getComponentFontSize(this.progressBar);
		size += 2 * SubstanceSizeUtils.getAdjustedSize(size, 1, 4, 1, false);
		return new Dimension(146 + SubstanceSizeUtils.getAdjustedSize(size, 0,
				1, 10, false), size);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicProgressBarUI#getPreferredInnerVertical()
	 */
	@Override
	protected Dimension getPreferredInnerVertical() {
		int size = SubstanceSizeUtils.getComponentFontSize(this.progressBar);
		size += 2 * SubstanceSizeUtils.getAdjustedSize(size, 1, 4, 1, false);
		return new Dimension(size, 146 + SubstanceSizeUtils.getAdjustedSize(
				size, 0, 1, 10, false));
	}
}
