

#include "replaygainscanner.h"
#include "replaygain_plugins.h"
#include "combobutton.h"
#include "tools.h"

#include <qlayout.h>
#include <qstringlist.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qprogressbar.h>
#include <qdir.h>
#include <qevent.h>
#include <qdragobject.h>
#include <qcheckbox.h>
#include <qtooltip.h>

#include <klocale.h>
#include <klistview.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <kdirlister.h>
#include <kpushbutton.h>


ReplayGainScanner::ReplayGainScanner( QWidget *parent, const char *name, bool modal, WFlags f )
    : QDialog( parent, name, modal, f )
{
    setAcceptDrops(true);

    setCaption( QString(i18n("ReplayGain Tool")).append(" - soundKonverter") );
    resize(600,500);
    setIcon(KGlobal::iconLoader()->loadIcon("soundkonverter_replaygain",KIcon::Small));

    QGridLayout *grid=new QGridLayout(this,4,1,10,5);

    QHBoxLayout *filterBox = new QHBoxLayout();
    grid->addLayout(filterBox,0,0);

    cAdd=new ComboButton(this,"cAdd");
    cAdd->insertItem(KGlobal::iconLoader()->loadIcon("folder",KIcon::Small),i18n("Add folder..."));
    cAdd->insertItem(KGlobal::iconLoader()->loadIcon("sound",KIcon::Small),i18n("Add files..."));
    filterBox->addWidget(cAdd);
    connect(cAdd,SIGNAL(clicked(int)),this,SLOT(addClicked(int)));

    pScanStatus=new QProgressBar(this,"pScanStatus");
    pScanStatus->setMaximumWidth(75);
    //pScanStatus->setEnabled(false);
    filterBox->addWidget(pScanStatus);

    filterBox->addStretch();

    cForce=new QCheckBox(i18n("Force recalculation"),this,"cForce");
    QToolTip::add( cForce, i18n("Recalculate ReplayGain tag for files that already have a ReplayGain tag set.") );
    filterBox->addWidget(cForce);

    /*QLabel *lFilter=new QLabel(i18n("Show:"),this,"lFilter");
    filterBox->addWidget(lFilter);

    cFilter=new QComboBox(this,"cFilter");
    cFilter->insertItem(i18n("All"));
    cFilter->insertItem(i18n("With ReplayGain"));
    cFilter->insertItem(i18n("Without ReplayGain"));
    cFilter->insertItem(i18n("Unknown"));
    filterBox->addWidget(cFilter);*/

    lList=new KListView(this,"lList");
    lList->addColumn( i18n("ReplayGain") );
    lList->addColumn( i18n("File") );
    lList->setSelectionMode( QListView::Extended );
    lList->setAllColumnsShowFocus(true);
    grid->addWidget(lList,1,0);

    QHBoxLayout *progressBox = new QHBoxLayout();
    grid->addLayout(progressBox,2,0);

    pProgressBar=new QProgressBar(this,"pProgressBar");
    progressBox->addWidget(pProgressBar);

    pCancel=new KPushButton(KGlobal::iconLoader()->loadIcon("cancel",KIcon::Small),i18n("Cancel"),this,"pCancel");
    pCancel->setEnabled(false);
    progressBox->addWidget(pCancel);
    connect(pCancel,SIGNAL(clicked()),this,SLOT(cancel()));

    QHBoxLayout *buttonBox = new QHBoxLayout();
    grid->addLayout(buttonBox,3,0);

    //pScanUnknown=new KPushButton(KGlobal::iconLoader()->loadIcon("reload",KIcon::Small),i18n("Scan unknown"),this,"pScanUnknown");
    //buttonBox->addWidget(pScanUnknown);

    pTagVisible=new KPushButton(KGlobal::iconLoader()->loadIcon("apply",KIcon::Small),i18n("Tag untagged"),this,"pTagVisible");
    QToolTip::add( pTagVisible, i18n("Calculate ReplayGain tag for all files in the file list without ReplayGain tag.") );
    buttonBox->addWidget(pTagVisible);
    connect(pTagVisible,SIGNAL(clicked()),this,SLOT(applyReplayGain()));

    pRemoveTag=new KPushButton(KGlobal::iconLoader()->loadIcon("cancel",KIcon::Small),i18n("Untag tagged"),this,"pRemoveTag");
    QToolTip::add( pRemoveTag, i18n("Remove the ReplayGain tag from all files in the file list.") );
    buttonBox->addWidget(pRemoveTag);
    connect(pRemoveTag,SIGNAL(clicked()),this,SLOT(removeReplayGain()));

    pRemoveFiles=new KPushButton(KGlobal::iconLoader()->loadIcon("fileclose",KIcon::Small),i18n("Remove Selected"),this,"pRemoveFiles");
    QToolTip::add( pRemoveFiles, i18n("Remove all selected files from the file list.") );
    buttonBox->addWidget(pRemoveFiles);
    connect(pRemoveFiles,SIGNAL(clicked()),this,SLOT(removeFiles()));

    buttonBox->addStretch();

    pOk=new KPushButton(KGlobal::iconLoader()->loadIcon("exit",KIcon::Small),i18n("Close"),this,"pOk");
    pOk->setFocus();
    buttonBox->addWidget(pOk);
    connect(pOk,SIGNAL(clicked()),this,SLOT(accept()));

    dirLister= new KDirLister();
    dirLister->setAutoUpdate(false);
    connect(dirLister,SIGNAL(newItems(const KFileItemList&)),this,SLOT(allFilesListed(const KFileItemList&)));
    connect(dirLister,SIGNAL(percent(int)),this,SLOT(scanStatus(int)));
    //connect(dirLister,SIGNAL(completed()),this,SLOT(scanCompleted()));


    connect(&pProcess,SIGNAL(processExited(KProcess*)),this,SLOT(processExited(KProcess*)));
}

ReplayGainScanner::~ReplayGainScanner()
{
}

void ReplayGainScanner::addClicked(int index)
{
    if(index == 1) {
        showFileDialogue();
    }
    else {
        showDirDialogue();
    }
}

void ReplayGainScanner::showFileDialogue()
{
    pScanStatus->hide();

    KFileDialog *fileDialog;
    fileDialog = new KFileDialog( QDir::homeDirPath(), tools.sFileOpenStringReplayGain, this, i18n("Choose files!"), true );
    fileDialog->setMode ( KFile::Files | KFile::ExistingOnly | KFile::LocalOnly );
    if( fileDialog->exec() == KDialog::Accepted ) {
        QStringList files=fileDialog->selectedFiles();
        for( QStringList::Iterator it=files.begin(); it!=files.end(); ++it ) {
            new KListViewItem(lList,"?",*it);
        }
    }
}

void ReplayGainScanner::showDirDialogue()
{
    pScanStatus->show();

    QString dirname = KFileDialog::getExistingDirectory( QDir::homeDirPath(), this, i18n("Choose a directory!") );
    if (dirname != NULL) {
        //pScanStatus->setEnabled(true);
        dirLister->openURL(dirname);
    }
}

void ReplayGainScanner::allFilesListed(const KFileItemList &items)
{
    for( KFileItemList::Iterator it=items.begin(); it!=items.end(); ++it )
    {
        QString name=(*it)->url().path();
        QFileInfo fileInfo(name);
        //if( fileInfo.isFile() && tools.supportedFormats.contains(tools.extension(name)) ) new KListViewItem(lList,"?",name);
        //if( fileInfo.isFile() && prefs.fileFormat(tools.fileExtension(name)).replaygain != "" && prefs.fileFormat(tools.fileExtension(name)).replaygain != i18n("built-in") ) new KListViewItem(lList,"?",name);
        if( fileInfo.isFile() && replayGainPlugins.pluginByFormat(tools.fileExtension(name)).info.name != i18n("Unnamed") ) new KListViewItem(lList,"?",name);
        else if( fileInfo.isDir()) dirLister->openURL(name,true);
    }
}

void ReplayGainScanner::scanStatus(int percent)
{
    pScanStatus->setProgress(percent,100);
}

/*void ReplayGainScanner::scanCompleted()
{
    pScanStatus->setProgress(0,0);
    pScanStatus->setEnabled(false);
}*/

void ReplayGainScanner::removeFiles()
{
    QListViewItem *item = lList->firstChild(), *nextitem = 0;

    while( item != 0 ) {
        if( lList->isSelected( item ) ) {
            nextitem = item->nextSibling();
            lList->takeItem( item );
        }

        if( nextitem != 0 ) { item = nextitem; nextitem = 0; }
        else { item = item->nextSibling(); }
    }
}

void ReplayGainScanner::applyReplayGain()
{
    pCancel->setEnabled(true);
    //cFilter->setEnabled(false);
    //pScanUnknown->setEnabled(false);
    pTagVisible->setEnabled(false);
    pRemoveFiles->setEnabled(false);
    pRemoveTag->setEnabled(false);
    pOk->setEnabled(false);

    pProgressBar->setProgress(0,lList->childCount());
    if(cForce->isChecked()) mode=Force;
    else mode=Apply;
    iFile=0;
    bKilled=false;
    processNextFile();
}

void ReplayGainScanner::removeReplayGain()
{
    pCancel->setEnabled(true);
    //cFilter->setEnabled(false);
    //pScanUnknown->setEnabled(false);
    pTagVisible->setEnabled(false);
    pRemoveFiles->setEnabled(false);
    pRemoveTag->setEnabled(false);
    pOk->setEnabled(false);

    pProgressBar->setProgress(0,lList->childCount());
    mode=Remove;
    iFile=0;
    bKilled=false;
    processNextFile();
}

void ReplayGainScanner::processNextFile()
{
    QListViewItem *item = lList->firstChild(), *nextitem = 0;

    while( item != 0 ) {
        if( ( mode == Apply && ( item->text(0) == "?" || item->text(0) == i18n("no") ) ) ||
            ( mode == Force && ( item->text(0) == "?" || item->text(0) == i18n("no") ) ) ||
            ( mode == Remove && ( item->text(0) == "?" || item->text(0) == i18n("yes") ) ) ) {
            currentItem=item;
            process();
            return;
        }

        if( nextitem != 0 ) { item = nextitem; nextitem = 0; }
        else { item = item->nextSibling(); }
    }

    pProgressBar->setProgress(lList->childCount(),lList->childCount());
    pCancel->setEnabled(false);
    //cFilter->setEnabled(true);
    //pScanUnknown->setEnabled(true);
    pTagVisible->setEnabled(true);
    pRemoveFiles->setEnabled(true);
    pRemoveTag->setEnabled(true);
    pOk->setEnabled(true);
}

void ReplayGainScanner::process()
{
    QString filename=currentItem->text(1);
    QStringList params;
    QString param, paramSplinter;

    pProcess.clearArguments();

    ReplayGainPlugins::PluginStructure plugin;

//     plugin=replayGainPlugins.plugin(prefs.fileFormat(tools.fileExtension(filename)).replaygain);
//     if(plugin.info.name==i18n("Unnamed")) {
        plugin=replayGainPlugins.pluginByFormat(tools.fileExtension(filename));
        if(plugin.info.name==i18n("Unnamed")) {
            currentItem->setText(0,i18n("Error"));
            processNextFile();
            return;
        }
//     }

    param=plugin.replaygain.bin;
    if( plugin.replaygain.param ) param.append(" "+plugin.replaygain.param);
    if( mode==Apply && plugin.replaygain.track ) param.append(" "+plugin.replaygain.track);
    else if( mode==Remove && plugin.replaygain.remove ) param.append(" "+plugin.replaygain.remove);
    else if( mode==Force && plugin.replaygain.force ) param.append(" "+plugin.replaygain.force);

    param.append(" "+plugin.inFiles);

    params=QStringList::split(' ',param);

    for( QStringList::Iterator it=params.begin(); it!=params.end(); ++it )
    {
        paramSplinter=*it;
        paramSplinter.replace("%i",filename);
        pProcess << paramSplinter;
    }

    pProcess.start( KProcess::NotifyOnExit, KProcess::AllOutput );
}

void ReplayGainScanner::processExited(KProcess*)
{
    if( bKilled == true || ( pProcess.exitStatus() != 0 && prefs.fileFormat(tools.fileExtension(currentItem->text(1))).replaygain.find("mp3gain") == -1  ) ) {
        bKilled=false;
        pCancel->setEnabled(false);
        //cFilter->setEnabled(true);
        //pScanUnknown->setEnabled(true);
        pTagVisible->setEnabled(true);
        pRemoveFiles->setEnabled(true);
        pRemoveTag->setEnabled(true);
        pOk->setEnabled(true);
        return;
    }
    if(mode==Apply) currentItem->setText(0,i18n("yes"));
    else if(mode==Force) currentItem->setText(0,i18n("yes"));
    else if(mode==Remove) currentItem->setText(0,i18n("no"));
    iFile++;
    pProgressBar->setProgress(iFile,lList->childCount());
    processNextFile();
}

void ReplayGainScanner::cancel()
{
    pProcess.kill();
    bKilled=true;
}

void ReplayGainScanner::addFiles(QStringList list)
{
    for( QStringList::Iterator it=list.begin(); it!=list.end(); ++it )
    {
            QString sFormat=tools.fileExtension(*it);
            QFileInfo fileInfo(*it);
            if( fileInfo.isFile() && replayGainPlugins.pluginByFormat(sFormat).info.name != i18n("Unnamed") )
            {
                new KListViewItem(lList,"?",*it);
            }
            else if( fileInfo.isDir() )
            {
                pScanStatus->show();
                dirLister->openURL(*it);
            }
    }
}

void ReplayGainScanner::dragEnterEvent(QDragEnterEvent *e)
{
    e->accept( QUriDrag::canDecode(e) );
}

void ReplayGainScanner::dropEvent(QDropEvent *e)
{
    QString sFormat;
    QString file;
    QStringList list;
    if( QUriDrag::decodeLocalFiles( e, list ) )
    {
        for( QStringList::Iterator it=list.begin(); it!=list.end(); ++it )
        {
            file=QDir::convertSeparators(*it);
            sFormat=tools.fileExtension(file);
            QFileInfo fileInfo(file);
            if( fileInfo.isFile() && replayGainPlugins.pluginByFormat(sFormat).info.name != i18n("Unnamed") )
            {
                new KListViewItem(lList,"?",file);
            }
            else if( fileInfo.isDir() )
            {
                pScanStatus->show();
                dirLister->openURL(file);
            }
        }
    }
}


