
#include "config_dialogue_backends.h"
#include "config_defaults.h"
#include "config.h"
#include "backend_plugins.h"
#include "replaygain_plugins.h"

#include <qlistbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qfile.h>
#include <qprogressbar.h>
#include <qdir.h>
#include <qcheckbox.h>
#include <qtooltip.h>

#include <klocale.h>
#include <kpushbutton.h>
#include <kio/job.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kfiledialog.h>


ConfigDialogueBackends::ConfigDialogueBackends( QWidget *parent, const char *name )
    : QWidget( parent, name )
{
    QVBoxLayout *box = new QVBoxLayout( this, 5 );

    QStringList list;
    QString string;

    QLabel *lDirectoriesLabel = new QLabel( i18n("Directories to be scanned")+":", this, "lDirectoriesLabel" );
    box->addWidget(lDirectoriesLabel);
    QHBoxLayout *directoriesBox = new QHBoxLayout(box);
    lDirectories = new QListBox( this, "lDirectories" );
    lDirectories->insertStringList(prefs.backend.directories);
    directoriesBox->addWidget(lDirectories);
    QVBoxLayout *directoriesMiddleBox = new QVBoxLayout(directoriesBox);
    pDirUp = new KPushButton( "", this, "pDirUp" );
    pDirUp->setPixmap( KGlobal::iconLoader()->loadIcon("up",KIcon::Toolbar) );
    QToolTip::add( pDirUp, i18n("Move selected directory one position up.\nThis effects which backend will be choosen if there multiple versions.") );
    directoriesMiddleBox->addWidget(pDirUp);
    connect(pDirUp,SIGNAL(clicked()),this,SLOT(dirUp()));
    directoriesMiddleBox->addStretch();
    pDirDown = new KPushButton( "", this, "pDirDown" );
    pDirDown->setPixmap( KGlobal::iconLoader()->loadIcon("down",KIcon::Toolbar) );
    QToolTip::add( pDirDown, i18n("Move selected directory one position down.\nThis effects which backend will be choosen if there multiple versions.") );
    directoriesMiddleBox->addWidget(pDirDown);
    connect(pDirDown,SIGNAL(clicked()),this,SLOT(dirDown()));
    QVBoxLayout *directoriesRightBox = new QVBoxLayout(directoriesBox);
    pAddDirectory = new KPushButton( KGlobal::iconLoader()->loadIcon("edit_add",KIcon::Small), i18n("Add..."), this, "pAddDirectory" );
    directoriesRightBox->addWidget(pAddDirectory);
    connect(pAddDirectory,SIGNAL(clicked()),this,SLOT(addDirectory()));
    pRemoveDirectory = new KPushButton( KGlobal::iconLoader()->loadIcon("edit_remove",KIcon::Small), i18n("Remove"), this, "pRemoveDirectory" );
    directoriesRightBox->addWidget(pRemoveDirectory);
    connect(pRemoveDirectory,SIGNAL(clicked()),this,SLOT(removeDirectory()));
    directoriesRightBox->addStretch();
    pRescanDirectories = new KPushButton( KGlobal::iconLoader()->loadIcon("reload",KIcon::Small), i18n("Rescan"), this, "pRescanDirectories" );
    QToolTip::add( pRescanDirectories, i18n("Rescan directories for new backends.") );
    directoriesRightBox->addWidget(pRescanDirectories);
    connect(pRescanDirectories,SIGNAL(clicked()),this,SLOT(rescan()));

    box->addSpacing(10);

    for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).enc.bin) == -1 && (*it).enc.enabled ) {
            list.append((*it).enc.bin);
        }
        if( list.findIndex((*it).dec.bin) == -1 && (*it).dec.enabled ) {
            list.append((*it).dec.bin);
        }
    }
    for( ReplayGainPlugins::PluginStructureList::Iterator it=replayGainPlugins.plugins.begin(); it!=replayGainPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).replaygain.bin) == -1 ) {
            list.append((*it).replaygain.bin);
        }
    }

    for( QStringList::Iterator jt=list.begin(); jt!=list.end(); ++jt )
    {
        if( !string.isEmpty() ) string.append(", ");
        string.append((*jt));
    }

    QLabel *lAllPrograms = new QLabel( i18n("Supported programs")+":\n"+string, this, "lAllPrograms" );
    lAllPrograms->setAlignment(Qt::WordBreak);
    box->addWidget(lAllPrograms);

    box->addSpacing(10);

    QLabel *lProgramsLabel = new QLabel( i18n("Programs found")+":", this, "lProgramsLabel" );
    box->addWidget(lProgramsLabel);
    QHBoxLayout *programsBox = new QHBoxLayout(box);
    lPrograms = new QListBox( this, "lPrograms" );
    lPrograms->insertStringList(prefs.backend.programs);
    programsBox->addWidget(lPrograms);
    connect(lPrograms,SIGNAL(highlighted(int)),this,SLOT(pluginsSelectionChanged(int)));
    QVBoxLayout *programsRightBox = new QVBoxLayout(programsBox);
    pRemoveProgram = new KPushButton(KGlobal::iconLoader()->loadIcon("cancel",KIcon::Small),i18n("Delete"),this,"pRemoveProgram");
    pRemoveProgram->setEnabled(false);
    programsRightBox->addWidget(pRemoveProgram);
    connect(pRemoveProgram,SIGNAL(clicked()),this,SLOT(removeProgram()));
    programsRightBox->addStretch();

    box->addSpacing(10);

    QLabel *lOnlineBackendsLabel = new QLabel(i18n("Available programs")+":",this,"lOnlineBackendsLabel");
    box->addWidget(lOnlineBackendsLabel);
    QHBoxLayout *onlineBackendsBox = new QHBoxLayout(box);
    lOnlineBackends = new QListBox(this,"lOnlineBackends");
    onlineBackendsBox->addWidget(lOnlineBackends);
    QVBoxLayout *onlineBackendsRightBox = new QVBoxLayout(onlineBackendsBox);
    pRefreshBackends = new KPushButton(KGlobal::iconLoader()->loadIcon("reload",KIcon::Small),i18n("Refresh"),this,"pRefreshBackends");
    QToolTip::add( pRefreshBackends, i18n("Download the latest list of available backends.") );
    onlineBackendsRightBox->addWidget(pRefreshBackends);
    connect(pRefreshBackends,SIGNAL(clicked()),this,SLOT(refreshBackendDownloads()));
    onlineBackendsRightBox->addStretch();
    pInstallBackend = new KPushButton(KGlobal::iconLoader()->loadIcon("filesave",KIcon::Small),i18n("Install"),this,"pInstallBackend");
    QToolTip::add( pInstallBackend, i18n("Download the selected backend and install it into the soundKonverter directory.") );
    onlineBackendsRightBox->addWidget(pInstallBackend);
    connect(pInstallBackend,SIGNAL(clicked()),this,SLOT(installBackend()));
    pAboutOnlineBackend = new KPushButton(KGlobal::iconLoader()->loadIcon("messagebox_info",KIcon::Small),i18n("About..."),this,"pAboutOnlineBackend");
    pAboutOnlineBackend->setEnabled(false);
    onlineBackendsRightBox->addWidget(pAboutOnlineBackend);
    pDownloadBackendProgress = new QProgressBar(this,"pDownloadBackendProgress");
    pDownloadBackendProgress->hide();
    box->addWidget(pDownloadBackendProgress);
    cCheckOnlineBackends = new QCheckBox(i18n("Check for new programs on every startup"),this,"cCheckOnlineBackends");
    box->addWidget(cCheckOnlineBackends);
    cCheckOnlineBackends->setChecked(prefs.check.online_backends);
    connect(cCheckOnlineBackends,SIGNAL(toggled(bool)),this,SLOT(cfgChanged()));

    box->addStretch();
}

ConfigDialogueBackends::~ConfigDialogueBackends()
{
}

void ConfigDialogueBackends::resetDefaults()
{
    QStringList listDefaults;

    QString datadir=locateLocal("data","soundkonverter/bin/");
    datadir.remove(datadir.length()-1,1);
    listDefaults.append(datadir);
    listDefaults.append(QDir::homeDirPath()+"/bin");
    listDefaults.append("/usr/local/bin");
    listDefaults.append("/usr/bin");
    lDirectories->clear();
    lDirectories->insertStringList(listDefaults);
    listDefaults.clear();
    lPrograms->clear();
    lPrograms->insertStringList(listDefaults);
    cCheckOnlineBackends->setEnabled(DEFAULT_CHECK_ONLINE_BACKENDS);
}

void ConfigDialogueBackends::saveSettings()
{
    prefs.backend.directories.clear();
    for( uint i=0; i<lDirectories->count(); i++ )
    {
        prefs.backend.directories.append(lDirectories->text(i));
    }
    prefs.backend.programs.clear();
    for( uint i=0; i<lPrograms->count(); i++ )
    {
        prefs.backend.programs.append(lPrograms->text(i));
    }
    prefs.check.online_backends=cCheckOnlineBackends->isChecked();
}

void ConfigDialogueBackends::cfgChanged()
{
    emit configChanged();
}

void ConfigDialogueBackends::dirUp()
{
    int index=lDirectories->currentItem();
    if( index > 0 ) {
        QString text=lDirectories->currentText();
        lDirectories->removeItem(index);
        lDirectories->insertItem(text,index-1);
        lDirectories->setSelected(index-1,true);
    }

    prefs.backend.directories.clear();
    for( uint i=0; i<lDirectories->count(); i++ )
    {
        prefs.backend.directories.append(lDirectories->text(i));
    }
}

void ConfigDialogueBackends::dirDown()
{
    int index=lDirectories->currentItem();
    if( index < lDirectories->count()-1 ) {
        QString text=lDirectories->currentText();
        lDirectories->removeItem(index);
        lDirectories->insertItem(text,index+1);
        lDirectories->setSelected(index+1,true);
    }

    prefs.backend.directories.clear();
    for( uint i=0; i<lDirectories->count(); i++ )
    {
        prefs.backend.directories.append(lDirectories->text(i));
    }
}

void ConfigDialogueBackends::addDirectory()
{
    QString dirname = KFileDialog::getExistingDirectory( "/", this, i18n("Choose a directory!") );
    if (dirname != NULL) {
        lDirectories->insertItem(dirname);
        emit configChanged();
    }

    prefs.backend.directories.clear();
    for( uint i=0; i<lDirectories->count(); i++ )
    {
        prefs.backend.directories.append(lDirectories->text(i));
    }
}

void ConfigDialogueBackends::removeDirectory()
{
    lDirectories->removeItem(lDirectories->currentItem());

    prefs.backend.directories.clear();
    for( uint i=0; i<lDirectories->count(); i++ )
    {
        prefs.backend.directories.append(lDirectories->text(i));
    }
}

void ConfigDialogueBackends::rescan()
{
    QStringList list;
    QFile fFile;

    lPrograms->clear();

    for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).enc.bin) == -1 && (*it).enc.enabled ) {
            for( uint i=0; i<lDirectories->count(); i++ )
            {
                fFile.setName(lDirectories->text(i)+"/"+(*it).enc.bin);
                if( fFile.exists() && list.findIndex((*it).enc.bin) == -1 ) {
                    lPrograms->insertItem(lDirectories->text(i)+"/"+(*it).enc.bin);
                    list.append((*it).enc.bin);
                }
            }
        }
        if( list.findIndex((*it).dec.bin) == -1 && (*it).dec.enabled ) {
            for( uint i=0; i<lDirectories->count(); i++ )
            {
                fFile.setName(lDirectories->text(i)+"/"+(*it).dec.bin);
                if( fFile.exists() && list.findIndex((*it).dec.bin) == -1 ) {
                    lPrograms->insertItem(lDirectories->text(i)+"/"+(*it).dec.bin);
                    list.append((*it).dec.bin);
                }
            }
        }
    }
    for( ReplayGainPlugins::PluginStructureList::Iterator it=replayGainPlugins.plugins.begin(); it!=replayGainPlugins.plugins.end(); ++it )
    {
        if( list.findIndex((*it).replaygain.bin) == -1 ) {
            for( uint i=0; i<lDirectories->count(); i++ )
            {
                fFile.setName(lDirectories->text(i)+"/"+(*it).replaygain.bin);
                if( fFile.exists() && list.findIndex((*it).replaygain.bin) == -1 ) {
                    lPrograms->insertItem(lDirectories->text(i)+"/"+(*it).replaygain.bin);
                    list.append((*it).replaygain.bin);
                }
            }
        }
    }

    prefs.backend.programs.clear();
    for( uint i=0; i<lPrograms->count(); i++ )
    {
        prefs.backend.programs.append(lPrograms->text(i));
    }

    emit recalcEnDecoderPage();
}

void ConfigDialogueBackends::refreshBackendDownloads()
{
    pRefreshBackends->setEnabled(false);

    refreshBackendsJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/backends/download.php?version=0.2",
    locateLocal("data","soundkonverter/backendlist.txt"),-1,true,false,false);
    connect(refreshBackendsJob,SIGNAL(result(KIO::Job*)),this,SLOT(backendDownloadFinished(KIO::Job*)));
    // TODO timeout setzen
}

void ConfigDialogueBackends::backendDownloadFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        char data[100];
        char name[100];

        lOnlineBackends->clear();

        QFile file(locateLocal("data","soundkonverter/backendlist.txt"));
        file.open(IO_ReadOnly);
        while( file.readLine(data,sizeof(data)) > 5 ) {
            sscanf(data,"%s",&name);
            lOnlineBackends->insertItem(name);
        }
        file.close();
    }
    else {
        KMessageBox::information( this,
        i18n("The backendlist could not be downloaded. Please ensure, that your internet connection works correct.\nMaybe our server is busy at the moment, please try it again later."),
        i18n("Error while loading backendlist") );
    }

    pRefreshBackends->setEnabled(true);
}

void ConfigDialogueBackends::installBackend()
{
    pInstallBackend->setEnabled(false);
    QString name=lOnlineBackends->currentText();
    lOnlineBackends->removeItem(lOnlineBackends->currentItem());

    installBackendJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/backends/"+name,
    locateLocal("data","soundkonverter/bin/"+name),-1,true,false,false);
    connect(installBackendJob,SIGNAL(result(KIO::Job*)),this,SLOT(backendInstallFinished(KIO::Job*)));
    // TODO timeout setzen
}

void ConfigDialogueBackends::backendInstallFinished(KIO::Job* job)
{
    pInstallBackend->setEnabled(true);
    rescan();
    emit recalcEnDecoderPage();
}

void ConfigDialogueBackends::removeProgram()
{
    QString text=lPrograms->currentText();

    int ret = KMessageBox::questionYesNoCancel( this,
        i18n("Do you really want to remove this file forever?")+":\n"+text,
        i18n("Delete file") );

    if( ret == 3 ) {
        QFile::remove(text); // TODO fehlermeldung bei fehlschlag
        lPrograms->removeItem(lPrograms->currentItem());
    }
}

void ConfigDialogueBackends::pluginsSelectionChanged(int index)
{
    QString text=lPrograms->text(index);

    QFileInfo file(text);
    if(file.isWritable()) pRemoveProgram->setEnabled(true);
    else pRemoveProgram->setEnabled(false);
}
