
#include "config_dialogue_backend_plugins.h"
#include "config.h"
#include "config_defaults.h"
#include "backend_plugins.h"
#include "replaygain_plugins.h"
#include "tools.h"

#include <qlistbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qcheckbox.h>
#include <qevent.h>
#include <qdragobject.h>
#include <qtooltip.h>

#include <klocale.h>
#include <kpushbutton.h>
#include <kio/job.h>
#include <kstandarddirs.h>
#include <kmessagebox.h>
#include <kiconloader.h>
#include <kfiledialog.h>


ConfigDialogueBackendPlugins::ConfigDialogueBackendPlugins( QWidget *parent, const char *name )
    : QWidget( parent, name )
{
    setAcceptDrops(true);

    QVBoxLayout *box = new QVBoxLayout( this, 5 );

    QLabel *lPluginsLabel = new QLabel( i18n("Installed plugins")+":", this, "lPluginsLabel" );
    box->addWidget(lPluginsLabel);
    QHBoxLayout *pluginsBox = new QHBoxLayout(box);
    lPlugins = new QListBox( this, "lPlugins" );
    lPlugins->insertStringList(backendPlugins.loadedPlugins());
    lPlugins->insertStringList(replayGainPlugins.loadedPlugins());
    pluginsBox->addWidget(lPlugins);
    connect(lPlugins,SIGNAL(highlighted(int)),this,SLOT(pluginsSelectionChanged(int)));

    QVBoxLayout *pluginsRightBox = new QVBoxLayout(pluginsBox);
    pAddPlugin = new KPushButton( KGlobal::iconLoader()->loadIcon("edit_add",KIcon::Small), i18n("Add..."), this, "pAddPlugin" );
    pluginsRightBox->addWidget(pAddPlugin);
    connect(pAddPlugin,SIGNAL(clicked()),this,SLOT(addPlugin()));
    pRemovePlugin = new KPushButton( KGlobal::iconLoader()->loadIcon("edit_remove",KIcon::Small), i18n("Remove"), this, "pRemovePlugin" );
    pRemovePlugin->setEnabled(false);
    pluginsRightBox->addWidget(pRemovePlugin);
    connect(pRemovePlugin,SIGNAL(clicked()),this,SLOT(removePlugin()));
    pluginsRightBox->addStretch();
    pAboutPlugin = new KPushButton( KGlobal::iconLoader()->loadIcon("messagebox_info",KIcon::Small), i18n("About..."), this, "pAboutPlugin" );
    pluginsRightBox->addWidget(pAboutPlugin);
    connect(pAboutPlugin,SIGNAL(clicked()),this,SLOT(showPluginInfo()));

    box->addSpacing(10);

    QLabel *lOnlinePluginsLabel = new QLabel(i18n("Available plugins")+":",this,"lOnlinePluginsLabel");
    box->addWidget(lOnlinePluginsLabel);
    QHBoxLayout *onlinePluginsBox = new QHBoxLayout(box);
    lOnlinePlugins = new QListBox(this,"lOnlinePlugins");
    onlinePluginsBox->addWidget(lOnlinePlugins);
    QVBoxLayout *onlinePluginsRightBox = new QVBoxLayout(onlinePluginsBox);
    pRefreshPlugins = new KPushButton(KGlobal::iconLoader()->loadIcon("reload",KIcon::Small),i18n("Refresh"),this,"pRefreshPlugins");
    QToolTip::add( pRefreshPlugins, i18n("Download the latest list of available plugins.") );
    onlinePluginsRightBox->addWidget(pRefreshPlugins);
    connect(pRefreshPlugins,SIGNAL(clicked()),this,SLOT(refreshPlugins()));
    onlinePluginsRightBox->addStretch();
    pInstallPlugin = new KPushButton(KGlobal::iconLoader()->loadIcon("filesave",KIcon::Small),i18n("Install"),this,"pInstallPlugin");
    QToolTip::add( pInstallPlugin, i18n("Download the selected plugin and install it into the soundKonverter directory.") );
    onlinePluginsRightBox->addWidget(pInstallPlugin);
    connect(pInstallPlugin,SIGNAL(clicked()),this,SLOT(installPlugin()));
    pAboutOnlinePlugin = new KPushButton(KGlobal::iconLoader()->loadIcon("messagebox_info",KIcon::Small), i18n("About..."),this,"pAboutOnlinePlugin");
    onlinePluginsRightBox->addWidget(pAboutOnlinePlugin);
    connect(pAboutOnlinePlugin,SIGNAL(clicked()),this,SLOT(getPluginInfo()));
    /*pDownloadProgress = new QProgressBar(this,"pDownloadProgress");
    box->addWidget(pDownloadProgress);*/
    cCheckOnlinePlugins = new QCheckBox(i18n("Check for new plugins on every startup"),this,"cCheckOnlinePlugins");
    box->addWidget(cCheckOnlinePlugins);
    cCheckOnlinePlugins->setChecked(prefs.check.online_plugins);
    connect(cCheckOnlinePlugins,SIGNAL(toggled(bool)),this,SLOT(cfgChanged()));

    box->addStretch();
}

ConfigDialogueBackendPlugins::~ConfigDialogueBackendPlugins()
{
}

void ConfigDialogueBackendPlugins::resetDefaults()
{
    cCheckOnlinePlugins->setChecked(DEFAULT_CHECK_ONLINE_PLUGINS);
}

void ConfigDialogueBackendPlugins::saveSettings()
{
    prefs.check.online_plugins=cCheckOnlinePlugins->isChecked();
}

void ConfigDialogueBackendPlugins::cfgChanged()
{
    emit configChanged();
}

void ConfigDialogueBackendPlugins::addPlugin()
{
    KFileDialog *fileDialog;
    fileDialog = new KFileDialog( QDir::homeDirPath(), i18n("*.soundkonverter.xml|Plugins (*.soundkonverter.xml)"), 0, i18n("Choose a plugin to add!" ), true);
    fileDialog->setMode( KFile::ExistingOnly | KFile::LocalOnly );
    if( fileDialog->exec() == KDialog::Accepted ) {
        QString filePath=fileDialog->selectedFile();
        QString fileName=tools.fileName(filePath);
        addPluginJob = KIO::file_copy(filePath,
            locateLocal("data","soundkonverter/plugins/")+fileName,-1,true,false,false);
        connect(addPluginJob,SIGNAL(result(KIO::Job*)),this,SLOT(addFinished(KIO::Job*)));
    }
}

void ConfigDialogueBackendPlugins::addFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        backendPlugins.reload();
        replayGainPlugins.reload();
        lPlugins->clear();
        lPlugins->insertStringList(backendPlugins.loadedPlugins());
        lPlugins->insertStringList(replayGainPlugins.loadedPlugins());
        emit rescanForBackends();
        emit reloadEnDecoderPage();
    }
    else {
        KMessageBox::error( this,
            i18n("The plugin could not be installed. Please ensure that you have writepermission on your whole user directory."),
            i18n("Error while installing plugin") );
    }
}

void ConfigDialogueBackendPlugins::removePlugin()
{
    backendPlugins.remove(lPlugins->currentText());
    replayGainPlugins.remove(lPlugins->currentText());
    backendPlugins.reload();
    replayGainPlugins.reload();
    lPlugins->clear();
    lPlugins->insertStringList(backendPlugins.loadedPlugins());
    lPlugins->insertStringList(replayGainPlugins.loadedPlugins());
    emit rescanForBackends();
    emit reloadEnDecoderPage();
}

void ConfigDialogueBackendPlugins::showPluginInfo()
{
    QString text=lPlugins->currentText();
    char version[32];

    for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
    {
        if( (*it).info.name == text ) {
            sprintf(version,"%i",(*it).info.version);
            KMessageBox::information( this,
                i18n((*it).info.about)+"\n"+i18n("Version")+": "+version+"\n"+i18n("Author")+": "+(*it).info.author,
                i18n("About")+": "+text,
                QString::null, KMessageBox::Notify | KMessageBox::AllowLink );
        }
    }

    for( ReplayGainPlugins::PluginStructureList::Iterator it=replayGainPlugins.plugins.begin(); it!=replayGainPlugins.plugins.end(); ++it )
    {
        if( (*it).info.name == text ) {
            sprintf(version,"%i",(*it).info.version);
            KMessageBox::information( this,
                i18n((*it).info.about)+"\n"+i18n("Version")+": "+version+"\n"+i18n("Author")+": "+(*it).info.author,
                i18n("About")+": "+text,
                QString::null, KMessageBox::Notify | KMessageBox::AllowLink );
        }
    }
}

void ConfigDialogueBackendPlugins::refreshPlugins()
{
    pRefreshPlugins->setEnabled(false);

    refreshPluginsJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/plugins/download.php?version=0.2",
        locateLocal("data","soundkonverter/pluginlist.txt"),-1,true,false,false);
    connect(refreshPluginsJob,SIGNAL(result(KIO::Job*)),this,SLOT(refreshFinished(KIO::Job*)));
}

/*void ConfigDialogue::downloadProgressbar(KIO::Job* job,unsigned long percent)
{
    pDownloadProgress->setProgress(percent,100);
}*/

void ConfigDialogueBackendPlugins::refreshFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        char data[100];
        char name[100];

        lOnlinePlugins->clear();

        QFile file(locateLocal("data","soundkonverter/pluginlist.txt"));
        file.open(IO_ReadOnly);
        while( file.readLine(data,sizeof(data)) > 5 ) {
            sscanf(data,"%s",&name);
            lOnlinePlugins->insertItem(name);
        }
        file.close();
    }
    else {
        KMessageBox::error( this,
            i18n("The pluginlist could not be downloaded. Please ensure, that your internet connection works correct.\nMaybe our server is busy at the moment, please try it again later."),
            i18n("Error while loading pluginlist") );
    }

    pRefreshPlugins->setEnabled(true);
}

void ConfigDialogueBackendPlugins::getPluginInfo()
{
    if(lOnlinePlugins->currentItem()==-1) return;

    pAboutOnlinePlugin->setEnabled(false);

    QString filename=lOnlinePlugins->currentText();
    filename=filename.left(filename.length()-23);
    filename+=".info.txt";

    getPluginInfoJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/plugins/"+filename,
        locateLocal("data","soundkonverter/plugin_info.txt"),-1,true,false,false);
    connect(getPluginInfoJob,SIGNAL(result(KIO::Job*)),this,SLOT(getInfoFinished(KIO::Job*)));
}

void ConfigDialogueBackendPlugins::getInfoFinished(KIO::Job* job)
{
    if( job->error() == 0 ) {
        char data[10000];

        QString filename=lOnlinePlugins->currentText();
        QString info_str;

        QFile file(locateLocal("data","soundkonverter/plugin_info.txt"));
        file.open(IO_ReadOnly);
        file.readLine(data,sizeof(data));
        file.close();
        KMessageBox::information( this,
            i18n(data),
            i18n("About")+": "+filename,
            QString::null, KMessageBox::Notify | KMessageBox::AllowLink );
    }
    else {
        KMessageBox::error( this,
            i18n("The plugin info could not be downloaded. Please ensure, that your internet connection works correct.\nMaybe I have forgotten to upload that file. Mail me if you want. I'm sorry for that."),
            i18n("Error while loading plugin info") );
    }

    pAboutOnlinePlugin->setEnabled(true);
}

void ConfigDialogueBackendPlugins::installPlugin()
{
    pInstallPlugin->setEnabled(false);
    QString name;

    for( uint i=0; i<lOnlinePlugins->count(); i++ ) {
        if( lOnlinePlugins->isSelected(i) ) {
            name=lOnlinePlugins->text(i);
            lOnlinePlugins->removeItem(i);
            break;
        }
    }

    installPluginJob = KIO::file_copy("http://kaligames.de/downloads/soundkonverter/plugins/"+name,
        locateLocal("data","soundkonverter/plugins/"+name),-1,true,false,false);
    connect(installPluginJob,SIGNAL(result(KIO::Job*)),this,SLOT(installFinished(KIO::Job*)));
}

void ConfigDialogueBackendPlugins::installFinished(KIO::Job*)
{
    backendPlugins.reload();
    replayGainPlugins.reload();
    lPlugins->clear();
    lPlugins->insertStringList(backendPlugins.loadedPlugins());
    lPlugins->insertStringList(replayGainPlugins.loadedPlugins());
    emit rescanForBackends();
    emit reloadEnDecoderPage();
    pInstallPlugin->setEnabled(true);
}

void ConfigDialogueBackendPlugins::pluginsSelectionChanged(int index)
{
    QString text=lPlugins->text(index);

    for( BackendPlugins::PluginStructureList::Iterator it=backendPlugins.plugins.begin(); it!=backendPlugins.plugins.end(); ++it )
    {
        if( (*it).info.name == text ) {
            QFileInfo file((*it).file_name);
            if(file.isWritable()) pRemovePlugin->setEnabled(true);
            else pRemovePlugin->setEnabled(false);
        }
    }

    for( ReplayGainPlugins::PluginStructureList::Iterator it=replayGainPlugins.plugins.begin(); it!=replayGainPlugins.plugins.end(); ++it )
    {
        if( (*it).info.name == text ) {
            QFileInfo file((*it).file_name);
            if(file.isWritable()) pRemovePlugin->setEnabled(true);
            else pRemovePlugin->setEnabled(false);
        }
    }
}

void ConfigDialogueBackendPlugins::dragEnterEvent(QDragEnterEvent *e)
{
    e->accept( QUriDrag::canDecode(e) );
}

void ConfigDialogueBackendPlugins::dropEvent(QDropEvent *e)
{
    QStringList list;
    if( QUriDrag::decodeLocalFiles( e, list ) )
    {
        for( QStringList::Iterator it=list.begin(); it!=list.end(); ++it )
        {
            QString filePath=QDir::convertSeparators(*it);
            QFileInfo fileInfo(filePath);
            if( fileInfo.isFile() )
            {
                QString fileName=tools.fileName(filePath);
                addPluginJob = KIO::file_copy(filePath,
                    locateLocal("data","soundkonverter/plugins/")+fileName,-1,true,false,false);
                connect(addPluginJob,SIGNAL(result(KIO::Job*)),this,SLOT(addFinished(KIO::Job*)));
            }
        }
    }
}

