/*  smplayer, GUI front-end for mplayer.
    Copyright (C) 2007 Ricardo Villalba <rvm@escomposlinux.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "basegui.h"

#include "filedialog.h"
#include <qmessagebox.h>
#include <qlabel.h>
#include <qslider.h>
#include <qpopupmenu.h>
#include <qfileinfo.h>
#include <qtextbrowser.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qapplication.h>
#include <qinputdialog.h>
#include <qcursor.h>
#include <qmenubar.h>
#include <qtimer.h>
#include <qstyle.h>
#include <qdockwindow.h>
#include <qdragobject.h>
#include <qstrlist.h>
#include <qurl.h>
#include <qregexp.h>
#include <qstatusbar.h>
#include "myaction.h"
#include "mypopupmenu.h"
#include <cmath>

#include "mplayerwindow.h"
#include "desktopinfo.h"
#include "helper.h"
#include "global.h"
#include "translator.h"
#include "images.h"
#include "preferences.h"
#include "inforeader.h"
#include "timeslider.h"
#include "preferencesdialog.h"
#include "logwindow.h"
#include "playlist.h"
#include "filepropertiesdialog.h"
#include "eqslider.h"
#include "videoequalizer.h"
#include "inputdvddirectory.h"
#include "recents.h"
#include "aboutdialog.h"
#include "csmanager.h"
#include "config.h"
#include "actionseditor.h"
#include "qt3_4_compat.h"

#if KDE_SUPPORT
#include <kurl.h>
#endif

// Menus IDs

#define OPEN_MENU 1
#define PLAY_MENU 2
#define VIDEO_MENU 3
#define AUDIO_MENU 4
#define SUBTITLES_MENU 5
#define BROWSE_MENU 6
#define OPTIONS_MENU 7
#define HELP_MENU 8

#define OPEN_RECENTS 1
#define RECENTS_CLEAR 1000

#define PLAY_SPEED 1

#define VIDEO_SIZE 1
#define VIDEO_ASPECT 2
#define VIDEO_DEINTERLACE 3
//#define VIDEO_DENOISE 4
#define VIDEO_FILTERS 5
#define VIDEO_PANSCAN 6

#define VIDEO_FILTERS_DENOISE 1

#define AUDIO_TRACK 1
#define AUDIO_CHANNELS 2
#define AUDIO_STEREO 3
#define AUDIO_FILTERS 4

#define SUBTITLES_TRACK 1

#define BROWSE_TITLE 1
#define BROWSE_CHAPTER 2
#define BROWSE_ANGLE 3

#define OPTIONS_OSD 1
#define OPTIONS_LOGS 2


BaseGui::BaseGui( QStringList files, QWidget* parent, const char* name, WFlags fl ) 
	: QMainWindow( parent, name, fl ) 
{
	setCaption( "SMPlayer" );

	initial_files = files;
	last_second = 0;
	near_top = false;
	near_bottom = false;

	// Objects not created yet
	csmanager = 0;
    popup = 0;
	info_reader = 0;
	pref_dialog = 0;
	file_dialog = 0;

	// Create objects:
	recents = new Recents(this, "recents list");

	createPanel();
	setCentralWidget(panel);

	createMplayerWindow();
	createCore();
	createPlaylist();
	createVideoEqualizer();

	// Mouse Wheel
	connect( this, SIGNAL(wheelUp()),
             core, SLOT(wheelUp()) );
	connect( this, SIGNAL(wheelDown()),
             core, SLOT(wheelDown()) );
	connect( mplayerwindow, SIGNAL(wheelUp()),
             core, SLOT(wheelUp()) );
	connect( mplayerwindow, SIGNAL(wheelDown()),
             core, SLOT(wheelDown()) );

	// Set style before changing color of widgets:
	// Set style
#if STYLE_SWITCHING
	#if QT_VERSION >= 0x040000
	qDebug( "Style name: '%s'", qApp->style()->name() );
	qDebug( "Style class name: '%s'", qApp->style()->className() );
	default_style = qApp->style()->name();
	#else
	qDebug( "Style name: '%s'", qApp->style().name() );
	qDebug( "Style class name: '%s'", qApp->style().className() );
	default_style = qApp->style().name();
	#endif
	if (!pref->style.isEmpty()) {
		qApp->setStyle( pref->style );
	}
#endif

    mplayer_log_window = new LogWindow(this, "mplayer_log_window");
	smplayer_log_window = new LogWindow(this, "smplayer_log_window");

	createActions();
	createMenus();

	languageChange();

	setAcceptDrops(TRUE);

	#if QT_VERSION < 0x040000
	layout()->setSpacing(-1); // Removes two extra lines
	#endif
	resize(580, 440);

	panel->setFocus();

	initializeGui();
}

void BaseGui::initializeGui() {
	if (pref->compact_mode) toggleCompactMode(TRUE);
	if (pref->stay_on_top) toggleStayOnTop(TRUE);
	toggleFrameCounter( pref->show_frame_counter );

#if QT_VERSION >= 0x040200
	changeStyleSheet(pref->iconset);
#endif

	recents->setMaxItems( pref->recents_max_items);
	updateRecents();

	// Call loadActions() outside initialization of the class.
	// Otherwise DefaultGui (and other subclasses) doesn't exist, and 
	// its actions are not loaded
	QTimer * load_actions_timer = new QTimer(this);
	connect( load_actions_timer, SIGNAL(timeout()), this, SLOT(loadActions()) );
	load_actions_timer->start( 20, TRUE);

	// Single instance
	if (pref->use_single_instance) {
		csmanager = new CSManager( pref->connection_port, initial_files, 
                                   this, "csmanager" );
		connect(csmanager, SIGNAL(requestOpen(QString)),
	            this, SLOT(remoteOpen(QString)));
		connect(csmanager, SIGNAL(requestOpenFiles(QStringList)),
	            this, SLOT(remoteOpenFiles(QStringList)));
		connect(csmanager, SIGNAL(requestFunction(QString)),
	            this, SLOT(processFunction(QString)));
		connect(csmanager, SIGNAL(mayStartUp(int)),
                this, SLOT(continueStartUp(int)) );
		connect(csmanager, SIGNAL(mayClose()),
                qApp, SLOT(quit()) );
	} else {
		//firstShow();
		// Call firstShow a little bit later, so constructor ends
        // and other subclasses can be created (baseguiplus reimplements
        // firstShow() 
		QTimer * show_timer = new QTimer(this);
		connect( show_timer, SIGNAL(timeout()), this, SLOT(firstShow()) );
		show_timer->start( 40, TRUE);
		
		if (!initial_files.empty()) {
			openFiles(initial_files);
		}
	}
}

void BaseGui::continueStartUp(int code) {
	qDebug("BaseGui::continueStartUp: code: %d", code);
	firstShow();
	if (!initial_files.empty()) {
		openFiles(initial_files);
	}

	if ((CSManager::StartingCode) code == CSManager::UnknownServerFound) {
		QMessageBox::warning( this, tr("SMPlayer - Warning"), 
			tr("Port %1 is already used by another application.\n"
               "Cannot start server.").arg(pref->connection_port), 
			QMessageBox::Ok, QMessageBox::NoButton );
	}
	else
	if ((CSManager::StartingCode) code == CSManager::ServerDoesNotRespond) {
		QMessageBox::warning( this, tr("SMPlayer - Warning"), 
			tr("Server at port %1 does not respond.\n"
               "The single instance option has been disabled.")
               .arg(pref->connection_port), 
			QMessageBox::Ok, QMessageBox::NoButton );
		pref->use_single_instance = false;
	}
}

void BaseGui::remoteOpen(QString file) {
	qDebug("BaseGui::remoteOpen: '%s'", file.utf8().data());
	if (isMinimized()) showNormal();
	if (!isVisible()) show();
	raise();
	setActiveWindow();
	open(file);
}

void BaseGui::remoteOpenFiles(QStringList files) {
	qDebug("BaseGui::remoteOpenFiles");
	if (isMinimized()) showNormal();
	if (!isVisible()) show();
	raise();
	setActiveWindow();
	openFiles(files);
}

void BaseGui::firstShow() {
	qDebug("BaseGui::firstShow");
	show();
}



BaseGui::~BaseGui() {
	delete core; // delete before mplayerwindow, otherwise, segfault...
	delete mplayer_log_window;
	delete smplayer_log_window;
	delete playlist;
}

void BaseGui::createActions() {
	// Menu File
	openFileAct = new MyAction( QKeySequence("Ctrl+F"), this, "open_file" );
	connect( openFileAct, SIGNAL(activated()),
             this, SLOT(openFile()) );

	openDirectoryAct = new MyAction( this, "open_directory" );
	connect( openDirectoryAct, SIGNAL(activated()),
             this, SLOT(openDirectory()) );

	openPlaylistAct = new MyAction( this, "open_playlist" );
	connect( openPlaylistAct, SIGNAL(activated()),
             playlist, SLOT(load()) );

	openVCDAct = new MyAction( this, "open_vcd" );
	connect( openVCDAct, SIGNAL(activated()),
             this, SLOT(openVCD()) );

	openAudioCDAct = new MyAction( this, "open_audio_cd" );
	connect( openAudioCDAct, SIGNAL(activated()),
             this, SLOT(openAudioCD()) );

#ifdef Q_OS_WIN
	// mplayer in windows can't play vcds
	openVCDAct->setEnabled(false);
#endif

	openDVDAct = new MyAction( this, "open_dvd" );
	connect( openDVDAct, SIGNAL(activated()),
             this, SLOT(openDVD()) );

	openDVDFolderAct = new MyAction( this, "open_dvd_folder" );
	connect( openDVDFolderAct, SIGNAL(activated()),
             this, SLOT(openDVDFromFolder()) );

	openURLAct = new MyAction( QKeySequence("Ctrl+U"), this, "open_url" );
	connect( openURLAct, SIGNAL(activated()),
             this, SLOT(openURL()) );

	exitAct = new MyAction( QKeySequence("Ctrl+X"), this, "close" );
	connect( exitAct, SIGNAL(activated()), this, SLOT(closeWindow()) );

	// Menu Play
	playAct = new MyAction( this, "play" );
	connect( playAct, SIGNAL(activated()),
             core, SLOT(play()) );

	playOrPauseAct = new MyAction( Qt::Key_MediaPlay, this, "play_or_pause" );
	connect( playOrPauseAct, SIGNAL(activated()),
             core, SLOT(play_or_pause()) );

	pauseAct = new MyAction( Qt::Key_Space, this, "pause" );
	connect( pauseAct, SIGNAL(activated()),
             core, SLOT(pause()) );

	pauseAndStepAct = new MyAction( this, "pause_and_frame_step" );
	connect( pauseAndStepAct, SIGNAL(activated()),
             core, SLOT(pause_and_frame_step()) );

	stopAct = new MyAction( Qt::Key_MediaStop, this, "stop" );
	connect( stopAct, SIGNAL(activated()),
             core, SLOT(stop()) );

	frameStepAct = new MyAction( Qt::Key_Period, this, "frame_step" );
	connect( frameStepAct, SIGNAL(activated()),
             core, SLOT(frameStep()) );

	rewind1Act = new MyAction( Qt::Key_Left, this, "rewind1" );
	connect( rewind1Act, SIGNAL(activated()),
             core, SLOT(srewind()) );

	rewind2Act = new MyAction( Qt::Key_Down, this, "rewind2" );
	connect( rewind2Act, SIGNAL(activated()),
             core, SLOT(rewind()) );

	rewind3Act = new MyAction( Qt::Key_Next, this, "rewind3" );
	connect( rewind3Act, SIGNAL(activated()),
             core, SLOT(fastrewind()) );

	forward1Act = new MyAction( Qt::Key_Right, this, "forward1" );
	connect( forward1Act, SIGNAL(activated()),
             core, SLOT(sforward()) );

	forward2Act = new MyAction( Qt::Key_Up, this, "forward2" );
	connect( forward2Act, SIGNAL(activated()),
             core, SLOT(forward()) );

	forward3Act = new MyAction( Qt::Key_Prior, this, "forward3" );
	connect( forward3Act, SIGNAL(activated()),
             core, SLOT(fastforward()) );

	repeatAct = new MyAction( this, "repeat" );
	repeatAct->setToggleAction( true );
	connect( repeatAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleRepeat(bool)) );

	// Submenu Speed
	normalSpeedAct = new MyAction( Qt::Key_Backspace, this, "normal_speed" );
	connect( normalSpeedAct, SIGNAL(activated()),
             core, SLOT(normalSpeed()) );

	halveSpeedAct = new MyAction( Qt::Key_BraceLeft, this, "halve_speed" );
	connect( halveSpeedAct, SIGNAL(activated()),
             core, SLOT(halveSpeed()) );

	doubleSpeedAct = new MyAction( Qt::Key_BraceRight, this, "double_speed" );
	connect( doubleSpeedAct, SIGNAL(activated()),
             core, SLOT(doubleSpeed()) );

	decSpeedAct = new MyAction( Qt::Key_BracketLeft, this, "dec_speed" );
	connect( decSpeedAct, SIGNAL(activated()),
             core, SLOT(decSpeed()) );

	incSpeedAct = new MyAction( Qt::Key_BracketRight, this, "inc_speed" );
	connect( incSpeedAct, SIGNAL(activated()),
             core, SLOT(incSpeed()) );

	speedAct = new QActionGroup( this, "speed" );
	speedAct->setUsesDropDown(true);
	speedAct->add( normalSpeedAct );
	speedAct->add( halveSpeedAct );
	speedAct->add( doubleSpeedAct );
	speedAct->add( decSpeedAct );
	speedAct->add( incSpeedAct );

	// Menu Video
	fullscreenAct = new MyAction( Qt::Key_F, this, "fullscreen" );
	fullscreenAct->setToggleAction( true );
	connect( fullscreenAct, SIGNAL(toggled(bool)),
             this, SLOT(toggleFullscreen(bool)) );

	compactAct = new MyAction( QKeySequence("Ctrl+C"), this, "compact" );
	compactAct->setToggleAction( true );
	connect( compactAct, SIGNAL(toggled(bool)),
             this, SLOT(toggleCompactMode(bool)) );

	equalizerAct = new MyAction( QKeySequence("Ctrl+E"), this, "equalizer" );
	equalizerAct->setToggleAction( true );
	connect( equalizerAct, SIGNAL(toggled(bool)),
             this, SLOT(showEqualizer(bool)) );

	screenshotAct = new MyAction( Qt::Key_S, this, "screenshot" );
	connect( screenshotAct, SIGNAL(activated()),
             core, SLOT(screenshot()) );

	onTopAct = new MyAction( this, "on_top" );
	onTopAct->setToggleAction( true );
	connect( onTopAct, SIGNAL(toggled(bool)),
             this, SLOT(toggleStayOnTop(bool)) );

	// Submenu filter
	postProcessingAct = new MyAction( this, "postprocessing" );
	postProcessingAct->setToggleAction( true );
	connect( postProcessingAct, SIGNAL(toggled(bool)),
             core, SLOT(togglePostprocessing(bool)) );

	phaseAct = new MyAction( this, "autodetect_phase" );
	phaseAct->setToggleAction( true );
	connect( phaseAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleAutophase(bool)) );

	deblockAct = new MyAction( this, "deblock" );
	deblockAct->setToggleAction( true );
	connect( deblockAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleDeblock(bool)) );

	deringAct = new MyAction( this, "dering" );
	deringAct->setToggleAction( true );
	connect( deringAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleDering(bool)) );

	addNoiseAct = new MyAction( this, "add_noise" );
	addNoiseAct->setToggleAction( true );
	connect( addNoiseAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleNoise(bool)) );

	videoFilterAct = new QActionGroup( this, "video_filter" );
	videoFilterAct->setUsesDropDown(true);
	videoFilterAct->setExclusive(false);
	videoFilterAct->add( postProcessingAct );
	videoFilterAct->add( phaseAct );
	videoFilterAct->add( deblockAct );
	videoFilterAct->add( deringAct );
	videoFilterAct->add( addNoiseAct );

	// Menu Audio
	muteAct = new MyAction( Qt::Key_M, this, "mute" );
	muteAct->setToggleAction( true );
	connect( muteAct, SIGNAL(toggled(bool)),
             core, SLOT(mute(bool)) );

	decVolumeAct = new MyAction( Qt::Key_9, this, "dec_volume" );
	connect( decVolumeAct, SIGNAL(activated()),
             core, SLOT(decVolume()) );

	incVolumeAct = new MyAction( Qt::Key_0, this, "inc_volume" );
	connect( incVolumeAct, SIGNAL(activated()),
             core, SLOT(incVolume()) );

	decAudioDelayAct = new MyAction( Qt::Key_Minus, this, "dec_audio_delay" );
	connect( decAudioDelayAct, SIGNAL(activated()),
             core, SLOT(decAudioDelay()) );

	incAudioDelayAct = new MyAction( Qt::Key_Plus, this, "inc_audio_delay" );
	connect( incAudioDelayAct, SIGNAL(activated()),
             core, SLOT(incAudioDelay()) );

	loadAudioAct = new MyAction( this, "load_audio_file" );
	connect( loadAudioAct, SIGNAL(activated()),
             this, SLOT(loadAudioFile()) );

	unloadAudioAct = new MyAction( this, "unload_audio_file" );
	connect( unloadAudioAct, SIGNAL(activated()),
             core, SLOT(unloadAudioFile()) );


	// Submenu Filters
	extrastereoAct = new MyAction( this, "extrastereo_filter" );
	extrastereoAct->setToggleAction( true );
	connect( extrastereoAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleExtrastereo(bool)) );

	karaokeAct = new MyAction( this, "karaoke_filter" );
	karaokeAct->setToggleAction( true );
	connect( karaokeAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleKaraoke(bool)) );

	volnormAct = new MyAction( this, "volnorm_filter" );
	volnormAct->setToggleAction( true );
	connect( volnormAct, SIGNAL(toggled(bool)),
             core, SLOT(toggleVolnorm(bool)) );

	audioFilterAct = new QActionGroup( this, "audio_filter" );
	audioFilterAct->setUsesDropDown(true);
	audioFilterAct->setExclusive(false);
	audioFilterAct->add( extrastereoAct );
	audioFilterAct->add( karaokeAct );
	audioFilterAct->add( volnormAct );

	// Menu Subtitles
	loadSubsAct = new MyAction( this, "load_subs" );
	connect( loadSubsAct, SIGNAL(activated()),
             this, SLOT(loadSub()) );

#if SUBTITLES_BY_INDEX
	unloadSubsAct = new MyAction( this, "unload_subs" );
	connect( unloadSubsAct, SIGNAL(activated()),
             core, SLOT(unloadSub()) );
#endif

	decSubDelayAct = new MyAction( Qt::Key_Z, this, "dec_sub_delay" );
	connect( decSubDelayAct, SIGNAL(activated()),
             core, SLOT(decSubDelay()) );

	incSubDelayAct = new MyAction( Qt::Key_X, this, "inc_sub_delay" );
	connect( incSubDelayAct, SIGNAL(activated()),
             core, SLOT(incSubDelay()) );

	decSubPosAct = new MyAction( Qt::Key_R, this, "dec_sub_pos" );
	connect( decSubPosAct, SIGNAL(activated()),
             core, SLOT(decSubPos()) );
	incSubPosAct = new MyAction( Qt::Key_T, this, "inc_sub_pos" );
	connect( incSubPosAct, SIGNAL(activated()),
             core, SLOT(incSubPos()) );

	decSubStepAct = new MyAction( Qt::Key_G, this, "dec_sub_step" );
	connect( decSubStepAct, SIGNAL(activated()),
             core, SLOT(decSubStep()) );

	incSubStepAct = new MyAction( Qt::Key_Y, this, "inc_sub_step" );
	connect( incSubStepAct, SIGNAL(activated()),
             core, SLOT(incSubStep()) );

	// Menu Options
	showPlaylistAct = new MyAction( QKeySequence("Ctrl+L"), this, "show_playlist" );
	showPlaylistAct->setToggleAction( true );
	connect( showPlaylistAct, SIGNAL(toggled(bool)),
             this, SLOT(showPlaylist(bool)) );

	showPropertiesAct = new MyAction( QKeySequence("Ctrl+I"), this, "show_file_properties" );
	connect( showPropertiesAct, SIGNAL(activated()),
             this, SLOT(showFilePropertiesDialog()) );

	frameCounterAct = new MyAction( this, "frame_counter" );
	frameCounterAct->setToggleAction( true );
	connect( frameCounterAct, SIGNAL(toggled(bool)),
             this, SLOT(toggleFrameCounter(bool)) );

	showPreferencesAct = new MyAction( QKeySequence("Ctrl+P"), this, "show_preferences" );
	connect( showPreferencesAct, SIGNAL(activated()),
             this, SLOT(showPreferencesDialog()) );

	// Submenu Logs
	showLogMplayerAct = new MyAction( QKeySequence("Ctrl+M"), this, "show_mplayer_log" );
	connect( showLogMplayerAct, SIGNAL(activated()),
             this, SLOT(showMplayerLog()) );

	showLogSmplayerAct = new MyAction( QKeySequence("Ctrl+S"), this, "show_smplayer_log" );
	connect( showLogSmplayerAct, SIGNAL(activated()),
             this, SLOT(showLog()) );

	showLogsAct = new QActionGroup( this, "show_logs" );
	showLogsAct->setUsesDropDown(true);
	showLogsAct->add( showLogMplayerAct );
	showLogsAct->add( showLogSmplayerAct );

	// Menu Help
	aboutQtAct = new MyAction( this, "about_qt" );
	connect( aboutQtAct, SIGNAL(activated()),
             this, SLOT(helpAboutQt()) );

	aboutThisAct = new MyAction( this, "about_smplayer" );
	connect( aboutThisAct, SIGNAL(activated()),
             this, SLOT(helpAbout()) );

	// Playlist
	playNextAct = new MyAction(Qt::Key_Greater, this, "play_next");
	connect( playNextAct, SIGNAL(activated()), playlist, SLOT(playNext()) );

	playPrevAct = new MyAction(Qt::Key_Less, this, "play_prev");
	connect( playPrevAct, SIGNAL(activated()), playlist, SLOT(playPrev()) );


	// Move video window and zoom
	moveUpAct = new MyAction(Qt::ALT | Qt::Key_Up, this, "move_up");
	connect( moveUpAct, SIGNAL(activated()), mplayerwindow, SLOT(moveUp()) );

	moveDownAct = new MyAction(Qt::ALT | Qt::Key_Down, this, "move_down");
	connect( moveDownAct, SIGNAL(activated()), mplayerwindow, SLOT(moveDown()) );

	moveLeftAct = new MyAction(Qt::ALT | Qt::Key_Left, this, "move_left");
	connect( moveLeftAct, SIGNAL(activated()), mplayerwindow, SLOT(moveLeft()) );

	moveRightAct = new MyAction(Qt::ALT | Qt::Key_Right, this, "move_right");
	connect( moveRightAct, SIGNAL(activated()), mplayerwindow, SLOT(moveRight()) );

	incZoomAct = new MyAction(Qt::Key_E, this, "inc_zoom");
	connect( incZoomAct, SIGNAL(activated()), core, SLOT(incPanscan()) );

	decZoomAct = new MyAction(Qt::Key_W, this, "dec_zoom");
	connect( decZoomAct, SIGNAL(activated()), core, SLOT(decPanscan()) );

	resetZoomAct = new MyAction(Qt::SHIFT | Qt::Key_E, this, "reset_zoom");
	connect( resetZoomAct, SIGNAL(activated()), core, SLOT(resetPanscan()) );


	// Actions not in menus or buttons
	// Volume 2
	decVolume2Act = new MyAction( Qt::Key_Slash, this, "dec_volume2" );
	connect( decVolume2Act, SIGNAL(activated()), core, SLOT(decVolume()) );

	incVolume2Act = new MyAction( Qt::Key_Asterisk, this, "inc_volume2" );
	connect( incVolume2Act, SIGNAL(activated()), core, SLOT(incVolume()) );

	// Exit fullscreen
	exitFullscreenAct = new MyAction( Qt::Key_Escape, this, "exit_fullscreen" );
	connect( exitFullscreenAct, SIGNAL(activated()), this, SLOT(exitFullscreen()) );

	nextOSDAct = new MyAction( Qt::Key_O, this, "next_osd");
	connect( nextOSDAct, SIGNAL(activated()), core, SLOT(nextOSD()) );

	decContrastAct = new MyAction( Qt::Key_1, this, "dec_contrast");
	connect( decContrastAct, SIGNAL(activated()), core, SLOT(decContrast()) );

	incContrastAct = new MyAction( Qt::Key_2, this, "inc_contrast");
	connect( incContrastAct, SIGNAL(activated()), core, SLOT(incContrast()) );

	decBrightnessAct = new MyAction( Qt::Key_3, this, "dec_brightness");
	connect( decBrightnessAct, SIGNAL(activated()), core, SLOT(decBrightness()) );

	incBrightnessAct = new MyAction( Qt::Key_4, this, "inc_brightness");
	connect( incBrightnessAct, SIGNAL(activated()), core, SLOT(incBrightness()) );

	decHueAct = new MyAction(Qt::Key_5, this, "dec_hue");
	connect( decHueAct, SIGNAL(activated()), core, SLOT(decHue()) );

	incHueAct = new MyAction( Qt::Key_6, this, "inc_hue");
	connect( incHueAct, SIGNAL(activated()), core, SLOT(incHue()) );

	decSaturationAct = new MyAction( Qt::Key_7, this, "dec_saturation");
	connect( decSaturationAct, SIGNAL(activated()), core, SLOT(decSaturation()) );

	incSaturationAct = new MyAction( Qt::Key_8, this, "inc_saturation");
	connect( incSaturationAct, SIGNAL(activated()), core, SLOT(incSaturation()) );

	decGammaAct = new MyAction( Qt::ALT | Qt::Key_1, this, "dec_gamma");
	connect( decGammaAct, SIGNAL(activated()), core, SLOT(decGamma()) );

	incGammaAct = new MyAction( Qt::ALT | Qt::Key_2, this, "inc_gamma");
	connect( incGammaAct, SIGNAL(activated()), core, SLOT(incGamma()) );

	nextAudioAct = new MyAction( Qt::Key_H, this, "next_audio");
	connect( nextAudioAct, SIGNAL(activated()), core, SLOT(nextAudio()) );

	nextSubtitleAct = new MyAction( Qt::Key_J, this, "next_subtitle");
	connect( nextSubtitleAct, SIGNAL(activated()), core, SLOT(nextSubtitle()) );

	nextChapterAct = new MyAction( Qt::Key_At, this, "next_chapter");
	connect( nextChapterAct, SIGNAL(activated()), core, SLOT(nextChapter()) );

	prevChapterAct = new MyAction( Qt::Key_Exclam, this, "prev_chapter");
	connect( prevChapterAct, SIGNAL(activated()), core, SLOT(prevChapter()) );

	doubleSizeAct = new MyAction( Qt::CTRL | Qt::Key_D, this, "toggle_double_size");
	connect( doubleSizeAct, SIGNAL(activated()), core, SLOT(toggleDoubleSize()) );
}


void BaseGui::languageChange() {
	setIcon( Images::icon("logo", 64) );

	// ACTIONS

	// Menu File
	openFileAct->change( Images::icon("open"), tr("&File...") );
	openDirectoryAct->change( Images::icon("openfolder"), tr("D&irectory...") );
	openPlaylistAct->change( Images::icon("open_playlist"), tr("&Playlist...") );
	openVCDAct->change( Images::icon("vcd"), tr("V&CD") );
	openAudioCDAct->change( Images::icon("cdda"), tr("&Audio CD") );
	openDVDAct->change( Images::icon("dvd"), tr("&DVD from drive") );
	openDVDFolderAct->change( Images::icon("dvd_hd"), tr("D&VD from folder...") );
	openURLAct->change( Images::icon("url"), tr("&URL...") );
	exitAct->change( Images::icon("close"), tr("C&lose") );

	// Menu Play
	playAct->change( Images::icon("play") , tr("P&lay") );
	pauseAct->change( Images::icon("pause"), tr("&Pause"));
	stopAct->change( Images::icon("stop"), tr("&Stop") );
	frameStepAct->change( Images::icon("frame_step"), tr("&Frame step") );

	playOrPauseAct->change( Images::icon("play"), tr("Play / Pause") );
	pauseAndStepAct->change( Images::icon("pause"), tr("Pause / Frame step") );

	setJumpTexts(); // Texts for rewind*Act and forward*Act

	repeatAct->change( Images::icon("repeat"), tr("&Repeat") );

	// Submenu speed
	normalSpeedAct->change( tr("&Normal speed") );
	halveSpeedAct->change( tr("&Halve speed") );
	doubleSpeedAct->change( tr("&Double speed") );
	decSpeedAct->change( tr("Speed &-10%") );
	incSpeedAct->change( tr("Speed &+10%") );

	speedAct->setMenuText( tr("Sp&eed") );
	speedAct->setIconSet( Images::icon("speed") );

	// Menu Video
	fullscreenAct->change( Images::icon("fullscreen"), tr("&Fullscreen") );
	compactAct->change( Images::icon("compact"), tr("&Compact mode") );
	equalizerAct->change( Images::icon("equalizer"), tr("&Equalizer") );
	screenshotAct->change( Images::icon("screenshot"), tr("&Screenshot") );
	onTopAct->change( Images::icon("ontop"), tr("S&tay on top") );

	decZoomAct->change( tr("Zoom &-") );
	incZoomAct->change( tr("Zoom &+") );
	resetZoomAct->change( tr("&Reset") );
	moveLeftAct->change( tr("Move &left") );
	moveRightAct->change( tr("Move &right") );
	moveUpAct->change( tr("Move &up") );
	moveDownAct->change( tr("Move &down") );

	// Submenu Filters
	postProcessingAct->change( tr("&Postprocessing") );
	phaseAct->change( tr("&Autodetect phase") );
	deblockAct->change( tr("&Deblock") );
	deringAct->change( tr("De&ring") );
	addNoiseAct->change( tr("Add n&oise") );
	videoFilterAct->setMenuText( tr("F&ilters") );
	videoFilterAct->setIconSet( Images::icon("video_filters") );

	// Menu Audio
	QIconSet icset( Images::icon("volume") );
	icset.setPixmap( Images::icon("mute"), QIconSet::Automatic, 
                     QIconSet::Normal, QIconSet::On  );
	muteAct->change( icset, tr("&Mute") );
	decVolumeAct->change( Images::icon("audio_down"), tr("Volume &-") );
	incVolumeAct->change( Images::icon("audio_up"), tr("Volume &+") );
	decAudioDelayAct->change( Images::icon("delay_down"), tr("&Delay -") );
	incAudioDelayAct->change( Images::icon("delay_up"), tr("D&elay +") );
	loadAudioAct->change( Images::icon("open"), tr("&Load external file...") );
	unloadAudioAct->change( Images::icon("unload"), tr("U&nload") );

	// Submenu Filters
	extrastereoAct->change( tr("&Extrastereo") );
	karaokeAct->change( tr("&Karaoke") );
	volnormAct->change( tr("Volume &normalization") );
	audioFilterAct->setMenuText( tr("&Filters") );
	audioFilterAct->setIconSet( Images::icon("audio_filters") );

	// Menu Subtitles
	loadSubsAct->change( Images::icon("open"), tr("&Load...") );
#if SUBTITLES_BY_INDEX
	unloadSubsAct->change( Images::icon("unload"), tr("U&nload") );
#endif
	decSubDelayAct->change( Images::icon("delay_down"), tr("Delay &-") );
	incSubDelayAct->change( Images::icon("delay_up"), tr("Delay &+") );
	decSubPosAct->change( Images::icon("sub_up"), tr("&Up") );
	incSubPosAct->change( Images::icon("sub_down"), tr("&Down") );
	decSubStepAct->change( Images::icon("dec_sub_step"), 
                           tr("&Previous line in subtitles") );
	incSubStepAct->change( Images::icon("inc_sub_step"), 
                           tr("N&ext line in subtitles") );

	// Menu Options
	showPlaylistAct->change( Images::icon("playlist"), tr("&Playlist") );
	showPropertiesAct->change( Images::icon("info"), tr("View &info and properties...") );
	frameCounterAct->change( Images::icon("frame_counter"),
                             tr("&Show frame counter") );
	showPreferencesAct->change( Images::icon("prefs"), tr("P&references") );

	// Submenu Logs
	showLogMplayerAct->change( "mplayer" );
	showLogSmplayerAct->change( "smplayer" );
	showLogsAct->setMenuText( tr("&View logs") );
	showLogsAct->setIconSet( Images::icon("logs") );

	// Menu Help
	aboutQtAct->change( Images::icon("qt"), tr("About &Qt") );
	aboutThisAct->change( Images::icon("logo_small"), tr("About &SMPlayer") );

	// Playlist
	playNextAct->change( Images::icon("next"), tr("&Next") );
	playPrevAct->change( Images::icon("previous"), tr("Pre&vious") );


	// Actions not in menus or buttons
	// Volume 2
	decVolume2Act->change( tr("Dec volume (2)") );
	incVolume2Act->change( tr("Inc volume (2)") );

	// Exit fullscreen
	exitFullscreenAct->change( tr("Exit fullscreen") );

	nextOSDAct->change( tr("OSD - Next level") );
	decContrastAct->change( tr("Dec contrast") );
	incContrastAct->change( tr("Inc contrast") );
	decBrightnessAct->change( tr("Dec brightness") );
	incBrightnessAct->change( tr("Inc brightness") );
	decHueAct->change( tr("Dec hue") );
	incHueAct->change( tr("Inc hue") );
	decSaturationAct->change( tr("Dec saturation") );
	incSaturationAct->change( tr("Inc saturation") );
	decGammaAct->change( tr("Dec gamma") );
	incGammaAct->change( tr("Inc gamma") );
	nextAudioAct->change( tr("Next audio") );
	nextSubtitleAct->change( tr("Next subtitle") );
	nextChapterAct->change( tr("Next chapter") );
	prevChapterAct->change( tr("Previous chapter") );
	doubleSizeAct->change( tr("Toggle double size") );

	// MENUS
	menuBar()->changeItem( OPEN_MENU, tr("&Open") );
	menuBar()->changeItem( PLAY_MENU, tr("&Play") );
	menuBar()->changeItem( VIDEO_MENU, tr("&Video") );
	menuBar()->changeItem( AUDIO_MENU, tr("&Audio") );
	menuBar()->changeItem( SUBTITLES_MENU, tr("&Subtitles") );
	menuBar()->changeItem( BROWSE_MENU, tr("&Browse") );
	menuBar()->changeItem( OPTIONS_MENU, tr("Op&tions") );
	menuBar()->changeItem( HELP_MENU, tr("&Help") );

	popup->changeItem( OPEN_MENU, Images::icon("open_menu"), tr("&Open") );
	popup->changeItem( PLAY_MENU, Images::icon("play_menu"), tr("&Play") );
	popup->changeItem( VIDEO_MENU, Images::icon("video_menu"), tr("&Video") );
	popup->changeItem( AUDIO_MENU, Images::icon("audio_menu"), tr("&Audio") );
	popup->changeItem( SUBTITLES_MENU, Images::icon("subtitles_menu"), tr("&Subtitles") );
	popup->changeItem( BROWSE_MENU, Images::icon("browse_menu"), tr("&Browse") );
	popup->changeItem( OPTIONS_MENU, Images::icon("options_menu"), tr("Op&tions") );

	// Menu Open
	openMenu->changeItem( OPEN_RECENTS, Images::icon("recents"), tr("&Recent files") );
	recentfiles_menu->changeItem( RECENTS_CLEAR, Images::icon("delete"), tr("&Clear") );

	// Menu Play
	playMenu->changeItem( PLAY_SPEED, Images::icon("speed"), tr("Sp&eed") );

	// Menu Video
	videoMenu->changeItem( VIDEO_SIZE, Images::icon("video_size"), 
                           tr("Si&ze") );
	videoMenu->changeItem( VIDEO_ASPECT, Images::icon("aspect"), 
                           tr("&Aspect ratio") );
	videoMenu->changeItem( VIDEO_DEINTERLACE, Images::icon("deinterlace"), 
                           tr("&Deinterlace") );
	videoMenu->changeItem( VIDEO_FILTERS, Images::icon("video_filters"),
                           tr("F&ilters") );
	videoMenu->changeItem( VIDEO_PANSCAN, Images::icon("panscan"),
                           tr("&Pan && scan") );

	aspect_menu->changeItem( MediaSettings::AspectAuto, tr("&Autodetect") );
	aspect_menu->changeItem( MediaSettings::Aspect43, tr("&4:3") );
	aspect_menu->changeItem( MediaSettings::Aspect54, tr("&5:4") );
	aspect_menu->changeItem( MediaSettings::Aspect149, tr("&14:9") );
	aspect_menu->changeItem( MediaSettings::Aspect169, tr("16:&9") );
	aspect_menu->changeItem( MediaSettings::Aspect1610, tr("1&6:10") );
	aspect_menu->changeItem( MediaSettings::Aspect235, tr("&2.35:1") );
	aspect_menu->changeItem( MediaSettings::Aspect43Letterbox, tr("4:3 &Letterbox") );
	aspect_menu->changeItem( MediaSettings::Aspect169Letterbox, tr("16:9 L&etterbox") );
	aspect_menu->changeItem( MediaSettings::Aspect43Panscan, tr("4:3 &Panscan") );
	aspect_menu->changeItem( MediaSettings::Aspect43To169, tr("4:3 &to 16:9") );

	deinterlace_menu->changeItem( MediaSettings::NoDeinterlace, tr("&None") );
	deinterlace_menu->changeItem( MediaSettings::L5, tr("&Lowpass5") );
	deinterlace_menu->changeItem( MediaSettings::Yadif, tr("&Yadif (normal)") );
	deinterlace_menu->changeItem( MediaSettings::Yadif_1, tr("Y&adif (double framerate)") );
	deinterlace_menu->changeItem( MediaSettings::LB, tr("Linear &Blend") );
	deinterlace_menu->changeItem( MediaSettings::Kerndeint, tr("&Kerndeint") );

	denoise_menu->changeItem( MediaSettings::NoDenoise, tr("&None") );
	denoise_menu->changeItem( MediaSettings::DenoiseNormal, tr("N&ormal") );
	denoise_menu->changeItem( MediaSettings::DenoiseSoft, tr("&Soft") );

	videofilter_menu->changeItem( VIDEO_FILTERS_DENOISE, Images::icon("denoise"), 
                                  tr("De&noise") );


	// Menu Audio
	audioMenu->changeItem( AUDIO_TRACK, Images::icon("audio_track"), 
                           tr("&Track") );
	audioMenu->changeItem( AUDIO_CHANNELS, Images::icon("audio_channels"), 
                           tr("&Channels") );
	audioMenu->changeItem( AUDIO_STEREO, Images::icon("stereo_mode"), 
                           tr("&Stereo mode") );
	audioMenu->changeItem( AUDIO_FILTERS, Images::icon("audio_filters"),
                           tr("&Filters") );

	audiochannels_menu->changeItem( MediaSettings::ChDefault, tr("&Default") );
	audiochannels_menu->changeItem( MediaSettings::ChStereo, tr("&Stereo") );
	audiochannels_menu->changeItem( MediaSettings::ChSurround, 
                                    tr("&4.0 Surround") );
	audiochannels_menu->changeItem( MediaSettings::ChFull51, tr("&5.1 Surround") );

	stereomode_menu->changeItem( MediaSettings::Stereo, tr("&Stereo") );
	stereomode_menu->changeItem( MediaSettings::Left, tr("&Left channel") );
	stereomode_menu->changeItem( MediaSettings::Right, tr("&Right channel") );

	// Menu Subtitle
	subtitlesMenu->changeItem( SUBTITLES_TRACK, Images::icon("sub"), 
                               tr("&Select") );

	// Menu Browse 
	browseMenu->changeItem( BROWSE_TITLE, Images::icon("title"), tr("&Title") );
	browseMenu->changeItem( BROWSE_CHAPTER, Images::icon("chapter"), 
                            tr("&Chapter") );
	browseMenu->changeItem( BROWSE_ANGLE, Images::icon("angle"), tr("&Angle") );

	// Menu Options
	optionsMenu->changeItem( OPTIONS_OSD, Images::icon("osd"), tr("&OSD") );
	optionsMenu->changeItem( OPTIONS_LOGS, Images::icon("logs"), 
                             tr("&View logs") );

	osd_menu->changeItem( Preferences::None, tr("&Disabled") );
	osd_menu->changeItem( Preferences::Seek, tr("&Seek bar") );
	osd_menu->changeItem( Preferences::SeekTimer, tr("&Time") );
	osd_menu->changeItem( Preferences::SeekTimerTotal, tr("Time + T&otal time") );

	// To be sure that the "<empty>" string is translated
	initializeMenus();


	// Other things
	mplayer_log_window->setCaption( tr("SMPlayer - mplayer log") );
	smplayer_log_window->setCaption( tr("SMPlayer - smplayer log") );

	updateRecents();
	updateWidgets();

	// Update actions view in preferences
	// It has to be done, here. The actions are translated after the
	// preferences dialog.
	if (pref_dialog) pref_dialog->actions_editor->updateView();
}

void BaseGui::setJumpTexts() {
	rewind1Act->change( Images::icon("rewind10s"), 
                        tr("-%1").arg(Helper::timeForJumps(pref->seeking1)) );
	rewind2Act->change( Images::icon("rewind1m"), 
                        tr("-%1").arg(Helper::timeForJumps(pref->seeking2)) );
	rewind3Act->change( Images::icon("rewind10m"), 
                        tr("-%1").arg(Helper::timeForJumps(pref->seeking3)) );

	forward1Act->change( Images::icon("forward10s"),
                         tr("+%1").arg(Helper::timeForJumps(pref->seeking1)) );
	forward2Act->change( Images::icon("forward1m"),
                         tr("+%1").arg(Helper::timeForJumps(pref->seeking2)) );
	forward3Act->change( Images::icon("forward10m"),
                         tr("+%1").arg(Helper::timeForJumps(pref->seeking3)) );
}

void BaseGui::setWindowCaption(const QString & title) {
	setCaption(title);
}

void BaseGui::createCore() {
	core = new Core( mplayerwindow, this, "core" );

	connect( core, SIGNAL(menusNeedInitialize()),
             this, SLOT(initializeMenus()) );
	connect( core, SIGNAL(widgetsNeedUpdate()),
             this, SLOT(updateWidgets()) );
	connect( core, SIGNAL(equalizerNeedsUpdate()),
             this, SLOT(updateEqualizer()) );

	connect( core, SIGNAL(showFrame(int)),
             this, SIGNAL(frameChanged(int)) );

	connect( core, SIGNAL(showTime(double)),
             this, SLOT(gotCurrentTime(double)) );

	connect( core, SIGNAL(needResize(int, int)),
             this, SLOT(resizeWindow(int,int)) );
	connect( core, SIGNAL(showMessage(QString)),
             this, SLOT(displayMessage(QString)) );
	connect( core, SIGNAL(stateChanged(int)),
             this, SLOT(displayState(int)) );

	connect( core, SIGNAL(mediaStartPlay()),
             this, SLOT(enterFullscreenOnPlay()) );
	connect( core, SIGNAL(mediaStoppedByUser()),
             this, SLOT(exitFullscreenOnStop()) );
	connect( core, SIGNAL(mediaLoaded()),
             this, SLOT(newMediaLoaded()) );
	connect( core, SIGNAL(mediaInfoChanged()),
             this, SLOT(updateMediaInfo()) );

	// Hide mplayer window
	connect( core, SIGNAL(noVideo()),
             this, SLOT(hidePanel()) );

}

void BaseGui::createMplayerWindow() {
    mplayerwindow = new MplayerWindow( panel, "mplayerwindow" );
	mplayerwindow->setColorKey( pref->color_key );

    panelLayout = new QGridLayout( panel, 1, 1, 0, 0 );
    panelLayout->addWidget( mplayerwindow, 0, 0 );

	// mplayerwindow
    connect( mplayerwindow, SIGNAL(rightButtonReleased(QPoint)),
	         this, SLOT(showPopupMenu(QPoint)) );

	// mplayerwindow mouse events
	connect( mplayerwindow, SIGNAL(doubleClicked()),
             this, SLOT(doubleClickFunction()) );
	connect( mplayerwindow, SIGNAL(leftClicked()),
             this, SLOT(leftClickFunction()) );
	connect( mplayerwindow, SIGNAL(mouseMoved(QPoint)),
             this, SLOT(checkMousePos(QPoint)) );


}

void BaseGui::createVideoEqualizer() {
	// Equalizer
	equalizer = new VideoEqualizer(this, "video_equalizer", 
                                   Qt::WType_Dialog );

	connect( equalizer->contrast, SIGNAL(valueChanged(int)), 
             core, SLOT(setContrast(int)) );
	connect( equalizer->brightness, SIGNAL(valueChanged(int)), 
             core, SLOT(setBrightness(int)) );
	connect( equalizer->hue, SIGNAL(valueChanged(int)), 
             core, SLOT(setHue(int)) );
	connect( equalizer->saturation, SIGNAL(valueChanged(int)), 
             core, SLOT(setSaturation(int)) );
	connect( equalizer->gamma, SIGNAL(valueChanged(int)), 
             core, SLOT(setGamma(int)) );
	connect( equalizer, SIGNAL(visibilityChanged()),
             this, SLOT(updateWidgets()) );
}

void BaseGui::createPlaylist() {
#if DOCK_PLAYLIST
	playlist = new Playlist(core, this, "playlist", 0);
#else
	//playlist = new Playlist(core, this, "playlist");
	playlist = new Playlist(core, 0, "playlist");
#endif

	connect( playlist, SIGNAL(playlistEnded()),
             this, SLOT(exitFullscreenOnStop()) );
	connect( playlist, SIGNAL(visibilityChanged()),
             this, SLOT(playlistVisibilityChanged()) );
}

void BaseGui::createPanel() {
	panel = new QWidget( this, "panel" );
	panel->setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );
	panel->setMinimumSize( QSize(0,0) );
	panel->setFocusPolicy( STRONGFOCUS );

	// panel
	#if QT_VERSION >= 0x040000
	panel->setAutoFillBackground(TRUE);
	#endif
	panel->setBackgroundColor( QColor(0,0,0) );
}

void BaseGui::createPreferencesDialog() {
	pref_dialog = new PreferencesDialog(this, "pref_dialog", FALSE);
	pref_dialog->setActionsList( actions_list );

	connect( pref_dialog, SIGNAL(applyButtonPressed()),
             this, SLOT(applyNewPreferences()) );
	// Clear recent's list
	connect( pref_dialog, SIGNAL(clearRecentsButtonPressed()),
             this, SLOT(clearRecentsList()) );
}

void BaseGui::createFilePropertiesDialog() {
	file_dialog = new FilePropertiesDialog(this, "file_dialog", FALSE);
	connect( file_dialog, SIGNAL(applyButtonPressed()),
             this, SLOT(applyFileProperties()) );
}


void BaseGui::createMenus() {
	// MENUS
	openMenu = new QPopupMenu(this);
	menuBar()->insertItem( "Open", openMenu, OPEN_MENU );
	playMenu = new QPopupMenu(this);
	menuBar()->insertItem( "Play", playMenu, PLAY_MENU );
	videoMenu = new QPopupMenu(this);
	menuBar()->insertItem( "Video", videoMenu, VIDEO_MENU );
	audioMenu = new QPopupMenu(this);
	menuBar()->insertItem( "Audio", audioMenu, AUDIO_MENU );
	subtitlesMenu = new QPopupMenu( this );
	menuBar()->insertItem( "Subtitles", subtitlesMenu, SUBTITLES_MENU );
	browseMenu = new QPopupMenu(this);
	menuBar()->insertItem( "Browse", browseMenu, BROWSE_MENU );
	optionsMenu = new QPopupMenu( this );
	menuBar()->insertItem( "Options", optionsMenu, OPTIONS_MENU );
	helpMenu = new QPopupMenu( this );
	menuBar()->insertItem( "Help", helpMenu, HELP_MENU );


	// OPEN MENU
	openFileAct->addTo( openMenu );

	recentfiles_menu = new QPopupMenu;
	recentfiles_menu->insertItem( "Clear", this, SLOT(clearRecentsList()), 0, 
                                  RECENTS_CLEAR );
	recentfiles_menu->insertSeparator();
	connect( recentfiles_menu, SIGNAL(activated(int)), 
             this, SLOT(openRecent(int)) );

	openMenu->insertItem( "Recent files", recentfiles_menu, OPEN_RECENTS );

	openDirectoryAct->addTo( openMenu );
	openPlaylistAct->addTo( openMenu );
	openDVDAct->addTo( openMenu );
	openDVDFolderAct->addTo( openMenu );
	openVCDAct->addTo( openMenu );
	openAudioCDAct->addTo( openMenu );
	openURLAct->addTo( openMenu );

	openMenu->insertSeparator();
	exitAct->addTo( openMenu );
	
	// PLAY MENU
	playAct->addTo( playMenu );
	pauseAct->addTo( playMenu );
	stopAct->addTo( playMenu );
	frameStepAct->addTo( playMenu );
	playMenu->insertSeparator();
	rewind1Act->addTo( playMenu );
	forward1Act->addTo( playMenu );
	rewind2Act->addTo( playMenu );
	forward2Act->addTo( playMenu );
	rewind3Act->addTo( playMenu );
	forward3Act->addTo( playMenu );
	playMenu->insertSeparator();

	// Speed submenu
	speed_menu = new QPopupMenu(this, "speed_menu");
	normalSpeedAct->addTo( speed_menu );
	halveSpeedAct->addTo( speed_menu );
	doubleSpeedAct->addTo( speed_menu );
	decSpeedAct->addTo( speed_menu );
	incSpeedAct->addTo( speed_menu );
	playMenu->insertItem( "Speed", speed_menu, PLAY_SPEED);

	repeatAct->addTo( playMenu );
	
	// VIDEO MENU
	videoMenu->setCheckable(TRUE);

	fullscreenAct->addTo( videoMenu );
	compactAct->addTo( videoMenu) ;

	// Size submenu
	videosize_menu = new MyPopupMenu(this, "videosize_menu");
	videosize_menu->insertItem( "5&0%", 50 );
	videosize_menu->insertItem( "7&5%", 75 );
	videosize_menu->insertItem( "&100%", 100 );
	videosize_menu->setAccel( Qt::CTRL | Qt::Key_1, 100 );
	videosize_menu->insertItem( "1&25%", 125 );
	videosize_menu->insertItem( "15&0%", 150 );
	videosize_menu->insertItem( "1&75%", 175 );
	videosize_menu->insertItem( "&200%", 200 );
	videosize_menu->setAccel( Qt::CTRL | Qt::Key_2, 200 );
	videosize_menu->insertItem( "&300%", 300 );
	videosize_menu->insertItem( "&400%", 400 );
	videosize_menu->setCheckable(TRUE);
	connect( videosize_menu, SIGNAL(activated(int)),
             core, SLOT(changeSize(int)) );
	videoMenu->insertItem( "Size", videosize_menu, VIDEO_SIZE );

	// Panscan submenu
	panscan_menu = new MyPopupMenu(this, "panscan_menu");
	resetZoomAct->addTo( panscan_menu );
	decZoomAct->addTo( panscan_menu );
	incZoomAct->addTo( panscan_menu );
	panscan_menu->insertSeparator();
	moveLeftAct->addTo( panscan_menu );
	moveRightAct->addTo( panscan_menu );
	moveUpAct->addTo( panscan_menu );
	moveDownAct->addTo( panscan_menu );
	videoMenu->insertItem( "Panscan", panscan_menu, VIDEO_PANSCAN );

	// Aspect submenu
	aspect_menu = new MyPopupMenu(this, "aspect_menu");
	aspect_menu->insertItem( "Autodetect", MediaSettings::AspectAuto );
	aspect_menu->insertItem( "4:3", MediaSettings::Aspect43 );
	aspect_menu->insertItem( "5:4", MediaSettings::Aspect54 );
	aspect_menu->insertItem( "14:9", MediaSettings::Aspect149 );
	aspect_menu->insertItem( "16:9", MediaSettings::Aspect169 );
	aspect_menu->insertItem( "16:10", MediaSettings::Aspect1610 );
	aspect_menu->insertItem( "2.35:1", MediaSettings::Aspect235 );
	aspect_menu->insertSeparator();
	aspect_menu->insertItem( "4:3 Letterbox", MediaSettings::Aspect43Letterbox );
	aspect_menu->insertItem( "16:9 Letterbox", MediaSettings::Aspect169Letterbox );
	aspect_menu->insertItem( "4:3 Panscan", MediaSettings::Aspect43Panscan );
	aspect_menu->insertItem( "4:3 to 16:9", MediaSettings::Aspect43To169 );
	aspect_menu->setCheckable(TRUE);

	videoMenu->insertItem( "Aspect ratio", aspect_menu, VIDEO_ASPECT );
	connect( aspect_menu, SIGNAL(activated(int)),
             core, SLOT(changeAspectRatio(int)) );

	// Deinterlace submenu
	deinterlace_menu = new MyPopupMenu(this, "deinterlace_menu");
	deinterlace_menu->insertItem( "None", MediaSettings::NoDeinterlace );
	deinterlace_menu->insertItem( "Lowpass5", MediaSettings::L5 );
	deinterlace_menu->insertItem( "Yadif", MediaSettings::Yadif );
	deinterlace_menu->insertItem( "Yadif_1", MediaSettings::Yadif_1 );
	deinterlace_menu->insertItem( "Linear Blend", MediaSettings::LB );
	deinterlace_menu->insertItem( "Kerndeint", MediaSettings::Kerndeint );

	videoMenu->insertItem( "Deinterlace", deinterlace_menu, VIDEO_DEINTERLACE );
	connect( deinterlace_menu, SIGNAL(activated(int)),
             core, SLOT(changeDeinterlace(int)) );

	// Video filter submenu
	videofilter_menu = new QPopupMenu(this, "video_filters");
	postProcessingAct->addTo( videofilter_menu );
	phaseAct->addTo( videofilter_menu );
	deblockAct->addTo( videofilter_menu );	
	deringAct->addTo( videofilter_menu );
	addNoiseAct->addTo( videofilter_menu );
	videoMenu->insertItem( "Filters", videofilter_menu, VIDEO_FILTERS);

	// Denoise submenu
	denoise_menu = new MyPopupMenu(this, "denoise_menu");
	denoise_menu->insertItem( "None", MediaSettings::NoDenoise );
	denoise_menu->insertItem( "Normal", MediaSettings::DenoiseNormal );
	denoise_menu->insertItem( "Soft", MediaSettings::DenoiseSoft );
	//videoMenu->insertItem( "Denoise", denoise_menu, VIDEO_DENOISE );
	videofilter_menu->insertItem( "Denoise", denoise_menu, VIDEO_FILTERS_DENOISE );
	connect( denoise_menu, SIGNAL(activated(int)), 
             core, SLOT(changeDenoise(int)) );

	equalizerAct->addTo( videoMenu );
	screenshotAct->addTo( videoMenu );
	onTopAct->addTo( videoMenu );

    // AUDIO MENU
	audioMenu->setCheckable(TRUE);

	// Audio track submenu
	audiotrack_menu = new MyPopupMenu(this, "audiotrack_menu");
    audiotrack_menu->setCheckable(TRUE);
    connect( audiotrack_menu, SIGNAL(activated(int)), 
	        core, SLOT(changeAudio(int)) );
    audioMenu->insertItem( "Track", audiotrack_menu, AUDIO_TRACK );
	loadAudioAct->addTo( audioMenu );
	unloadAudioAct->addTo( audioMenu );

	// Filter submenu
	audiofilter_menu = new QPopupMenu(this, "audio_filters");
	extrastereoAct->addTo( audiofilter_menu );
	karaokeAct->addTo( audiofilter_menu );
	volnormAct->addTo( audiofilter_menu );
	audioMenu->insertItem( "Filters", audiofilter_menu, AUDIO_FILTERS );

	// Audio channels submenu
	audiochannels_menu = new MyPopupMenu(this, "audiochannels_menu");
	audiochannels_menu->setCheckable(TRUE);
	audiochannels_menu->insertItem( "Default", MediaSettings::ChDefault );
	audiochannels_menu->insertItem( "Stereo", MediaSettings::ChStereo );
	audiochannels_menu->insertItem( "4.0 Surround", MediaSettings::ChSurround );
	audiochannels_menu->insertItem( "5.1 Surround", MediaSettings::ChFull51 );
	audioMenu->insertItem( "Channels", audiochannels_menu, AUDIO_CHANNELS );
	connect( audiochannels_menu, SIGNAL(activated(int)),
             core, SLOT(setAudioChannels(int)) );

	// Stereo mode submenu
	stereomode_menu = new MyPopupMenu(this, "stereomode_menu");
	stereomode_menu->setCheckable(TRUE);
	stereomode_menu->insertItem( "Stereo", MediaSettings::Stereo );
	stereomode_menu->insertItem( "Left channel", MediaSettings::Left );
	stereomode_menu->insertItem( "Right channel", MediaSettings::Right );
	audioMenu->insertItem( "Stereo mode", stereomode_menu, AUDIO_STEREO );
	connect( stereomode_menu, SIGNAL(activated(int)),
             core, SLOT(setStereoMode(int)) );

	muteAct->addTo( audioMenu );
	audioMenu->insertSeparator();
	decVolumeAct->addTo( audioMenu );
	incVolumeAct->addTo( audioMenu );
	audioMenu->insertSeparator();
	decAudioDelayAct->addTo( audioMenu );
	incAudioDelayAct->addTo( audioMenu );


	// SUBTITLES MENU
	// Track submenu
    subtitlestrack_menu = new MyPopupMenu(this, "subtitlestrack_menu");
	subtitlestrack_menu->setCheckable(TRUE);
    connect( subtitlestrack_menu, SIGNAL(activated(int)), 
	         core, SLOT(changeSubtitle(int)) );
    subtitlesMenu->insertItem( "Select", subtitlestrack_menu, SUBTITLES_TRACK );
	loadSubsAct->addTo( subtitlesMenu );
#if SUBTITLES_BY_INDEX
	unloadSubsAct->addTo( subtitlesMenu );
#endif
	subtitlesMenu->insertSeparator();
	decSubDelayAct->addTo( subtitlesMenu );
	incSubDelayAct->addTo( subtitlesMenu );
	subtitlesMenu->insertSeparator();
	decSubPosAct->addTo( subtitlesMenu );
	incSubPosAct->addTo( subtitlesMenu );
	subtitlesMenu->insertSeparator();
	decSubStepAct->addTo( subtitlesMenu );
	incSubStepAct->addTo( subtitlesMenu );

	// BROWSE MENU
	// Titles submenu
	titles_menu = new MyPopupMenu(this, "titles_menu");
	titles_menu->setCheckable(TRUE);
	connect( titles_menu, SIGNAL(activated(int)),
			core, SLOT(changeTitle(int)) );
	browseMenu->insertItem( "Title", titles_menu, BROWSE_TITLE );

	// Chapters submenu
	chapters_menu = new MyPopupMenu(this, "chapters_menu");
	chapters_menu->setCheckable(TRUE);
	connect( chapters_menu, SIGNAL(activated(int)),
			core, SLOT(changeChapter(int)) );
	browseMenu->insertItem( "Chapter", chapters_menu, BROWSE_CHAPTER );

	// Angles submenu
	angles_menu= new MyPopupMenu(this, "angles_menu");
	angles_menu->setCheckable(TRUE);
	connect( angles_menu, SIGNAL(activated(int)),
			core, SLOT(changeAngle(int)) );
	browseMenu->insertItem( "Angle", angles_menu, BROWSE_ANGLE );


	// OPTIONS MENU
	showPropertiesAct->addTo( optionsMenu );
	showPlaylistAct->addTo( optionsMenu );
	frameCounterAct->addTo( optionsMenu );

	// OSD submenu
	osd_menu = new MyPopupMenu(this, "osd_menu");
	osd_menu->insertItem( "Disabled", Preferences::None );
	osd_menu->insertItem( "Seek bar", Preferences::Seek );
	osd_menu->insertItem( "Time", Preferences::SeekTimer );
	osd_menu->insertItem( "Time + Total time", Preferences::SeekTimerTotal );
	optionsMenu->insertItem( "OSD", osd_menu, OPTIONS_OSD );
	connect( osd_menu, SIGNAL(activated(int)),
             core, SLOT(changeOSD(int)) );

	// Logs submenu
	logs_menu = new QPopupMenu(this, "logs_menu");
	showLogMplayerAct->addTo( logs_menu );
	showLogSmplayerAct->addTo( logs_menu );
	optionsMenu->insertItem( "View logs", logs_menu, OPTIONS_LOGS );

	showPreferencesAct->addTo( optionsMenu );


	// HELP MENU
	aboutQtAct->addTo( helpMenu );
	aboutThisAct->addTo( helpMenu );

	// POPUP MENU
	if (!popup)
		popup = new QPopupMenu;
	else
		popup->clear();

	popup->insertItem( "Open", openMenu, OPEN_MENU );
	popup->insertItem( "Play", playMenu, PLAY_MENU );
	popup->insertItem( "Video", videoMenu, VIDEO_MENU );
	popup->insertItem( "Audio", audioMenu, AUDIO_MENU );
	popup->insertItem( "Subtitles", subtitlesMenu, SUBTITLES_MENU );
	popup->insertItem( "Browse", browseMenu, BROWSE_MENU );
	popup->insertItem( "Options", optionsMenu, OPTIONS_MENU );

	// let's show something, even a <empty> entry
	initializeMenus();
}

/*
void BaseGui::closeEvent( QCloseEvent * e )  {
	qDebug("BaseGui::closeEvent");

	qDebug("mplayer_log_window: %d x %d", mplayer_log_window->width(), mplayer_log_window->height() );
	qDebug("smplayer_log_window: %d x %d", smplayer_log_window->width(), smplayer_log_window->height() );

	mplayer_log_window->close();
	smplayer_log_window->close();
	playlist->close();
	equalizer->close();

	core->stop();
	e->accept();
}
*/


void BaseGui::closeWindow() {
	qDebug("BaseGui::closeWindow");

	core->stop();
	//qApp->closeAllWindows();
	qApp->quit();
}

void BaseGui::showPlaylist() {
	showPlaylist( !playlist->isVisible() );
}

void BaseGui::showPlaylist(bool b) {
	if ( !b ) {
		playlist->hide();
	} else {
		exitFullscreenIfNeeded();
		playlist->show();
	}
	updateWidgets();
}

void BaseGui::showEqualizer() {
	showEqualizer( !equalizer->isVisible() );
}

void BaseGui::showEqualizer(bool b) {
	if (!b) {
		equalizer->hide();
	} else {
		// Exit fullscreen, otherwise dialog is not visible
		exitFullscreenIfNeeded();
		equalizer->show();
	}
	updateWidgets();
}

void BaseGui::showPreferencesDialog() {
	qDebug("BaseGui::showPreferencesDialog");

	exitFullscreenIfNeeded();
	
	if (!pref_dialog) 
		createPreferencesDialog();

	if (!pref_dialog->hasDriversList()) {
		if ( !info_reader ) {
			qDebug("BaseGui::showPreferencesDialog: creating info_reader");
			info_reader = new InfoReader( pref->mplayer_bin );
			info_reader->getInfo();
		}
		pref_dialog->setDrivers( info_reader->voList(), info_reader->aoList() );
	}

	pref_dialog->setMplayerPath( pref->mplayer_bin );
	pref_dialog->setScreenshotDir( pref->screenshot_directory );
	pref_dialog->setVO( pref->vo );
	pref_dialog->setAO( pref->ao );
	pref_dialog->setEq2( pref->use_soft_video_eq );
	pref_dialog->setSoftVol( pref->use_soft_vol );
	pref_dialog->setAc3DTSPassthrough( pref->use_hwac3 );
	pref_dialog->setVolNorm( pref->use_volnorm );
	pref_dialog->setAmplification( pref->softvol_max );
	pref_dialog->setPostprocessing( pref->use_postprocessing );

	pref_dialog->setRememberSettings( !pref->dont_remember_media_settings );
	pref_dialog->setDontRememberTimePos( pref->dont_remember_time_pos );
	pref_dialog->setStartInFullscreen( pref->start_in_fullscreen );
	pref_dialog->setAudioLang( pref->audio_lang );
	pref_dialog->setSubtitleLang( pref->subtitle_lang );

	pref_dialog->setSubtitlesOnScreenshots( pref->subtitles_on_screenshots );

	pref_dialog->setDVDDevice( pref->dvd_device );
	pref_dialog->setCDRomDevice( pref->cdrom_device );

	pref_dialog->setDisableScreensaver( pref->disable_screensaver );
	pref_dialog->setResizeMethod( pref->resize_method );
	//pref_dialog->setWheelSeek( pref->wheel_seek );
	pref_dialog->setMonitorAspect( pref->monitor_aspect );

	pref_dialog->setUseSingleInstance(pref->use_single_instance);
	pref_dialog->setServerPort(pref->connection_port);
	pref_dialog->setRecentsMaxItems(pref->recents_max_items);

	pref_dialog->setSeeking1(pref->seeking1);
	pref_dialog->setSeeking2(pref->seeking2);
	pref_dialog->setSeeking3(pref->seeking3);
	pref_dialog->setSeeking4(pref->seeking4);

	pref_dialog->setLeftClickFunction( pref->mouse_left_click_function );
	pref_dialog->setDoubleClickFunction( pref->mouse_double_click_function );
	pref_dialog->setWheelFunction( pref->wheel_function );
	pref_dialog->setInitialVolume( pref->initial_volume );

	pref_dialog->setLanguage( pref->language );
	pref_dialog->setIconSet( pref->iconset );

	pref_dialog->setFontName( pref->font_name );
	pref_dialog->setFontFile( pref->font_file );
	pref_dialog->setUseFontconfig( pref->use_fontconfig );
	pref_dialog->setFontAutoscale( pref->font_autoscale );
	pref_dialog->setFontTextscale( pref->font_textscale );
	pref_dialog->setAutoloadSub( pref->autoload_sub );
	pref_dialog->setFontFuzziness( pref->subfuzziness );
	pref_dialog->setFontEncoding( pref->subcp );
	pref_dialog->setUseFontASS( pref->use_ass_subtitles );
	pref_dialog->setAssColor( pref->ass_color );
	pref_dialog->setAssBorderColor( pref->ass_border_color );
	pref_dialog->setAssStyles( pref->ass_styles );
	pref_dialog->setSubPos( pref->initial_sub_pos );
	pref_dialog->setUseSubfont( pref->use_subfont );

	pref_dialog->setCacheEnabled( pref->use_cache );
	pref_dialog->setCache( pref->cache );

	pref_dialog->setClearBackground( pref->always_clear_background );
	pref_dialog->setUseMplayerWindow( pref->use_mplayer_window );
	pref_dialog->setMplayerAdditionalArguments( pref->mplayer_additional_options );
	pref_dialog->setMplayerAdditionalVideoFilters( pref->mplayer_additional_video_filters );
	pref_dialog->setMplayerAdditionalAudioFilters( pref->mplayer_additional_audio_filters );
	pref_dialog->setColorKey( pref->color_key );

	pref_dialog->setLogMplayer( pref->log_mplayer );
	pref_dialog->setLogSmplayer( pref->log_smplayer );
	pref_dialog->setLogFilter( pref->log_filter );

	pref_dialog->setPriority( pref->priority );
	pref_dialog->setFrameDrop( pref->frame_drop );
	pref_dialog->setHardFrameDrop( pref->hard_frame_drop );
	pref_dialog->setAutoq( pref->autoq );
	pref_dialog->setAutoSyncActivated( pref->autosync );
	pref_dialog->setAutoSyncFactor( pref->autosync_factor );
	pref_dialog->setFastChapterSeeking( pref->fast_chapter_change );
	pref_dialog->setFastAudioSwitching( !pref->audio_change_requires_restart );

	pref_dialog->setEndOfFileText( pref->rx_endoffile );
	pref_dialog->setNoVideoText( pref->rx_novideo );

#if STYLE_SWITCHING
	pref_dialog->setStyle( pref->style );
#endif

	//if ( pref_dialog->actions_editor->isEmpty() ) {
	pref_dialog->actions_editor->clear();
	pref_dialog->actions_editor->addActions(this);
	pref_dialog->actions_editor->addActions(playlist);
	//}

	pref_dialog->show();
}

// The user has pressed OK in preferences dialog
void BaseGui::applyNewPreferences() {
	qDebug("BaseGui::applyNewPreferences");

	bool need_restart = false;
	bool need_update_language = false;

#define TEST_AND_SET( Pref, Dialog ) \
	if ( Pref != Dialog ) { Pref = Dialog; need_restart = TRUE; }

	TEST_AND_SET(pref->mplayer_bin, pref_dialog->mplayerPath());

	if (pref->screenshot_directory != pref_dialog->screenshotDir()) {
		pref->screenshot_directory = pref_dialog->screenshotDir();
		updateWidgets();
		need_restart = TRUE;
	}

	TEST_AND_SET(pref->vo, pref_dialog->VO());
	TEST_AND_SET(pref->ao, pref_dialog->AO());
	TEST_AND_SET(pref->use_soft_video_eq, pref_dialog->eq2());
	TEST_AND_SET(pref->use_soft_vol, pref_dialog->softVol());
	TEST_AND_SET(pref->use_hwac3, pref_dialog->Ac3DTSPassthrough());
	TEST_AND_SET(pref->use_volnorm, pref_dialog->volNorm());
	TEST_AND_SET(pref->softvol_max, pref_dialog->amplification());
	TEST_AND_SET(pref->use_postprocessing, pref_dialog->postprocessing());

	bool dont_remember_ms = !pref_dialog->rememberSettings();
	TEST_AND_SET(pref->dont_remember_media_settings, dont_remember_ms);
	TEST_AND_SET(pref->dont_remember_time_pos, pref_dialog->dontRememberTimePos());
	pref->start_in_fullscreen = pref_dialog->startInFullscreen();
	pref->audio_lang = pref_dialog->audioLang();
	pref->subtitle_lang = pref_dialog->subtitleLang();
	TEST_AND_SET(pref->subtitles_on_screenshots, pref_dialog->subtitlesOnScreenshots());

	pref->dvd_device = pref_dialog->dvdDevice();
	pref->cdrom_device = pref_dialog->cdromDevice();

	TEST_AND_SET(pref->disable_screensaver, pref_dialog->disableScreensaver());
	pref->resize_method = pref_dialog->resizeMethod();
	TEST_AND_SET(pref->monitor_aspect, pref_dialog->monitorAspect());
	mplayerwindow->setMonitorAspect( pref->monitor_aspect_double() );

	pref->use_single_instance = pref_dialog->useSingleInstance();
	pref->connection_port = pref_dialog->serverPort();

	if (pref->recents_max_items != pref_dialog->recentsMaxItems()) {
		pref->recents_max_items = pref_dialog->recentsMaxItems();
		recents->setMaxItems(pref->recents_max_items);
		updateRecents();
	}

	pref->seeking1 = pref_dialog->seeking1();
	pref->seeking2 = pref_dialog->seeking2();
	pref->seeking3 = pref_dialog->seeking3();
	pref->seeking4 = pref_dialog->seeking4();
	setJumpTexts(); // Update texts in menus

	pref->mouse_left_click_function = pref_dialog->leftClickFunction();
	pref->mouse_double_click_function = pref_dialog->doubleClickFunction();
	pref->wheel_function = pref_dialog->wheelFunction();
	pref->initial_volume = pref_dialog->initialVolume();

	if (pref->language != pref_dialog->language()) {
		pref->language = pref_dialog->language();
		need_update_language = true;
		qDebug("BaseGui::applyNewPreferences: chosen language: '%s'", pref->language.utf8().data());
	}

	if (pref->iconset != pref_dialog->iconSet()) {
		pref->iconset = pref_dialog->iconSet();
		need_update_language = true; // Not a mistake
		// Stylesheet
		#if QT_VERSION >= 0x040200
		changeStyleSheet(pref->iconset);
		#endif
	}

	TEST_AND_SET(pref->font_name, pref_dialog->fontName());
	TEST_AND_SET(pref->font_file, pref_dialog->fontFile());
	TEST_AND_SET(pref->use_fontconfig, pref_dialog->useFontconfig());
	TEST_AND_SET(pref->font_autoscale, pref_dialog->fontAutoscale());
	TEST_AND_SET(pref->font_textscale, pref_dialog->fontTextscale());
	TEST_AND_SET(pref->autoload_sub, pref_dialog->autoloadSub());
	TEST_AND_SET(pref->subfuzziness, pref_dialog->fontFuzziness());
	TEST_AND_SET(pref->subcp, pref_dialog->fontEncoding());
	TEST_AND_SET(pref->use_ass_subtitles, pref_dialog->useFontASS());
	TEST_AND_SET(pref->ass_color, pref_dialog->assColor());
	TEST_AND_SET(pref->ass_border_color, pref_dialog->assBorderColor());
	TEST_AND_SET(pref->ass_styles, pref_dialog->assStyles());
	pref->initial_sub_pos = pref_dialog->subPos();
	TEST_AND_SET(pref->use_subfont, pref_dialog->useSubfont());

	if (pref->always_clear_background != pref_dialog->clearBackground()) {
		pref->always_clear_background = pref_dialog->clearBackground();
		core->prepareMplayerWindow();
	}

	TEST_AND_SET(pref->use_cache, pref_dialog->cacheEnabled());
	TEST_AND_SET(pref->cache, pref_dialog->cache());

	TEST_AND_SET(pref->use_mplayer_window, pref_dialog->useMplayerWindow());
	TEST_AND_SET(pref->mplayer_additional_options, pref_dialog->mplayerAdditionalArguments());
	TEST_AND_SET(pref->mplayer_additional_video_filters, pref_dialog->mplayerAdditionalVideoFilters());
	TEST_AND_SET(pref->mplayer_additional_audio_filters, pref_dialog->mplayerAdditionalAudioFilters());
	if (pref->color_key != pref_dialog->colorKey()) {
		pref->color_key = pref_dialog->colorKey();
		mplayerwindow->setColorKey( pref->color_key );
		need_restart = true;
	}

	pref->log_mplayer = pref_dialog->logMplayer();
	pref->log_smplayer = pref_dialog->logSmplayer();
	pref->log_filter = pref_dialog->logFilter();

	TEST_AND_SET(pref->priority, pref_dialog->priority());
	TEST_AND_SET(pref->frame_drop, pref_dialog->frameDrop());
	TEST_AND_SET(pref->hard_frame_drop, pref_dialog->hardFrameDrop());
	TEST_AND_SET(pref->autoq, pref_dialog->autoq());
	TEST_AND_SET(pref->autosync, pref_dialog->autoSyncActivated());
	TEST_AND_SET(pref->autosync_factor, pref_dialog->autoSyncFactor());
	TEST_AND_SET(pref->fast_chapter_change, pref_dialog->fastChapterSeeking());
	bool audio_requires_restart = !pref_dialog->fastAudioSwitching();
	TEST_AND_SET(pref->audio_change_requires_restart, audio_requires_restart);

	TEST_AND_SET(pref->rx_endoffile, pref_dialog->endOfFileText());
	TEST_AND_SET(pref->rx_novideo, pref_dialog->noVideoText());

#if STYLE_SWITCHING
	if ( pref->style != pref_dialog->style() ) {
		pref->style = pref_dialog->style();
		qDebug( "selected style: '%s'", pref->style.utf8().data() );
		if ( !pref->style.isEmpty()) {
			qApp->setStyle( pref->style );
		} else {
			qDebug("setting default style: '%s'", default_style.utf8().data() );
			qApp->setStyle( default_style );
		}
	}
#endif

	pref_dialog->actions_editor->applyChanges();
	saveActions();

	if (need_update_language) {
		translator->load(pref->language);
	}

	// Restart the video to apply
	if (need_restart)
		core->restart();

	pref->save();
}


void BaseGui::showFilePropertiesDialog() {
	qDebug("BaseGui::showFilePropertiesDialog");

	exitFullscreenIfNeeded();

	if (!file_dialog) 
		createFilePropertiesDialog();

	if (!file_dialog->hasCodecsList()) {
		if ( !info_reader ) {
			qDebug("BaseGui::showFilePropertiesDialog: creating info_reader");
			info_reader = new InfoReader( pref->mplayer_bin );
			info_reader->getInfo();
		}
		file_dialog->setCodecs( info_reader->vcList(), 
                                info_reader->acList(), 
                                info_reader->demuxerList() );
	}

	setDataToFileProperties();

	file_dialog->show();
}

void BaseGui::setDataToFileProperties() {
	// Save a copy of the original values
	if (core->mset.original_demuxer.isEmpty()) 
		core->mset.original_demuxer = core->mdat.demuxer;

	if (core->mset.original_video_codec.isEmpty()) 
		core->mset.original_video_codec = core->mdat.video_codec;

	if (core->mset.original_audio_codec.isEmpty()) 
		core->mset.original_audio_codec = core->mdat.audio_codec;

	QString demuxer = core->mset.forced_demuxer;
	if (demuxer.isEmpty()) demuxer = core->mdat.demuxer;

	QString ac = core->mset.forced_audio_codec;
	if (ac.isEmpty()) ac = core->mdat.audio_codec;

	QString vc = core->mset.forced_video_codec;
	if (vc.isEmpty()) vc = core->mdat.video_codec;

	file_dialog->setDemuxer(demuxer, core->mset.original_demuxer);
	file_dialog->setAudioCodec(ac, core->mset.original_audio_codec);
	file_dialog->setVideoCodec(vc, core->mset.original_video_codec);

	file_dialog->setMplayerAdditionalArguments( core->mset.mplayer_additional_options );
	file_dialog->setMplayerAdditionalVideoFilters( core->mset.mplayer_additional_video_filters );
	file_dialog->setMplayerAdditionalAudioFilters( core->mset.mplayer_additional_audio_filters );

	file_dialog->setMediaData( core->mdat );
}

void BaseGui::applyFileProperties() {
	qDebug("BaseGui::applyFileProperties");

	bool need_restart = false;

	QString demuxer = file_dialog->demuxer();
	if (demuxer == core->mset.original_demuxer) demuxer="";
	TEST_AND_SET(core->mset.forced_demuxer, demuxer);

	QString ac = file_dialog->audioCodec();
	if (ac == core->mset.original_audio_codec) ac="";
	TEST_AND_SET(core->mset.forced_audio_codec, ac);

	QString vc = file_dialog->videoCodec();
	if (vc == core->mset.original_video_codec) vc="";
	TEST_AND_SET(core->mset.forced_video_codec, vc);

	TEST_AND_SET(core->mset.mplayer_additional_options, file_dialog->mplayerAdditionalArguments());
	TEST_AND_SET(core->mset.mplayer_additional_video_filters, file_dialog->mplayerAdditionalVideoFilters());
	TEST_AND_SET(core->mset.mplayer_additional_audio_filters, file_dialog->mplayerAdditionalAudioFilters());

	// Restart the video to apply
	if (need_restart) {
		core->restart();
	}
}


void BaseGui::updateMediaInfo() {
    qDebug("BaseGui::updateMediaInfo");

	if (file_dialog) {
		if (file_dialog->isVisible()) setDataToFileProperties();
	}

	setWindowCaption( core->mdat.displayName() + " - SMPlayer" );
}

void BaseGui::newMediaLoaded() {
    qDebug("BaseGui::newMediaLoaded");

	recents->add( core->mdat.filename );
	updateRecents();

	// If a VCD, Audio CD or DVD, add items to playlist
	if ( (core->mdat.type == TYPE_VCD) || (core->mdat.type == TYPE_DVD) || 
         (core->mdat.type == TYPE_AUDIO_CD) ) 
	{
		int first_title = 1;
		if (core->mdat.type == TYPE_VCD) first_title = pref->vcd_initial_title;

		QString type = "dvd";
		if (core->mdat.type == TYPE_VCD) type="vcd";
		else
		if (core->mdat.type == TYPE_AUDIO_CD) type="cdda";

		if (core->mset.current_title_id == first_title) {
			playlist->clear();
			QStringList l;
			QString s;
			QString folder;
			if (core->mdat.type == TYPE_DVD) {
				folder = Helper::dvdSplitFolder( core->mdat.filename );
			}
			for (int n=0; n < core->mdat.titles.numItems(); n++) {
				s = type + "://" + QString::number(core->mdat.titles.itemAt(n).ID());
				if ( !folder.isEmpty() ) {
					s += ":" + folder;
				}
				l.append(s);
			}
			playlist->addFiles(l);
			//playlist->setModified(false); // Not a real playlist
		}
	}
}

void BaseGui::showMplayerLog() {
    qDebug("BaseGui::showMplayerLog");

	exitFullscreenIfNeeded();

    mplayer_log_window->setText( core->mplayer_log );
	mplayer_log_window->show();
}

void BaseGui::showLog() {
    qDebug("BaseGui::showLog");

	exitFullscreenIfNeeded();

	smplayer_log_window->setText( Helper::log() );
    smplayer_log_window->show();
}


void BaseGui::initializeMenus() {
	qDebug("BaseGui::initializeMenus");

#define EMPTY 1

	int n;

	// Subtitles
    subtitlestrack_menu->clear();
    subtitlestrack_menu->insertItem( tr("&None"), MediaSettings::SubNone );
#if SUBTITLES_BY_INDEX
	for (n=0; n < core->mdat.subs.numItems(); n++) {
		subtitlestrack_menu->insertItem( core->mdat.subs.itemAt(n).displayName(), n);
	}
#else
	for (n=0; n < core->mdat.subtitles.numItems(); n++) {
		qDebug("BaseGui::initializeMenus: n: %d", n);
		subtitlestrack_menu->insertItem( core->mdat.subtitles.itemAt(n).displayName(), core->mdat.subtitles.itemAt(n).ID() );
	}
#endif

	// Audio
	audiotrack_menu->clear();
	if (core->mdat.audios.numItems()==0) {
		audiotrack_menu->insertItem( tr("<empty>"), EMPTY );
		audiotrack_menu->setItemEnabled(EMPTY, FALSE);
	} else {
		for (n=0; n < core->mdat.audios.numItems(); n++) {
			audiotrack_menu->insertItem( core->mdat.audios.itemAt(n).displayName(), core->mdat.audios.itemAt(n).ID() );
		}
	}

	// Titles
	titles_menu->clear();
	if (core->mdat.titles.numItems()==0) {
		titles_menu->insertItem( tr("<empty>"), EMPTY );
		titles_menu->setItemEnabled(EMPTY, FALSE);
	} else {
		for (n=0; n < core->mdat.titles.numItems(); n++) {
			titles_menu->insertItem( core->mdat.titles.itemAt(n).displayName(), core->mdat.titles.itemAt(n).ID() );
		}
	}

	// DVD Chapters
	chapters_menu->clear();
	if ( (core->mdat.type == TYPE_DVD) && (core->mset.current_title_id > 0) ) {
		for (n=1; n <= core->mdat.titles.item(core->mset.current_title_id).chapters(); n++) {
			chapters_menu->insertItem( QString::number( n ), n );
		}
	} else {
		// *** Matroshka chapters ***
		if (core->mdat.mkv_chapters > 0) {
			for (n=0; n <= core->mdat.mkv_chapters; n++) {
				chapters_menu->insertItem( QString::number( n+1 ), n );
			}
		} else {
			chapters_menu->insertItem( tr("<empty>"), EMPTY );
			chapters_menu->setItemEnabled(EMPTY, FALSE);
		}
	}

	// Angles
	angles_menu->clear();
	if (core->mset.current_angle_id > 0) {
		for (n=1; n <= core->mdat.titles.item(core->mset.current_title_id).angles(); n++) {
			angles_menu->insertItem( QString::number( n ), n );
		}
	} else {
		angles_menu->insertItem( tr("<empty>"), EMPTY );
		angles_menu->setItemEnabled(EMPTY, FALSE);
	}
}

void BaseGui::updateRecents() {
	qDebug("BaseGui::updateRecents");

	//recentfiles_menu->clear();
	// Not clear the first 2 items
	while (recentfiles_menu->count() > 2) 
		recentfiles_menu->removeItemAt(2);

	if (recents->count() > 0) {
		int max_items = recents->count();
		if (max_items > pref->recents_max_items) {
			max_items=pref->recents_max_items;
		}
		for (int n=0; n < max_items; n++) {
			QString file = recents->item(n);
			QFileInfo fi(file);
			if (fi.exists()) file = fi.fileName();
			recentfiles_menu->insertItem( file, n );
		}
	} else {
		recentfiles_menu->insertItem( tr("<empty>"), EMPTY );
		recentfiles_menu->setItemEnabled(EMPTY, FALSE);
	}
}

void BaseGui::clearRecentsList() {
	// Delete items in menu
	/*
	for (int n=0; n < recents->count(); n++) {
		qDebug("borrando item %d de %d", n, recents->count());
		recentfiles_menu->removeItem(n);
	}
	*/

	recents->clear();
	updateRecents();
}

void BaseGui::updateWidgets() {
	qDebug("BaseGui::updateWidgets");

	// Subtitles menu
	subtitlestrack_menu->setChecked( core->mset.current_sub_id );

#if SUBTITLES_BY_INDEX
	// Disable the unload subs action if there's no external subtitles
	unloadSubsAct->setEnabled( !core->mset.external_subtitles.isEmpty() );
#else
	// If using an external subtitles, disable the rest
	bool b = core->mset.external_subtitles.isEmpty();
    for (unsigned int n=1; n < subtitlestrack_menu->count(); n++) {
		subtitlestrack_menu->setItemEnabled(subtitlestrack_menu->idAt(n),b);
	}
#endif
	
	// Audio menu
	audiotrack_menu->setChecked( core->mset.current_audio_id );
	audiochannels_menu->setChecked( core->mset.audio_use_channels );
	stereomode_menu->setChecked( core->mset.stereo_mode );

	// Disable the unload audio file action if there's no external audio file
	unloadAudioAct->setEnabled( !core->mset.external_audio.isEmpty() );

	// Aspect ratio
	aspect_menu->setChecked( core->mset.aspect_ratio_id );

	// OSD
	osd_menu->setChecked( pref->osd );

	// Titles
	titles_menu->setChecked( core->mset.current_title_id );

	// Chapters
	chapters_menu->setChecked( core->mset.current_chapter_id );

	// Angles
	angles_menu->setChecked( core->mset.current_angle_id );

	// Deinterlace menu
	deinterlace_menu->setChecked( core->mset.current_deinterlacer );

	// Video size menu
	videosize_menu->setChecked( pref->size_factor );

	// Auto phase
	phaseAct->setOn( core->mset.phase_filter );

	// Deblock
	deblockAct->setOn( core->mset.deblock_filter );

	// Dering
	deringAct->setOn( core->mset.dering_filter );

	// Add noise
	addNoiseAct->setOn( core->mset.noise_filter );

	// Postprocessing
	postProcessingAct->setOn( core->mset.postprocessing_filter );

	// Denoise submenu
	denoise_menu->setChecked( core->mset.current_denoiser );

	/*
	// Fullscreen button
	fullscreenbutton->setOn(pref->fullscreen); 

	// Mute button
	mutebutton->setOn(core->mset.mute);
	if (core->mset.mute) 
		mutebutton->setPixmap( Images::icon("mute_small") );
	else
		mutebutton->setPixmap( Images::icon("volume_small") );

	// Volume slider
	volumeslider->setValue( core->mset.volume );
	*/

	// Mute menu option
	muteAct->setOn( core->mset.mute );

	// Karaoke menu option
	karaokeAct->setOn( core->mset.karaoke_filter );

	// Extrastereo menu option
	extrastereoAct->setOn( core->mset.extrastereo_filter );

	// Volnorm menu option
	volnormAct->setOn( core->mset.volnorm_filter );

	// Repeat menu option
	repeatAct->setOn( pref->loop );

	// Fullscreen action
	fullscreenAct->setOn( pref->fullscreen );

	// Time slider
	if (core->state==STOPPED) {
		//FIXME
		//timeslider->setValue( (int) core->mset.current_sec );
	}

	// Video equalizer
	equalizerAct->setOn( equalizer->isVisible() );

	// Playlist
#if !DOCK_PLAYLIST
	showPlaylistAct->setOn( playlist->isVisible() );
#endif

	// Frame counter
	frameCounterAct->setOn( pref->show_frame_counter );

	// Compact mode
	compactAct->setOn( pref->compact_mode );

	// Stay on top
	onTopAct->setOn( pref->stay_on_top );

	// Screenshot option
	bool valid_directory = ( (!pref->screenshot_directory.isEmpty()) &&
                             (QFileInfo(pref->screenshot_directory).isDir()) );
	screenshotAct->setEnabled( valid_directory );


	// Enable or disable subtitle options
	bool e = !(core->mset.current_sub_id == MediaSettings::SubNone);
	decSubDelayAct->setEnabled(e);
	incSubDelayAct->setEnabled(e);
	decSubPosAct->setEnabled(e);
	incSubPosAct->setEnabled(e);
	decSubStepAct->setEnabled(e);
	incSubStepAct->setEnabled(e);
}

void BaseGui::updateEqualizer() {
	// Equalizer
	equalizer->contrast->setValue( core->mset.contrast );
	equalizer->brightness->setValue( core->mset.brightness );
	equalizer->hue->setValue( core->mset.hue );
	equalizer->saturation->setValue( core->mset.saturation );
	equalizer->gamma->setValue( core->mset.gamma );
}


void BaseGui::playlistVisibilityChanged() {
#if !DOCK_PLAYLIST
	bool visible = playlist->isVisible();

	showPlaylistAct->setOn( visible );
#endif
}

void BaseGui::openRecent(int item) {
	qDebug("BaseGui::openRecent: %d", item);
	if ((item > -1) && (item < RECENTS_CLEAR)) { // 1000 = Clear item
		open( recents->item(item) );
	}
}

void BaseGui::open(QString file) {
	// If file is a playlist, open that playlist
	QString extension = QFileInfo(file).extension(FALSE).lower();
	if ( (extension=="m3u") || (extension=="m3u8") ) {
		playlist->load_m3u(file);
	} 
	else 
	if (QFileInfo(file).isDir()) {
		openDirectory(file);
	} 
	else {
		// Let the core to open it, autodetecting the file type
		core->open(file);
	}
}

void BaseGui::openFiles(QStringList files) {
	qDebug("BaseGui::openFiles");
	if (files.empty()) return;

	if (files.count()==1) {
		open(files[0]);
	} else {
		if (playlist->maybeSave()) {
			playlist->clear();
			playlist->addFiles(files);
			open(files[0]);
		}
	}
}

void BaseGui::openURL() {
	qDebug("BaseGui::openURL");

	exitFullscreenIfNeeded();

    bool ok;
    QString s = QInputDialog::getText(
            "SMplayer", "URL:", QLineEdit::Normal,
            pref->last_url, &ok, this );

    if ( ok && !s.isEmpty() ) {

		//playlist->clear();
		//playlistdock->hide();

		openURL(s);
    } else {
        // user entered nothing or pressed Cancel
    }
}

void BaseGui::openURL(QString url) {
	if (!url.isEmpty()) {
		pref->last_url = url;
		core->openStream(url);
	}
}


void BaseGui::openFile() {
	qDebug("BaseGui::fileOpen");

	exitFullscreenIfNeeded();

    QString s = MyFileDialog::getOpenFileName(pref->latest_dir, 
                       tr("Video") +" (*.avi *.mpg *.mpeg *.mkv *.wmv "
                       "*.ogm *.vob *.flv *.mov *.ts *.rmvb *.mp4 *.iso);;" +
                       tr("Audio") +" (*.mp3 *.ogg *.wav *.wma *.ac3);;" +
                       tr("Playlists") +" (*.m3u *.m3u8);;" +
                       tr("All files") +" (*.*)", +
                        this, 
                        "open file dialog", 
                        tr("Choose a file") );

    if ( !s.isEmpty() ) {
		openFile(s);
	}
}

void BaseGui::openFile(QString file) {
   if ( !file.isEmpty() ) {

		//playlist->clear();
		//playlistdock->hide();

		// If file is a playlist, open that playlist
		QString extension = QFileInfo(file).extension(FALSE).lower();
		if ( (extension=="m3u") || (extension=="m3u8") ) {
			playlist->load_m3u(file);
		} 
		else
		if (extension=="iso") {
			if (playlist->maybeSave()) {
				core->open(file);
			}
		}
		else {
			pref->latest_dir = QFileInfo(file).dirPath(TRUE);
			core->openFile(file);
		}
	}
}

void BaseGui::openVCD() {
	qDebug("BaseGui::openVCD");

	if ( (pref->dvd_device.isEmpty()) || 
         (pref->cdrom_device.isEmpty()) )
	{
		QMessageBox::information( this, tr("SMPlayer - Information"),
			tr("The CDROM / DVD drives are not configured yet.\n"
			   "The configuration dialog will be shown now, "
               "so you can do it."), QMessageBox::Ok);
	
		showPreferencesDialog();
		pref_dialog->showSection( PreferencesDialog::Drives );
	} else {
		if (playlist->maybeSave()) {
			core->openVCD( pref->vcd_initial_title );
		}
	}
}

void BaseGui::openAudioCD() {
	qDebug("BaseGui::openAudioCD");

	if ( (pref->dvd_device.isEmpty()) || 
         (pref->cdrom_device.isEmpty()) )
	{
		QMessageBox::information( this, tr("SMPlayer - Information"),
			tr("The CDROM / DVD drives are not configured yet.\n"
			   "The configuration dialog will be shown now, "
               "so you can do it."), QMessageBox::Ok);
	
		showPreferencesDialog();
		pref_dialog->showSection( PreferencesDialog::Drives );
	} else {
		if (playlist->maybeSave()) {
			core->openAudioCD();
		}
	}
}

void BaseGui::openDVD() {
	qDebug("BaseGui::openDVD");

	if ( (pref->dvd_device.isEmpty()) || 
         (pref->cdrom_device.isEmpty()) )
	{
		QMessageBox::information( this, tr("SMPlayer - Information"),
			tr("The CDROM / DVD drives are not configured yet.\n"
			   "The configuration dialog will be shown now, "
               "so you can do it."), QMessageBox::Ok);
	
		showPreferencesDialog();
		pref_dialog->showSection( PreferencesDialog::Drives );
	} else {
		if (playlist->maybeSave()) {
			core->openDVD("dvd://1");
		}
	}
}

void BaseGui::openDVDFromFolder() {
	qDebug("BaseGui::openDVDFromFolder");

	if (playlist->maybeSave()) {
		InputDVDDirectory *d = new InputDVDDirectory(this);
		d->setFolder( pref->last_dvd_directory );

		if (d->exec() == QDialog::Accepted) {
			qDebug("BaseGui::openDVDFromFolder: accepted");
			openDVDFromFolder( d->folder() );
		}

		delete d;
	}
}

void BaseGui::openDVDFromFolder(QString directory) {
	//core->openDVD(TRUE, directory);
	pref->last_dvd_directory = directory;
	core->openDVD( "dvd://1:" + directory);
}

void BaseGui::openDirectory() {
	qDebug("BaseGui::openDirectory");

	QString s = MyFileDialog::getExistingDirectory(
                    pref->latest_dir,
                    this,
                    "get existing directory",
                    tr("Choose a directory"),
                    TRUE );

	if (!s.isEmpty()) {
		openDirectory(s);
	}
}

void BaseGui::openDirectory(QString directory) {
	qDebug("BaseGui::openDirectory: '%s'", directory.utf8().data());

	if (Helper::directoryContainsDVD(directory)) {
		core->open(directory);
	} 
	else {
		QFileInfo fi(directory);
		if ( (fi.exists()) && (fi.isDir()) ) {
			playlist->clear();
			//playlist->addDirectory(directory);
			playlist->addDirectory( fi.absFilePath() );
			playlist->startPlay();
		} else {
			qDebug("BaseGui::openDirectory: directory is not valid");
		}
	}
}

void BaseGui::loadSub() {
	qDebug("BaseGui::loadSub");

	exitFullscreenIfNeeded();

    QString s = MyFileDialog::getOpenFileName(
	    pref->latest_dir, 
        tr("Subtitles") +" (*.srt *.sub *.ssa *.ass *.idx"
                         " *.txt *.smi *.rt *.utf *.aqt);;" +
        tr("All files") +" (*.*)", 
        this, "open_file_dialog", tr("Choose a file") );

	if (!s.isEmpty()) core->loadSub(s);
}

void BaseGui::loadAudioFile() {
	qDebug("BaseGui::loadAudioFile");

	exitFullscreenIfNeeded();

	QString s = MyFileDialog::getOpenFileName(
	    pref->latest_dir, 
        tr("Audio") +" (*.mp3 *.ogg *.wav *.wma *.ac3);;" +
        tr("All files") +" (*.*)", 
        this, "open_file_dialog", tr("Choose a file") );

	if (!s.isEmpty()) core->loadAudioFile(s);
}

void BaseGui::helpAbout() {
	AboutDialog d(this);
	d.exec();
}

void BaseGui::helpAboutQt() {
	QMessageBox::aboutQt(this, tr("About Qt") );
}

void BaseGui::exitFullscreen() {
	if (pref->fullscreen) {
		toggleFullscreen(false);
	}
}

void BaseGui::toggleFullscreen() {
	qDebug("BaseGui::toggleFullscreen");

	toggleFullscreen(!pref->fullscreen);
}

void BaseGui::toggleFullscreen(bool b) {
	qDebug("BaseGui::toggleFullscreen: %d", b);

	if (b==pref->fullscreen) {
		// Nothing to do
		return;
	}

	pref->fullscreen = b;

	// If using mplayer window
	if (pref->use_mplayer_window) {
		core->tellmp("vo_fullscreen " + QString::number(b) );
		updateWidgets();
		return;
	}

	if (!panel->isVisible()) return; // mplayer window is not used.


	if (pref->fullscreen) {
		#ifdef Q_OS_WIN
		win_pos = pos();
		win_size = size();
		hide();
		#endif

		aboutToEnterFullscreen();
		showFullScreen();
	} else {
		#ifdef Q_OS_WIN
		hide();
		#endif

		aboutToExitFullscreen();
		showNormal();

		#ifdef Q_OS_WIN
		move( win_pos );
		resize( win_size );
		#endif
	}

	updateWidgets();
}


void BaseGui::aboutToEnterFullscreen() {
	if (!pref->compact_mode) {
		menuBar()->hide();
		statusBar()->hide();
	}
}

void BaseGui::aboutToExitFullscreen() {
	if (!pref->compact_mode) {
		menuBar()->show();
		statusBar()->show();
	}
}

void BaseGui::toggleFrameCounter() {
	toggleFrameCounter( !pref->show_frame_counter );
}

void BaseGui::toggleFrameCounter(bool b) {
    pref->show_frame_counter = b;
    updateWidgets();
}


void BaseGui::leftClickFunction() {
	qDebug("BaseGui::leftClickFunction");

	if (!pref->mouse_left_click_function.isEmpty()) {
		processFunction(pref->mouse_left_click_function);
	}
}

void BaseGui::doubleClickFunction() {
	qDebug("BaseGui::doubleClickFunction");

	if (!pref->mouse_double_click_function.isEmpty()) {
		processFunction(pref->mouse_double_click_function);
	}
}

void BaseGui::processFunction(QString function) {
	qDebug("BaseGui::processFunction: '%s'", function.utf8().data());

	QACTION * action = ActionsEditor::findAction(this, function);
	if (!action) action = ActionsEditor::findAction(playlist, function);

	if (action) {
		qDebug("BaseGui::processFunction: action found");
		if (action->isToggleAction()) 
			action->toggle();
		else
			action->activate();
	}
}

void BaseGui::dragEnterEvent( QDragEnterEvent *e ) {
	qDebug("BaseGui::dragEnterEvent");
	//qDebug(" text: %d", QTextDrag::canDecode(e));
	#if QT_VERSION < 0x040000
	e->accept( (QUriDrag::canDecode(e) || QTextDrag::canDecode(e)) );
	#else
	e->accept( QUriDrag::canDecode(e) );
	#endif
}



void BaseGui::dropEvent( QDropEvent *e ) {
	qDebug("BaseGui::dropEvent");

	QStringList files;

	if (QUriDrag::canDecode(e)) {
		QStrList l;
		//QStringList l;
		QUriDrag::decode(e, l);

		QString s;
        for ( unsigned int i= 0; i < l.count(); ++i ) {
			//s = l[i];
			s = l.at(i); 

			qDebug(" s: '%s'", s.utf8().data() );
#if KDE_SUPPORT
			KURL u(s);
#else
			QUrl u(s);
#endif
			qDebug(" * protocol: '%s'", u.protocol().utf8().data());
			qDebug(" * path: '%s'", u.path().utf8().data());
			//qDebug(" * path (utf8): '%s'", u.toString().utf8().data());
			//qDebug(" filename:='%s'", u.fileName().utf8().data());
			if (u.protocol()=="file") {
				s = u.path();
			}
			files.append( s );
		}
	}

	#if QT_VERSION < 0x040000
	else
	if (QTextDrag::canDecode(e)) {
		QString s;
		if (QTextDrag::decode(e, s)) {
			qDebug(" * '%s'", s.utf8().data() );
			files.append( s );
		}
	}
	#endif

	qDebug( " count: %d", files.count());
	if (files.count() > 0) {
		if (files.count() == 1) {
			QFileInfo fi( files[0] );

			QRegExp ext_sub("srt|sub|ssa|ass|idx|txt|smi|rt|utf|aqt");
			ext_sub.setCaseSensitive(false);
			if (ext_sub.search(fi.extension(false)) > -1) {
				core->loadSub( files[0] );
			}
			else
			if (fi.isDir()) {
				openDirectory( files[0] );
			} else {
				//openFile( files[0] );
				open( files[0] );
			}
		} else {
			playlist->clear();
			playlist->addFiles(files);
			//openFile( files[0] );
			playlist->startPlay();
		}
	}
}

void BaseGui::showPopupMenu( QPoint p ) {
	qDebug("BaseGui::showPopupMenu");

	popup->move( p );
	popup->show();
}

void BaseGui::mouseReleaseEvent( QMouseEvent * e ) {
	qDebug("BaseGui::mouseReleaseEvent");

	if (e->button() == Qt::LeftButton) {
		e->accept();
		emit leftClicked();
	}
	/*
	else
	if (e->button() == Qt::RightButton) {
		showPopupMenu( e->globalPos() );
    }
	*/
	else 
		e->ignore();
}

void BaseGui::mouseDoubleClickEvent( QMouseEvent * e ) {
	e->accept();
	emit doubleClicked();
}

void BaseGui::wheelEvent( QWheelEvent * e ) {
	qDebug("BaseGui::wheelEvent: delta: %d", e->delta());
	e->accept();

	if (e->delta() >= 0) 
		emit wheelUp();
	else
		emit wheelDown();
}


// Called when a video has started to play
void BaseGui::enterFullscreenOnPlay() {
	if ( (pref->start_in_fullscreen) && (!pref->fullscreen) ) {
		toggleFullscreen(TRUE);
	}
}

// Called when the playlist has stopped
void BaseGui::exitFullscreenOnStop() {
    if (pref->fullscreen) {
		toggleFullscreen(FALSE);
	}
}

void BaseGui::displayState(int state) {
	qDebug("BaseGui::displayState: %d", state);
	switch (state) {
		case PLAYING:	statusBar()->message( tr("Playing %1").arg(core->mdat.filename), 2000); break;
		case PAUSED:	statusBar()->message( tr("Pause") ); break;
		case STOPPED:	statusBar()->message( tr("Stop") , 2000); break;
	}
	if (state == STOPPED) setWindowCaption( "SMPlayer" );
}

void BaseGui::displayMessage(QString message) {
	statusBar()->message(message, 2000);
}

void BaseGui::gotCurrentTime(double sec) {
	//qDebug( "DefaultGui::displayTime: %f", sec);

	if (floor(sec)==last_second) return; // Update only once per second
	last_second = (int) floor(sec);

	QString time = Helper::formatTime( (int) sec ) + " / " +
                           Helper::formatTime( (int) core->mdat.duration );

	//qDebug( " duration: %f, current_sec: %f", core->mdat.duration, core->mset.current_sec);

	int perc = 0;
    //Update slider
	if ( (core->mdat.duration > 1) && (core->mset.current_sec > 1) && 
         (core->mdat.duration > core->mset.current_sec) ) 
	{
	    perc = ( (int) core->mset.current_sec * 100) / (int) core->mdat.duration;
	}

	emit timeChanged( sec, perc, time );
}


void BaseGui::resizeWindow(int w, int h) {
	qDebug("BaseGui::resizeWindow: %d, %d", w, h);

	// If fullscreen, don't resize!
	if (pref->fullscreen) return;

	if ( (pref->resize_method==Preferences::Never) && (panel->isVisible()) ) {
		return;
	}

	if (!panel->isVisible()) {
		//hide();
		panel->show();
		QPoint p = pos();
		adjustSize();
		move(p);
		//show();
	}

	if (pref->size_factor != 100) {
		double zoom = (double) pref->size_factor/100;
		w = w * zoom;
		h = h * zoom;
	}

	int width = size().width() - panel->size().width();
	int height = size().height() - panel->size().height();

	width += w;
	height += h;

	resize(width,height);

	qDebug("width: %d, height: %d", width, height);
	qDebug("mplayerwindow->size: %d, %d", 
           mplayerwindow->size().width(),  
           mplayerwindow->size().height() );

	mplayerwindow->setFocus(); // Needed?
}

void BaseGui::hidePanel() {
	qDebug("BaseGui::hidePanel");

	if (panel->isVisible()) {
		// Exit from compact mode first
		if (pref->compact_mode) toggleCompactMode(false);

		//resizeWindow( size().width(), 0 );
		resize( size().width(), size().height() - panel->size().height() );
		panel->hide();
	}
}

void BaseGui::displayGotoTime(int t) {
	int jump_time = (int)core->mdat.duration * t / 100;
	//QString s = tr("Jump to %1").arg( Helper::formatTime(jump_time) );
	QString s = QString("Jump to %1").arg( Helper::formatTime(jump_time) );
	statusBar()->message( s, 1000 );

	if (pref->fullscreen) {
		core->tellmp("osd_show_text \"" + s + "\" 3000 1");
	}
}

void BaseGui::toggleCompactMode() {
	toggleCompactMode( !pref->compact_mode );
}

void BaseGui::toggleCompactMode(bool b) {
	qDebug("BaseGui::toggleCompactMode: %d", b);

	if (b) 
		aboutToEnterCompactMode();
	else
		aboutToExitCompactMode();

	pref->compact_mode = b;
	updateWidgets();
}

void BaseGui::aboutToEnterCompactMode() {
	menuBar()->hide();
	statusBar()->hide();
}

void BaseGui::aboutToExitCompactMode() {
	menuBar()->show();
	statusBar()->show();
}


void BaseGui::toggleStayOnTop() {
	toggleStayOnTop( !pref->stay_on_top );
}

void BaseGui::toggleStayOnTop(bool b) {
	bool visible = isVisible();

	QPoint old_pos = pos();

	if (b) {
		#if QT_VERSION >= 0x040000
		setWindowFlags(Qt::WStyle_StaysOnTop);
		#else
		reparent(0, Qt::WStyle_StaysOnTop, pos() );
		#endif
	}
	else {
		#if QT_VERSION >= 0x040000
		setWindowFlags(0);
		#else
		reparent(0, 0, pos() );
		#endif
	}

	move(old_pos);

	if (visible) {
		show();
	}

	pref->stay_on_top = b;

	updateWidgets();
}


// Called when a new window (equalizer, preferences..) is opened.
void BaseGui::exitFullscreenIfNeeded() {
	/*
	if (pref->fullscreen) {
		toggleFullscreen(FALSE);
	}
	*/
}

void BaseGui::checkMousePos(QPoint p) {
	//qDebug("BaseGui::checkMousePos: %d, %d", p.x(), p.y());

	if (!pref->fullscreen) return;

	#define MARGIN 70
	if (p.y() > mplayerwindow->height() - MARGIN) {
		qDebug("BaseGui::checkMousePos: %d, %d", p.x(), p.y());
		if (!near_bottom) {
			emit cursorNearBottom(p);
			near_bottom = true;
		}
	} else {
		if (near_bottom) {
			emit cursorFarEdges();
			near_bottom = false;
		}
	}

	if (p.y() < MARGIN) {
		qDebug("BaseGui::checkMousePos: %d, %d", p.x(), p.y());
		if (!near_top) {
			emit cursorNearTop(p);
			near_top = true;
		}
	} else {
		if (near_top) {
			emit cursorFarEdges();
			near_top = false;
		}
	}
}

void BaseGui::loadQss(QString filename) {
#if QT_VERSION >= 0x040200
	QFile file( filename );
	file.open(QFile::ReadOnly);
	QString styleSheet = QLatin1String(file.readAll());

	qApp->setStyleSheet(styleSheet);
#endif
}

void BaseGui::changeStyleSheet(QString style) {
#if QT_VERSION >= 0x040200
	if (style.isEmpty())  {
		qApp->setStyleSheet("");
	} 
	else {
		QString qss_file = Helper::themesPath() +"/"+ pref->iconset +"/style.qss";
		if (QFileInfo(qss_file).exists()) {
			loadQss(qss_file);
		} else {
			qApp->setStyleSheet("");
		}
	}
#endif
}


void BaseGui::loadActions() {
	qDebug("BaseGui::loadActions");
	ActionsEditor::loadFromConfig(this, settings);
	ActionsEditor::loadFromConfig(playlist, settings);

	actions_list = ActionsEditor::actionsNames(this);
	actions_list += ActionsEditor::actionsNames(playlist);

	if (csmanager)
		csmanager->setActionsList( actions_list );
}

void BaseGui::saveActions() {
	qDebug("BaseGui::saveActions");

	ActionsEditor::saveToConfig(this, settings);
	ActionsEditor::saveToConfig(playlist, settings);
}
