/*
 *  Part of the shrinkta program, a dvd copy tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <bonobo.h>
#include <gnome.h>

#include <dvd.h>

#include "preferences.h"
#include "app.h"
#include "dialogs.h"
#include "pipe.h"
#include "author.h"

/* message from dvdauthor */
	/* "INFO: dvdauthor creating VTS */
	/* "STAT: Picking VTS 01" */
	/* "\n" */
	/* "STAT: Processing movie.mpg..." */
	/* "STAT: VOBU 2848 at 988MB, 1 PGCS\n" */
	/* "STAT: fixing VOBU at 1124MB (4650/145678, 43%)\n" */
	/* "STAT: fixed 12258 VOBUS\n" */
	/* "INFO: dvdauthor creating table of contents\n" */
	/* "INFO: Scanning dvd/VIDEO_TS/VTS_01_0.IFO\n" */

	
static gchar*	get_chapter_opts	(DvdTitle	*title,
					 guint8		 first_chapter,
					 guint8		 last_chapter);
static gchar*	get_audio_opts		(DvdAudio	*audio1,
					 DvdAudio	*audio2);
static void	child_setup_func	(gpointer	 data);


static void
child_setup_func		(gpointer	 data)
{
	PipeHandle *handle;

	handle = (PipeHandle *) data;
}

static gchar
*get_chapter_opts		(DvdTitle	*title,
				 guint8		 first_chapter,
				 guint8		 last_chapter)
{
	gchar *ch_opts = NULL;
	GString *ch_str = NULL;
	gint chapters, i;
	guint32 start_ts = 0;
	
	chapters = dvd_title_get_chapters (title);
	g_assert (chapters > 0);
	for (i = first_chapter;
	     i <= last_chapter;
	     i++) {
		DvdChapter *chapter;
		const DvdTime *chapter_start_time;
		DvdTime *time;
		
		chapter = dvd_title_get_chapter (title, i);
		if (ch_str == NULL) {
			ch_str = g_string_new ("");
			const DvdTime *first_time;
			
			first_time = dvd_chapter_get_start_time (chapter);
			start_ts = first_time->ts;
		} else {
			ch_str = g_string_append_c (ch_str, ',');
			chapter = dvd_title_get_chapter (title, i);
		}
		chapter_start_time = dvd_chapter_get_start_time (chapter);
		time = dvd_time_new_from_time_stamp (chapter_start_time->ts - start_ts);
		g_string_append_printf (ch_str,"%.1d:%.2d:%.2d.%02d",
					time->bk_hours,
					time->bk_minutes,
					time->bk_seconds,
					time->bk_milliseconds);
		g_object_unref (G_OBJECT (chapter));
		g_free (time);
	}
	ch_opts = g_string_free (ch_str, FALSE);
	return ch_opts;
}

static gchar
*get_audio_opts			(DvdAudio	*audio1,
				 DvdAudio	*audio2)
{
	const gchar *AUDIO_FORMAT[7] = {"ac3+", "", "mp2+", "", "pcm+", "", "dts+"};
	const gchar *AUDIO_QUANT[4] = {"+16bps", "+20bps", "+24bps", "+drc"};
	
	gchar *audio_opts = NULL;
	GString *audio_str = NULL;
	DvdLanguage audio_lang;
	DvdAudioFormat audio_format;
	DvdAudioQuant audio_quant;
	guint8 channels;
	
	
	
	/* [-a] ac3+drc+6ch+en,dts+drc+6ch+en */
	g_assert (audio1 != NULL);
	
	audio_str = g_string_new ("");
	
	/* audio 1 */
	audio_lang = dvd_audio_get_lang (audio1);
	audio_format = dvd_audio_get_format (audio1);
	audio_quant = dvd_audio_get_quant (audio1);
	channels = dvd_audio_get_channels (audio1);
	
	audio_str = g_string_append (audio_str, AUDIO_FORMAT[audio_format]);
	if ((audio_lang > DVD_LANG_UNDEFINED) &&
	    (audio_lang < DVD_LANG_UNKNOWN)) {
		audio_str = g_string_append (audio_str, dvd_language_get_code (audio_lang));
	} else {
		audio_str = g_string_append (audio_str, "nolang");
	}
	audio_str = g_string_append (audio_str, AUDIO_QUANT[audio_quant]);
	g_string_append_printf (audio_str, "+%dch", channels);
	if (audio2 != NULL) {
		/* audio 2 */
		audio_lang = dvd_audio_get_lang (audio2);
		audio_format = dvd_audio_get_format (audio2);
		audio_quant = dvd_audio_get_quant (audio2);
		channels = dvd_audio_get_channels (audio2);
		
		audio_str = g_string_append (audio_str, ",");
		audio_str = g_string_append (audio_str, AUDIO_FORMAT[audio_format]);
		
		if ((audio_lang > DVD_LANG_UNDEFINED) &&
		    (audio_lang < DVD_LANG_UNKNOWN)) {
			audio_str = g_string_append (audio_str, dvd_language_get_code (audio_lang));
		} else {
			audio_str = g_string_append (audio_str, "nolang");
		}
		audio_str = g_string_append (audio_str, AUDIO_QUANT[audio_quant]);
		g_string_append_printf (audio_str, "+%dch", channels);
	}
	audio_opts = g_string_free (audio_str, FALSE);
	return audio_opts;
}

gboolean
author_dvd		(const gchar	*dvd_dir,
			 const gchar	*movie_path,
			 DvdTitle	*title,
			 DvdAudio	*dvd_audio1,
			 DvdAudio	*dvd_audio2,
			 guint8		 first_chapter,
			 guint8		 last_chapter,
			 guint		 size_mb,
			 progress_func	 func)
{
	PipeHandle author_handle;
	PipeHandle file_handle;
	gchar *args[11];
	gchar *tmpargs;
	gboolean ok;
	gchar *tmp_fifo_name;
	gchar *dir;

	func ("Creating DVD titleset", 0.0);
	
	/* open files */
	file_handle.read_fd = open (movie_path, O_RDONLY);
	if (file_handle.read_fd < 1) {
		g_warning ("Unable to open read file %s", movie_path);
		return FALSE;
	}
	
	dir = g_path_get_dirname (movie_path);
	tmp_fifo_name = g_strdup_printf ("%s/dvdauthor.mpg", dir);
	g_free (dir);
	g_message ("making fifo");
	if (mkfifo (tmp_fifo_name, 0664) < 0) {
		g_warning ("Unable to make named pipe %s", tmp_fifo_name);
		g_free (tmp_fifo_name);
		return FALSE;
	}
						
	/* dvdauthor will open the fifo for reading */
	
	/* why isn't the args param of g_spawn_async_with_pipes () a constant? */
	args[0] = "dvdauthor";
	args[1] = "-t";
	args[2] = "-o";
	args[3] = g_strdup (dvd_dir);
	args[4] = "-c";
	args[5] = get_chapter_opts (title, first_chapter, last_chapter);
	args[6] = "-a";
	args[7] = get_audio_opts (dvd_audio1, dvd_audio2);
	args[8] = "-f";
	args[9] = tmp_fifo_name;
	args[10] = NULL;
	
	g_message ("chapter opts \"%s\"", args[4]);
	g_message ("audio opts \"%s\"", args[6]);
	ok = g_spawn_async_with_pipes (NULL,
				       args,
				       NULL,
				       G_SPAWN_SEARCH_PATH | G_SPAWN_DO_NOT_REAP_CHILD,
				       child_setup_func,
				       &author_handle,
				       &author_handle.child,
				       NULL,
				       NULL,
				       NULL,
				       NULL);
	g_free (args[3]);
	g_free (args[5]);
	g_free (args[7]);
	if (ok == FALSE) {
		g_warning ("can't spawn dvdauthor process");
		if (unlink (tmp_fifo_name) == -1) {
			g_warning ("Unable to remove fifo file");
		}
		g_free (tmp_fifo_name);
		return FALSE;
	}
	/* open fifo after dvdauthor is running or open will hang */
	g_message ("opening fifo");
	file_handle.write_fd = open (tmp_fifo_name, O_WRONLY);
	if (file_handle.write_fd < 1) {
		g_warning ("Unable to open named pipe %s", tmp_fifo_name);
		if (unlink (tmp_fifo_name) == -1) {
			g_warning ("Unable to remove fifo file");
		}
		g_free (tmp_fifo_name);
		return FALSE;
	}
	file_handle.read_mbs = size_mb;
	file_handle.func = func;
	
	/* "dvdauthor -o rs -x dvd.xml"  or,*/
	/* "dvdauthor -o dvd -v pal -a ac3+drc+6ch+en,dts+drc+6ch+en movie.mpg" then, */
	/* "dvdauthor -T -o dvd" (create TOC is quick) */
	
	/* pipe file into dvdauthor */
	g_message ("starting pipe thread");
	file_handle.thread = g_thread_create (pipe_thread_func,
						&file_handle,
						TRUE,
						NULL);
	
	g_message ("join file thread");
	g_thread_join (file_handle.thread);
	
	/* file write finished - but dvdauthor can't finish untill we close the fifo */
	g_message ("close file fds");
	close (file_handle.read_fd);
	close (file_handle.write_fd);
	
	func ("Checking dvd structure...will take a few minutes", 99.9);
	/* wait untill dvdauthor has finished up */
	while (1) {
		pid_t retpid;
		gint status = 0;
		
		retpid = waitpid (author_handle.child, &status, WNOHANG);
		
		if (status > 0) {
			if (WIFEXITED (status)) {
				g_message ("dvdauthor normal exit");
				break;
			} else if (WIFSIGNALED (status)) {
				g_warning ("dvdauthor signaled to death");
				break;
			}
		}
		if (retpid >= 0) {
			
		} else if (retpid < 0) {
			if (errno != ECHILD) {
				perror ("dvdauthor_mpg");
			}
			break;
		}
		g_usleep (200);
	}
	g_message ("removing fifo");
	if (unlink (tmp_fifo_name) == -1) {
		g_warning ("Unable to remove fifo file");
	}
	g_free (tmp_fifo_name);
	
	/* make TOC */
	func ("Writing DVD table of contents", 0.0);
	tmpargs = g_strdup_printf ("dvdauthor -T -o \"%s\"", dvd_dir);
	if (g_spawn_command_line_async (tmpargs, NULL) == FALSE) {
		g_free (tmpargs);
		g_warning ("Unable to make TOC");
		return FALSE;
	}
	g_free (tmpargs);
	g_message ("DVD author done");
	return TRUE;
}
