/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvdread/dvd_reader.h>
#include <dvdread/ifo_types.h>
#include <dvdread/ifo_read.h>
#include <dvdread/nav_read.h>


#include "dvd.h"
#include "dvd-time.h"

static DvdClass *dvd_disk_parent_class = NULL;
enum {
	BACKUP_PROGRESS,
	LAST_SIGNAL
};
static guint dvd_disk_signals[LAST_SIGNAL];

static void	dvd_disk_class_init		(DvdDiskClass	*class);
static void	dvd_disk_instance_init		(GTypeInstance	*instance,
						 gpointer	 g_class);
static void	dvd_disk_dispose		(GObject	*object);
/* imported from dvd-title-disk.c */
extern gboolean	dvd_title_disk_read_properties	(DvdTitleDisk	 *title,
						 dvd_reader_t	 *dvd_handle,
						 const gchar	 *device,
						 guint8		  title_id,
						 GError		**error);
static gboolean	dvd_disk_backup_vts 		(DvdDisk         *dvd_disk,
						 dvd_reader_t	 *disk_handle,
						 guint16	  vts_iter,
						 gboolean	  menu,
						 guint32	  blocks,
						 const gchar	 *video_path,
						 GError		**error);
static gboolean	dvd_disk_backup_ifo 		(DvdDisk         *dvd_disk,
						 dvd_reader_t	 *disk_handle,
						 guint16	  vts_iter,
						 guint32	  blocks,
						 const gchar	 *video_path,
						 GError		**error);
static void
dvd_disk_update_backup_progress	(DvdDisk         *dvd);
static void
dvd_disk_class_init		(DvdDiskClass	*class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_disk_signals[BACKUP_PROGRESS] =
		g_signal_new("backup_progress",
			     G_TYPE_FROM_CLASS(class),
			     G_SIGNAL_RUN_LAST,
			     G_STRUCT_OFFSET (DvdDiskClass, backup_progress),
			     NULL, NULL,
			     g_cclosure_marshal_VOID__UINT,
			     G_TYPE_NONE, 1, G_TYPE_UINT);
	dvd_disk_parent_class = g_type_class_ref (DVD_TYPE);
	object_class->dispose = dvd_disk_dispose;
}

static void
dvd_disk_instance_init		(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdDisk *dvd_disk;
	
	dvd_disk = DVD_DISK (instance);

	dvd_disk->device = NULL;
	dvd_disk->backup_mutex = g_mutex_new ();
	dvd_disk->backing_up = FALSE;
	dvd_disk->backup_canned = FALSE;
	DVD (dvd_disk)->source = DVD_SOURCE_DISK;
}

static void
dvd_disk_dispose		(GObject	*object)
{
	DvdDisk *dvd_disk = DVD_DISK (object);

	g_free (dvd_disk->device);
	g_free (dvd_disk->vts);
	g_mutex_free (dvd_disk->backup_mutex);
	G_OBJECT_CLASS (dvd_disk_parent_class)->dispose (G_OBJECT (dvd_disk));
}

/**
 * dvd_disk_get_type:
 *
 * Returns the GType for the DvdDisk class.
 */

GType
dvd_disk_get_type		(void)
{
	static GType dvd_disk_type = 0;

	if (dvd_disk_type == 0) {
		GTypeInfo dvd_disk_info = {
			sizeof (DvdDiskClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_disk_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDisk),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_disk_instance_init
	    	};
		dvd_disk_type = g_type_register_static (DVD_TYPE,
							"DvdDisk",
							&dvd_disk_info, 0);
	}
	return dvd_disk_type;
}

/**
 * Create a new dvd_disk object.
 * You must call g_object_unref() when you are done with the returned object.
 * @returns A #DvdDisk object.
 */
DvdDisk
*dvd_disk_new			(void)
{
	DvdDisk *dvd;
	
	dvd = g_object_new (dvd_disk_get_type (), NULL);
	return dvd;
}

/**
 * Reads the the DVD disk properties, adding the appropriate titles,
 * chapters and cells to the #DvdDisk object.
 * @param dvd A #DvdDisk.
 * @param device The full path to a DVD_DISK rom block device,
 * or path to fully authored dvd file structure.
 * @param error The return location for an allocated GError, or NULL to ignore errors.
 * @returns TRUE if successful, FALSE on fail.
 */
gboolean
dvd_disk_read_properties	(DvdDisk	 *dvd_disk,
				 const gchar	 *device,
				 GError		**error)
{
	dvd_reader_t *dvd_handle;
	ifo_handle_t *ifo_zero;
	vmgi_mat_t *vmgi_mat;
	guint8 title_id;
	guchar volsetid[129];
	gchar volid[33]; /* latin-1 encoded (8bit unicode) */
	guint8 title_count;
	guint16 vts_iter;
	
	g_assert (dvd_disk != NULL);
	g_assert (device != NULL);
	
	dvd_handle = DVDOpen (device);
	if (dvd_handle == NULL) {
		g_warning ("Can't open disc %s!", device);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("There may not be a disk in the drive or the disk is unreadable"));
		return FALSE;
	}
	memset (volsetid, '\0', 129);
	if ((DVDUDFVolumeInfo (dvd_handle,
			       volid,
			       33,
			       volsetid,
			       128) == -1) &&
	    (DVDISOVolumeInfo (dvd_handle,
			       volid,
			       33,
			       volsetid,
			       128) == -1)) {
		g_warning ("Can't read volume info of disc %s!", device);
		
		dvd_set_volume_set (DVD (dvd_disk), "UNKNOWN");
		dvd_set_name (DVD (dvd_disk), "UNKNOWN");
	} else {
		dvd_set_volume_set (DVD (dvd_disk), (gchar *) volsetid);
		dvd_set_name (DVD (dvd_disk), volid);
	}
	ifo_zero = ifoOpen (dvd_handle, 0);
	if (ifo_zero == NULL) {
		g_warning ("Can't open main ifo!");
		DVDClose (dvd_handle);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open the DVD video manager. "
			       "The disk in the drive is not a DVD video or the disk is unreadable"));
		return FALSE;
	}
	
	title_count = ifo_zero->tt_srpt->nr_of_srpts;
	vmgi_mat = ifo_zero->vmgi_mat;
	/* region code is byte 36 of VIDEO_TS.IFO */
	/* a 0 bit means the region is set, tho opposite to the region in dvd-obj.h */
	/* vmgi_mat->vmg_category is 32 bits, we only want the second byte */
	dvd_set_region (DVD (dvd_disk), ~(vmgi_mat->vmg_category >> 16));
	dvd_set_provider (DVD (dvd_disk), vmgi_mat->provider_identifier);
	dvd_disk->vts_count = vmgi_mat->vmg_nr_of_title_sets + 1;
	/*g_message ("No of VTS's %d", dvd_disk->vts_count);
	g_message ("Titles = %d", title_count);*/
	ifoClose (ifo_zero);
	
	/* determine disk size in blocks */
	
	dvd_disk->vts = (g_new0 (DvdDiskVts, dvd_disk->vts_count));
	for (vts_iter = 0;
	     vts_iter < dvd_disk->vts_count;
	     ++vts_iter) {
		DvdDiskVts *vts;
		dvd_file_t *disk_file;
		
		vts = dvd_disk->vts + vts_iter;
		
		/* ifo/bup */
		disk_file = DVDOpenFile (dvd_handle, vts_iter, DVD_READ_INFO_FILE);
		if (disk_file != NULL) {
			vts->ifo_blocks = DVDFileSize (disk_file);
			/* BUP is identical */
			dvd_disk->blocks += (vts->ifo_blocks * 2);
			DVDCloseFile (disk_file);
		}
		/* VOB */
		disk_file = DVDOpenFile (dvd_handle, vts_iter, DVD_READ_MENU_VOBS);
		if (disk_file != NULL) {
			vts->vts_menu_blocks = DVDFileSize (disk_file);
			DVDCloseFile (disk_file);
		}
		if (vts_iter != 0) {
			disk_file = DVDOpenFile (dvd_handle, vts_iter, DVD_READ_TITLE_VOBS);
			if (disk_file != NULL) {
				vts->vts_blocks = DVDFileSize (disk_file);
				DVDCloseFile (disk_file);
			} else {
				DVDClose (dvd_handle);
				g_free (dvd_disk->vts);
				dvd_disk->vts = NULL;
				dvd_disk->blocks = 0;
				dvd_disk->backup_block = 0;
				g_set_error (error,
					     DVD_ERROR,
					     DVD_ERROR_DISK_OPEN,
					     _("Unable to open the DVD vts file. "
					       "The disk in the drive is not a DVD video or the disk is unreadable"));
				return FALSE;
			}
		}
		dvd_disk->blocks += vts->vts_menu_blocks;
		dvd_disk->blocks += vts->vts_blocks;
		
	}
	/*g_message ("Disk size is %d blocks", dvd_disk->blocks);*/
	if (dvd_disk->blocks > DVD_MEDIA_SL_BLOCKS) {
		/*g_message ("Disk is a DL disk (8.4Gb)");*/
		dvd_disk->media = DVD_MEDIA_DL;
	} else {
		/*g_message ("Disk will fit on an SL disk (4.7Gb)");*/
		dvd_disk->media = DVD_MEDIA_SL;
	}
	for (title_id = 0;
	     title_id < title_count;
	     title_id++) {
		DvdTitleDisk *title;
		gboolean read_ok;
		
		/*g_message ("reading title %d", title_id);*/
		title = dvd_title_disk_new ();
		read_ok = dvd_title_disk_read_properties (title,
							  dvd_handle,
							  device,
							  title_id,
							  error);
		if (read_ok == FALSE) {
			g_object_unref (G_OBJECT (title));
			g_warning ("dvd-disk Failed to read title %u - %s", title_id, (*error)->message);
			DVDClose (dvd_handle);
			g_free (dvd_disk->vts);
			dvd_disk->vts = NULL;
			dvd_disk->blocks = 0;
			dvd_disk->backup_block = 0;
			dvd_delete_all (DVD (dvd_disk));
			return FALSE;
		} else {
			dvd_append_title (DVD (dvd_disk), DVD_TITLE (title));
		}
	}
	/*DVD (dvd)->title_count = title_count;*/
	dvd_disk->device = g_strdup (device);
	DVDClose (dvd_handle);
	return TRUE;
}

static void
dvd_disk_update_backup_progress	(DvdDisk         *dvd_disk)
{
	guint8 percent;
	
	if (dvd_disk->backup_block == dvd_disk->blocks) {
		percent = 100;
	} else {
		percent = (guint8) ((gdouble) dvd_disk->backup_block / (gdouble) dvd_disk->blocks * 100.0);
	}
	if (dvd_disk->percent != percent) {
		dvd_disk->percent = percent;
		g_signal_emit (G_OBJECT (dvd_disk),
			       dvd_disk_signals[BACKUP_PROGRESS],
		 	       0, percent);
	}
}

static gboolean
dvd_disk_backup_ifo 		(DvdDisk         *dvd_disk,
				 dvd_reader_t	 *disk_handle,
				 guint16	  vts_iter,
				 guint32	  blocks,
				 const gchar	 *video_path,
				 GError		**error)
{
	gchar *ifo_file_path;
	gchar *bup_file_path;
	gboolean backup_ok = TRUE;
	gboolean check_region = FALSE;
	guint32 block;
	DvdFileIO *ifo_file_io;
	DvdFileIO *bup_file_io;
	dvd_file_t *ifo_disk_file = NULL;
	
	/*g_message ("backing up %d ifo blocks for vts %d", blocks, vts_iter);*/
	/* filenames */
	if (vts_iter == 0) {
		ifo_file_path = g_build_filename (video_path, "VIDEO_TS.IFO", NULL);
		bup_file_path = g_build_filename (video_path, "VIDEO_TS.BUP", NULL);
		check_region = TRUE;
	} else {
		gchar *ifo_file;
		gchar *bup_file;
		
		ifo_file = g_strdup_printf ("VTS_%.2d_0.IFO", vts_iter);
		bup_file = g_strdup_printf ("VTS_%.2d_0.BUP", vts_iter);
		ifo_file_path = g_build_filename (video_path, ifo_file, NULL);
		bup_file_path = g_build_filename (video_path, bup_file, NULL);
		g_free (ifo_file);
		g_free (bup_file);
	}
	
	/* open files */
	ifo_file_io = dvd_file_io_new ();
	bup_file_io = dvd_file_io_new ();
	if ((dvd_file_io_open (ifo_file_io, ifo_file_path, DVD_FILE_IO_WRITE, FALSE, error) == FALSE) ||
	    (dvd_file_io_open (bup_file_io, bup_file_path, DVD_FILE_IO_WRITE, FALSE, error) == FALSE)) {
		backup_ok = FALSE;
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("Unable to open backup movie information file"));
	} else {
		ifo_disk_file = DVDOpenFile (disk_handle, vts_iter, DVD_READ_INFO_FILE);
		if (ifo_disk_file == NULL) {
			backup_ok = FALSE;
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_OPEN,
				     _("Unable to open DVD disk file. Try cleaning the disk."));
		}
	}
	/* copy disk ifo to backup ifo and bup */
	block = 0;
	while ((block < blocks) &&
	       (backup_ok == TRUE)) {
		gssize bytes;
		guint8 data[2048];
		
		/* check for dvd_disk_stop_backup called in different thread! */
		g_mutex_lock (dvd_disk->backup_mutex);
		if (dvd_disk->backup_canned == TRUE) {
			dvd_disk->backing_up = FALSE;
			backup_ok = FALSE;
			g_mutex_unlock (dvd_disk->backup_mutex);
			g_warning ("Backup Cancelled");
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_CANCELLED,
				     _("The operation has been cancelled"));
			break;
		}
		g_mutex_unlock (dvd_disk->backup_mutex);
		
		/* do I need a loop here like read () ? */
		bytes = DVDReadBytes (ifo_disk_file, data, 2048);
		if (bytes != 2048) {
			g_warning ("DVDReadBytes failed");
			backup_ok = FALSE;
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_READ,
				     _("Unable to read DVD disk. Try cleaning the disk."));
			break;
		}
		if ((block == 0) &&
		    (vts_iter == 0) &&
		    (check_region == TRUE)) {
			DvdRegion region;
			
			region = dvd_get_region	(DVD (dvd_disk));
			
			if (data[35] != ~region) {							
				g_message ("DVD VIDEO_TS.IFO region has been changed from 0x%x to 0x%x", (data[35]),(guint8) ~region);
				data[35] = ~region;
			}
		}
		backup_ok = dvd_file_io_write (ifo_file_io, data, bytes, error);
		if (backup_ok == TRUE) {
			backup_ok = dvd_file_io_write (bup_file_io, data, bytes, error);
		}
		++block;
		++dvd_disk->backup_block;
	}
	dvd_file_io_close (ifo_file_io, NULL);
	dvd_file_io_close (bup_file_io, NULL);
	g_object_unref (ifo_file_io);
	g_object_unref (bup_file_io);
	g_free (ifo_file_path);
	g_free (bup_file_path);
	if (ifo_disk_file != NULL) {
		DVDCloseFile (ifo_disk_file);
	}
	dvd_disk_update_backup_progress (dvd_disk);
	return backup_ok;
}

static gboolean
dvd_disk_backup_vts 		(DvdDisk         *dvd_disk,
				 dvd_reader_t	 *disk_handle,
				 guint16	  vts_iter,
				 gboolean	  menu,
				 guint32	  blocks,
				 const gchar	 *video_path,
				 GError		**error)
{
	gchar *vts_file_path;
	gboolean backup_ok = TRUE;
	guint32 block, this_vob_block;
	DvdFileIO *vts_file_io;
	dvd_file_t *vts_disk_file = NULL;
	gint vob_type;
	guint8 this_vob = 0;
	
	if (blocks == 0) {
		return TRUE;
	}
	/* filename */
	if (vts_iter == 0) {
		if (menu == FALSE) {
			/* quietly return */
			return TRUE;
		}
		vts_file_path = g_build_filename (video_path, "VIDEO_TS.VOB", NULL);
		vob_type = DVD_READ_MENU_VOBS;
	} else {
		gchar *vts_file;
		
		if (menu == TRUE) {
			this_vob = 0;
			vob_type = DVD_READ_MENU_VOBS;
		} else {
			this_vob = 1;
			vob_type = DVD_READ_TITLE_VOBS;
		}
		vts_file = g_strdup_printf ("VTS_%.2d_%d.VOB", vts_iter, this_vob);
		vts_file_path = g_build_filename (video_path, vts_file, NULL);
		g_free (vts_file);
	}
	
	/* open file */
	vts_disk_file = DVDOpenFile (disk_handle, vts_iter, vob_type);
	if (vts_disk_file == NULL) {
		if (menu == TRUE) {
			/* probably no menu! */
			g_warning ("vts %d has no menu", vts_iter);
		} else {
			backup_ok = FALSE;
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_OPEN,
				     _("Unable to open DVD disk. Try cleaning the disk."));
		}
		g_free (vts_file_path);
		return backup_ok;
	} else {
		vts_file_io = dvd_file_io_new ();

		if (dvd_file_io_open (vts_file_io, vts_file_path, DVD_FILE_IO_WRITE, FALSE, error) == FALSE) {
			g_object_unref (vts_file_io);
			g_free (vts_file_path);
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_OPEN,
				     _("Unable to open backup movie file"));
			return FALSE;
		}
	}
	/* copy disk ifo to backup ifo and bup */
	block = 0;
	this_vob_block = 0;
	while ((block < blocks) &&
	       (backup_ok == TRUE)) {
		gssize blocks_read;
		guint8 data[2048];
		
		/* check for dvd_disk_stop_backup called in different thread! */
		g_mutex_lock (dvd_disk->backup_mutex);
		if (dvd_disk->backup_canned == TRUE) {
			dvd_disk->backing_up = FALSE;
			backup_ok = FALSE;
			g_mutex_unlock (dvd_disk->backup_mutex);
			g_warning ("Backup Cancelled");
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_CANCELLED,
				     _("The operation has been cancelled"));
			break;
		}
		g_mutex_unlock (dvd_disk->backup_mutex);
		
		/* do I need a loop here like read () ? */
		blocks_read = DVDReadBlocks (vts_disk_file, block, 1, data);
		if (blocks_read != 1) {
			backup_ok = FALSE;
			g_warning ("Couldn't read disk block");
			g_set_error (error,
				     DVD_ERROR,
				     DVD_ERROR_DISK_READ,
				     _("Unable to read DVD disk. Try cleaning the disk."));
			break;
		}
		backup_ok = dvd_file_io_write (vts_file_io, data, blocks_read * 2048, error);
		block += blocks_read;
		this_vob_block += blocks_read;
		dvd_disk->backup_block += blocks_read;
		if (this_vob_block == 524272) {
			gchar *vts_file;
			
			dvd_file_io_close (vts_file_io, NULL);
			this_vob += 1;
			vts_file = g_strdup_printf ("VTS_%.2d_%d.VOB", vts_iter, this_vob);
			g_free (vts_file_path);
			vts_file_path = g_build_filename (video_path, vts_file, NULL);
			g_free (vts_file);
			if (dvd_file_io_open (vts_file_io, vts_file_path, DVD_FILE_IO_WRITE, FALSE, error) == FALSE) {
				g_set_error (error,
					     DVD_ERROR,
					     DVD_ERROR_DISK_OPEN,
					     _("Unable to open backup movie file"));
				backup_ok = FALSE;
			}
			this_vob_block = 0;
		}
		dvd_disk_update_backup_progress (dvd_disk);
	}
	dvd_file_io_close (vts_file_io, NULL);
	g_object_unref (vts_file_io);
	g_free (vts_file_path);
	if (vts_disk_file != NULL) {
		DVDCloseFile (vts_disk_file);
	}
	return backup_ok;
}

/* returns the path to the folder where the dvd is written */
/* you must check error and cleanup the returned path if it is not null */
gchar
*dvd_disk_backup		 (DvdDisk	 *dvd_disk,
				  const gchar	 *path,
				  GError	**error)
{
	dvd_reader_t *disk_handle;
	guint16 vts_iter;
	gchar *disk_path = NULL;
	gchar *video_path;
	gchar *audio_path;
	gboolean backup_ok = TRUE;
	
	g_assert (dvd_disk != NULL);
	g_assert (dvd_disk->device != NULL);
	g_assert (path != NULL);
	
	g_mutex_lock (dvd_disk->backup_mutex);
	dvd_disk->backing_up = TRUE;
	g_mutex_unlock (dvd_disk->backup_mutex);
	
	dvd_disk->backup_block = 0;
	dvd_disk->percent = 100;
	if (g_file_test (path, G_FILE_TEST_IS_DIR) == FALSE) {
		g_warning ("Invalid directory %s!", path);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_INVALID_DIR,
			     _("The folder does not exist or is a regular file"));
		g_mutex_lock (dvd_disk->backup_mutex);
		dvd_disk->backing_up = FALSE;
		g_mutex_unlock (dvd_disk->backup_mutex);
		return NULL;
	}
	
	/* open disk */
	disk_handle = DVDOpen (dvd_disk->device);
	if (disk_handle == NULL) {
		g_warning ("Can't open disc %s!", dvd_disk->device);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_DISK_OPEN,
			     _("There may not be a disk in the drive or the disk is unreadable"));
		g_mutex_lock (dvd_disk->backup_mutex);
		dvd_disk->backing_up = FALSE;
		g_mutex_unlock (dvd_disk->backup_mutex);
		return NULL;
	}
	
	/* build paths */
	disk_path  = g_build_filename (path, dvd_get_name (DVD (dvd_disk)), NULL);
	video_path = g_build_filename (disk_path, "VIDEO_TS", NULL);
	audio_path = g_build_filename (disk_path, "AUDIO_TS", NULL);
	/*g_message ("%s", disk_path);
	g_message ("%s", video_path);
	g_message ("%s", audio_path);*/
	
	/* make directories */
	if ((mkdir (disk_path, 0774) == -1) ||
	    (mkdir (video_path, 0774) == -1) ||
	    (mkdir (audio_path, 0774) == -1)) {
		/* ToDo */
		/*GFileError file_error;*/
			
		/*file_error = g_file_error_from_errno (errno);*/
		/*g_set_error (error,
			     G_FILE_ERROR,
			     file_error,
			     FILE_IO_ERRORS[file_error]);*/

		g_warning ("Invalid directory %s!", path);
		g_set_error (error,
			     DVD_ERROR,
			     DVD_ERROR_INVALID_DIR,
			     _("Unable to create files in folder. You may not have permission"));
		DVDClose (disk_handle);
		g_free (disk_path);
		g_free (video_path);
		g_free (audio_path);
		g_mutex_lock (dvd_disk->backup_mutex);
		dvd_disk->backing_up = FALSE;
		g_mutex_unlock (dvd_disk->backup_mutex);
		return NULL;
	}
	
	for (vts_iter = 0;
	     vts_iter < dvd_disk->vts_count;
	     ++vts_iter) {
		DvdDiskVts *vts;
		
		vts = dvd_disk->vts + vts_iter;
		if ((dvd_disk_backup_vts (dvd_disk, disk_handle, vts_iter, TRUE, vts->vts_menu_blocks, video_path, error) == FALSE) ||
		    (dvd_disk_backup_vts (dvd_disk, disk_handle, vts_iter, FALSE, vts->vts_blocks, video_path, error) == FALSE) ||
		    (dvd_disk_backup_ifo (dvd_disk, disk_handle, vts_iter, vts->ifo_blocks, video_path, error) == FALSE)) {
			backup_ok = FALSE;
			break;
		}
	}
	DVDClose (disk_handle);
	g_free (video_path);
	g_free (audio_path);
	g_mutex_lock (dvd_disk->backup_mutex);
	dvd_disk->backing_up = FALSE;
	g_mutex_unlock (dvd_disk->backup_mutex);
	return disk_path;
}


void
dvd_disk_stop_backup	 	(DvdDisk       *dvd_disk)
{
	g_mutex_lock (dvd_disk->backup_mutex);
	if (dvd_disk->backing_up == FALSE) {
		g_warning ("dvd_disk_stop_backup called and no backup is in progress");
		g_mutex_unlock (dvd_disk->backup_mutex);
		return;
	} else if (dvd_disk->backup_canned == TRUE) {
		g_warning ("dvd_disk_stop_backup called more than once");
		g_mutex_unlock (dvd_disk->backup_mutex);
		return;
	} else {
		dvd_disk->backup_canned = TRUE;
		g_mutex_unlock (dvd_disk->backup_mutex);
	}
	while (1) {
		g_usleep (500);
		g_mutex_lock (dvd_disk->backup_mutex);
		if (dvd_disk->backing_up == FALSE) {
			dvd_disk->backup_canned = FALSE;
			g_mutex_unlock (dvd_disk->backup_mutex);
			break;
		}
		g_mutex_unlock (dvd_disk->backup_mutex);
	}
}
