/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>
#include "pes.h" /* only for getbits () */

#define DTS_SYNCFRAME_ID 0x7ffe8001

static DvdDecoderClass *dvd_decoder_dts_parent_class = NULL;

static void     dvd_decoder_dts_class_init	(DvdDecoderDtsClass *class);
static void     dvd_decoder_dts_instance_init	(GTypeInstance	 *instance,
						 gpointer	  g_class);
static void     dvd_decoder_dts_dispose		(GObject	 *object);
static void	dvd_decoder_dts_read_startcode	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_dts_read_header	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_dts_read_frame	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	read_dts_bsi_hdr		(guint8		 *stream,
						 DvdDtsBSHeader	 *hdr);

static void
dvd_decoder_dts_class_init	(DvdDecoderDtsClass *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_decoder_dts_parent_class = g_type_class_ref (DVD_DECODER_TYPE);
	
	object_class->dispose = dvd_decoder_dts_dispose;
}

static void
dvd_decoder_dts_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdDecoder *decoder;
	DvdDecoderDts *dts_decoder;
	
	decoder = DVD_DECODER (instance);
	dts_decoder = DVD_DECODER_DTS (instance);
	decoder->stream_type = DVD_STREAM_DTS_AUDIO;
	
	/* these functions called by parent DvdDecoder */
	decoder->read_startcode = dvd_decoder_dts_read_startcode;
	decoder->read_header    = dvd_decoder_dts_read_header;
	decoder->read_frame     = dvd_decoder_dts_read_frame;
	dts_decoder->sync_hdr.sync_word = 0xffffffff;
}

static void
dvd_decoder_dts_dispose		(GObject	*object)
{
	DvdDecoderDts *decoder;
	
	decoder = DVD_DECODER_DTS (object);
	
	G_OBJECT_CLASS (dvd_decoder_dts_parent_class)->dispose (object);
}

/**
 * dvd_decoder_dts_get_type
 * @return The GType for the DvdDecoderDts class.
 */
GType
dvd_decoder_dts_get_type	(void)
{
	static GType decoder_dts_type = 0;

	if (decoder_dts_type == 0) {
		GTypeInfo decoder_dts_info = {
			sizeof (DvdDecoderDtsClass),
			NULL,
			NULL,
			(GClassInitFunc) dvd_decoder_dts_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDecoderDts),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_decoder_dts_instance_init
	    	};
		decoder_dts_type = g_type_register_static (DVD_DECODER_TYPE,
							   "DvdDecoderDts",
							   &decoder_dts_info, 0);
	}
	return decoder_dts_type;
}


static void
dvd_decoder_dts_read_startcode	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	guint8 *iter;
	DvdDecoderDts *dts_decoder;
	
	dts_decoder = DVD_DECODER_DTS (decoder);
	
	for (iter = *start;
	     iter != end;
	     iter++) {
		if (dts_decoder->sync_hdr.sync_word == DTS_SYNCFRAME_ID) {
			dts_decoder->sync_hdr.sync_word = 0xffffffff;
			dts_decoder->bytes_skipped -= 4;
			if (dts_decoder->bytes_skipped > 0) {
				g_message ("dts bytes skiped = %d", dts_decoder->bytes_skipped);
			}
			dvd_decoder_ensure_buffer_size (decoder, 18);
			dvd_decoder_write_frame_byte (decoder, 0x7f);
			dvd_decoder_write_frame_byte (decoder, 0xfe);
			dvd_decoder_write_frame_byte (decoder, 0x80);
			dvd_decoder_write_frame_byte (decoder, 0x01);
			dvd_decoder_set_state (decoder, DVD_DECODER_STATE_HEADER);
			if (dts_decoder->clocks_per_frame == 0) {
				dvd_decoder_set_start_pts (decoder, pts);
				dvd_decoder_set_current_pts (decoder, pts);
			} else {
				dvd_decoder_add_to_current_pts (decoder, dts_decoder->clocks_per_frame);
			}
			break;
		} else {
			dts_decoder->sync_hdr.sync_word = (dts_decoder->sync_hdr.sync_word << 8) + *iter;
			dts_decoder->bytes_skipped++;
		}
	}
	*start = iter;
}

static void
read_dts_bsi_hdr		(guint8		 *stream,
				 DvdDtsBSHeader	 *hdr)
{
	guint8 bit;
	gboolean print;
	
	if (hdr->frame_type == 0) {
		print = TRUE;
	} else {
		print = FALSE;
	}
	memset (hdr, '0', sizeof (DvdDtsBSHeader));
	bit = 0;
	hdr->frame_type			= getbits (&stream, 1, &bit);
	hdr->deficit_sample_count	= getbits (&stream, 5, &bit);
	hdr->crc_present		= getbits (&stream, 1, &bit);
	hdr->pcm_sample_blocks		= getbits (&stream, 7, &bit);
	hdr->frame_size			= getbits (&stream, 14, &bit);
	hdr->audio_mode			= getbits (&stream, 6, &bit);
	hdr->sample_fequency		= getbits (&stream, 4, &bit);
	hdr->bit_rate			= getbits (&stream, 5, &bit);
	hdr->down_mix_enabled		= getbits (&stream, 1, &bit);
	hdr->dynamic_range_flag		= getbits (&stream, 1, &bit);
	hdr->time_stamp_flag		= getbits (&stream, 1, &bit);
	hdr->aux_data_flag		= getbits (&stream, 1, &bit);
	hdr->hdcd			= getbits (&stream, 1, &bit);
	hdr->ext_audio_desc_flag	= getbits (&stream, 3, &bit);
	hdr->ext_coding_flag		= getbits (&stream, 1, &bit);
	hdr->syncword_insertion_flag	= getbits (&stream, 1, &bit);
	hdr->lfe_flag			= getbits (&stream, 2, &bit);
	hdr->hflag			= getbits (&stream, 1, &bit);
	if (hdr->crc_present == 1) {
		stream += 2;
	}
	hdr->multirate_interpolator	= getbits (&stream, 1, &bit);
	hdr->software_revision		= getbits (&stream, 4, &bit);
	hdr->copy_history		= getbits (&stream, 2, &bit);
	hdr->source_pcm_resolution	= getbits (&stream, 3, &bit);
	hdr->front_sum_difference	= getbits (&stream, 1, &bit);
	hdr->surr_sum_difference	= getbits (&stream, 1, &bit);
	hdr->dialog_normalisation	= getbits (&stream, 4, &bit);
	/*
	if (print == TRUE) {
		g_print ("\nAudio track %d DTS bit stream properties\n", track);
		g_print ("^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^\n");
		g_print ("Audio Start Time Stamp 90kHz  = %d\n", decoder->audio_its);
		dts_print_bsi (&decoder->bs_info);
		g_print ("________________________________________\n\n");
	}*/
}

static void
dvd_decoder_dts_read_header	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	guint8 *iter;
	DvdDecoderDts *dts_decoder;
	
	dts_decoder = DVD_DECODER_DTS (decoder);
	
	/* read 16 bytes */
	iter = *start;
	
	while ((decoder->buffer_pos < 16) &&
	       (iter != end)) {
		dvd_decoder_write_frame_byte (decoder, *iter);
		iter++;
	}
	*start = iter;
	if (decoder->buffer_pos == 16) {
		read_dts_bsi_hdr (&decoder->buffer[4], &dts_decoder->bs_hdr);
		dts_decoder->bytes_per_frame = dts_decoder->bs_hdr.frame_size + 1;
		dvd_decoder_ensure_buffer_size (decoder, dts_decoder->bytes_per_frame);
		dvd_decoder_set_state (decoder, DVD_DECODER_STATE_FRAME);
		dts_decoder->bytes_skipped = 0;
		/* set clocks per frame */
		if (dts_decoder->clocks_per_frame == 0) {
			gdouble samples_per_frame;
			gdouble frames_per_second;
			
			if (dts_decoder->bs_hdr.sample_fequency != 13) { /* 48000 samples per sec */
				g_warning ("unsupported DTS sample freq");
			}
			/* ***WARNING*** 96000 (DTS ES [extended]) audio pts will fail here */
			/* ToDo - check if dts extended! - calculate clocks per frame for es */
			samples_per_frame = (dts_decoder->bs_hdr.pcm_sample_blocks + 1) * 32;
			frames_per_second = 48000.0 / samples_per_frame;
			dts_decoder->clocks_per_frame = 90000 / frames_per_second;
			dvd_decoder_set_frame_clocks (decoder, dts_decoder->clocks_per_frame * 300);
			g_message ("dts - samples per frame = %d", (gint) samples_per_frame);
			g_message ("dts - mpeg (90kHz) clocks per frame = %d", dts_decoder->clocks_per_frame);
		}
	}
}

static void
dvd_decoder_dts_read_frame	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderDts *dts_decoder;
	guint16 length;
	guint16 data_length;
	
	dts_decoder = DVD_DECODER_DTS (decoder);

	
	data_length = end - *start;
	length = dts_decoder->bytes_per_frame - decoder->buffer_pos;
	if (length > data_length) {
		dvd_decoder_write_frame_bytes (decoder, *start, data_length);
		*start = end;
	} else {
		dvd_decoder_write_frame_bytes (decoder, *start, length);
		dvd_decoder_output_frame (decoder);
		*start = *start + length;
		dvd_decoder_set_state (decoder, DVD_DECODER_STATE_STARTCODE);
	}
}
