//==============================================================================
// default_skin.c - implementation of the default skin for the seti_applet.
//
// (C)2002 Richard Kinder <r_kinder@yahoo.com>
//
// Changelog:
// ==========
// $Log: default-skin.c,v $
// Revision 1.1.2.10  2003/03/03 22:45:11  r_kinder
// Namespace clash fixups.
//
// Revision 1.1.2.9  2003/03/03 10:45:43  r_kinder
// Add a new skin (nestor) and update default skin for new pixmap dir structure.
//
// Revision 1.1.2.8  2003/01/30 22:32:13  r_kinder
// Fixes for vertically aligned panels - shrink it to be only a radar.
//
// Revision 1.1.2.7  2003/01/28 08:11:21  r_kinder
// Minor bug fixes, change the default skin look so it looks good on small
// panels (when fixed size with only the radar is selected).
//
// Revision 1.1.2.6  2003/01/23 23:37:27  r_kinder
// Do a 'make indent' on the files. Remove legacy code that is no longer
// pertinant.
//
// Revision 1.1.2.5  2003/01/14 00:30:45  r_kinder
// Various fixes.
//
// Revision 1.1.2.4  2002/11/29 07:08:32  r_kinder
// Port to GNOME 2.x intermediate checkin.
//
// Revision 1.1.2.3  2002/11/05 21:31:30  jmardantz
// Gnome 2 port
//
// Revision 1.1.2.2  2002/04/21 23:17:20  r_kinder
// Fix up the spike item text.
//
// Revision 1.1.2.1  2002/04/09 11:28:54  r_kinder
// Initial commit.
//
//==============================================================================
#include "config.h"
#include "seti_applet.h"

#include <panel-applet.h>

#include "pixmaps/skins/default/logo-red.xpm"
#include "pixmaps/skins/default/logo-green.xpm"
#include "pixmaps/skins/default/logo-grey.xpm"

typedef struct _skin_data_t
{
    GdkPixbuf  *the_red_radar;
    GdkPixbuf  *the_green_radar;
    GdkPixbuf  *the_grey_radar;
} i_skin_data_t;

#define TITLE_OFFSET (0.0)
#define LINE_1_OFFSET (36.0)
#define LINE_2_OFFSET (72.0)
#define LINE_3_OFFSET (108.0)
#define LINE_2_G_OFFSET (80.0)
#define LINE_3_G_OFFSET (120.0)

#define SIZE_SCALE_FACTOR (9)
#define FONT_POINT_SIZE (((applet_size->panel_size)/48.0)*SIZE_SCALE_FACTOR)

#define ALIGN_ITEM_1_SIDE_Y        14.0
#define ALIGN_ITEM_2_SIDE_X        0.0

static gboolean
default_setup(void)
{
    return TRUE;
}

static skin_data *
internal_default_setup(void)
{
    i_skin_data_t *the_images = g_new0(i_skin_data_t,1);
    the_images->the_green_radar =
        gdk_pixbuf_new_from_xpm_data((const char **)
                                     new_seti_applet_logo_green_xpm);
    the_images->the_red_radar =
        gdk_pixbuf_new_from_xpm_data((const char **)
                                     new_seti_applet_logo_red_xpm);
    the_images->the_grey_radar =
        gdk_pixbuf_new_from_xpm_data((const char **)
                                     new_seti_applet_logo_grey_xpm);
    return (skin_data *)the_images;
}

gboolean
default_cleanup(skin_data *the_images)
{
    return TRUE;
}

static gchar *
format_cpu_string(gdouble cum_time,
                  PanelAppletOrient orient)
{
    gchar      *cpuinfo;
    if(cum_time > 315440000000)
    {
        if(orient == PANEL_APPLET_ORIENT_LEFT
           || orient == PANEL_APPLET_ORIENT_RIGHT)
        {
            cpuinfo = g_strdup_printf("%s",
                                      /* ky indicates 10000 years, but must be a
                                         small string to use in a vertical panel */
                                      _(">10ky"));
        }
        else
        {
            cpuinfo = g_strdup_printf("%s", _(">10000yrs"));
        }
    }
    else if(cum_time > 31557600)
    {
        cpuinfo =
            g_strdup_printf("%.2fyrs", cum_time / (60 * 60 * 24 * 365.25));
    }
    else
    {
        if(orient == PANEL_APPLET_ORIENT_LEFT
           || orient == PANEL_APPLET_ORIENT_RIGHT)
        {
            cpuinfo = g_strdup_printf("%3.0f%s", cum_time / (60 * 60 * 24),
                                      /* d indicates days, but must be a small
                                         string to use in a vertical panel */
                                      _("d"));
        }
        else
        {
            cpuinfo = g_strdup_printf("%.0f %s",
                                      cum_time / (60 * 60 * 24), _("days"));
        }
    }
    return cpuinfo;
}


GnomeCanvasItem *
default_cputime_overall(skin_data *data,
                        GnomeCanvasGroup * group,
                        gdouble cputimeoverall,
                        seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *cpu_time_item = GNOME_CANVAS_ITEM(GNOME_CANVAS_GROUP
                                                    (gnome_canvas_item_new
                                                    (group,
                                                    gnome_canvas_group_get_type
                                                    (), "x", 0.0, "y",
                                                    0.0, NULL)));
    gchar      *cpu_label;
    gchar      *cpu_value;
    if(applet_size->only_radar != 0)
    {
        //We are leaking things here - FIXME
        return NULL;
    }

    if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        cpu_label = _("CPU:");
    }
    else
    {
        cpu_label = _("CPU Time:");
    }
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(cpu_time_item),
                          gnome_canvas_text_get_type(),
                          "text", cpu_label,
                          "x", 0.0,
                          "y", LINE_1_OFFSET,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NW,
                          "fill_color_rgba", 0x000000FF, NULL);
    cpu_value = format_cpu_string(cputimeoverall, applet_size->orientation);
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(cpu_time_item),
                          gnome_canvas_text_get_type(),
                          "text", cpu_value,
                          "x", 0.0,
                          "y", LINE_2_OFFSET,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NW,
                          "fill_color_rgba", 0x003300FF, NULL);
    g_free(cpu_value);
    return cpu_time_item;
}

GnomeCanvasItem *
default_cputime_current(skin_data *data,
                        GnomeCanvasGroup * group,
                        gdouble cputimecurrent,
                        seti_applet_size_t * applet_size)
{
    if(applet_size->only_radar != 0)
    {
        //We are leaking things here - FIXME
        return NULL;
    }
    return NULL;
}

GnomeCanvasItem *
default_gaussian(skin_data *data,
                 GnomeCanvasGroup * group,
                 gdouble * gaus_values,
                 gdouble best_gaussian_power,
                 gdouble best_gaussian_score,
                 gdouble max_gaus,
                 guint num_values,
                 seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *gaus_graph_item = GNOME_CANVAS_ITEM(GNOME_CANVAS_GROUP
                                                         (gnome_canvas_item_new
                                                          (group,
                                                           gnome_canvas_group_get_type
                                                           (), "x", 0.0, "y",
                                                           0.0, NULL)));
    gint        i;
    gint        scale_factor = 0;
    GnomeCanvasPoints *points;
    gdouble     theAffine[6];
    gdouble     y1 = 0.0, y2t = 0.0;
    GnomeCanvasItem *tempItem;

    if(applet_size->only_radar != 0)
    {
        //We are leaking things here - FIXME
        return NULL;
    }
    if(applet_size->panel_size >= 48.0)
    {
        scale_factor = 20;
        y1 = 35.0;
        y2t = 15.0;
    }
    else
    {
        scale_factor = 10;
        y1 = 18.0;
        y2t = 12.0;
    }

    if(!
       (applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
        || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT))
    {
        points = gnome_canvas_points_new(256);
        for(i = 0; i < 512; i += 8)
        {
            points->coords[i] = (i / 8) * 1.25;
            points->coords[i + 1] = 0;
            points->coords[i + 2] = (i / 8) * 1.25;
            points->coords[i + 3] =
                (gaus_values[i / 8] * scale_factor) / max_gaus;
            points->coords[i + 4] = ((i / 8) + 1) * 1.25;
            points->coords[i + 5] =
                (gaus_values[i / 8] * scale_factor) / max_gaus;
            points->coords[i + 6] = ((i / 8) + 1) * 1.25;
            points->coords[i + 7] = 0;
        }
        tempItem =
            gnome_canvas_item_new(GNOME_CANVAS_GROUP(gaus_graph_item),
                                  gnome_canvas_polygon_get_type(), "points",
                                  points, "width_pixels", 1, "fill_color_rgba",
                                  0x999999FF, "outline_color_rgba", 0x999999FF,
                                  NULL);
        art_affine_identity(theAffine);
        art_affine_rotate(theAffine, 180.0);
        gnome_canvas_item_affine_relative(tempItem, theAffine);
        art_affine_identity(theAffine);
        art_affine_translate(theAffine, 81.0, y1);
        gnome_canvas_item_affine_relative(tempItem, theAffine);
        tempItem =
            gnome_canvas_item_new(GNOME_CANVAS_GROUP(gaus_graph_item),
                                  gnome_canvas_polygon_get_type(), "points",
                                  points, "width_pixels", 1, "fill_color",
                                  "red", "outline_color", "red", NULL);
        art_affine_identity(theAffine);
        art_affine_rotate(theAffine, 180.0);
        gnome_canvas_item_affine_relative(tempItem, theAffine);
        art_affine_identity(theAffine);
        art_affine_translate(theAffine, 83.0, y1 - 2.0);
        gnome_canvas_item_affine_relative(tempItem, theAffine);
        gnome_canvas_points_unref(points);
        gnome_canvas_item_new(GNOME_CANVAS_GROUP(gaus_graph_item),
                              gnome_canvas_text_get_type(),
                              "text", _("Gaussian:"),
                              "x", 0.0,
                              "y", LINE_1_OFFSET,
                              "anchor", GTK_ANCHOR_NW,
                              "size_points", FONT_POINT_SIZE,
                              "fill_color_rgba", 0x000000FF, NULL);
    }
    else
    {
        gnome_canvas_item_new(GNOME_CANVAS_GROUP(gaus_graph_item),
                              gnome_canvas_text_get_type(), "text",
                              _("Gaus:"), "x", 0.0, "y", ALIGN_ITEM_1_SIDE_Y,
                              "anchor", GTK_ANCHOR_NW,
                              "size_points", FONT_POINT_SIZE,
                              "fill_color_rgba", 0x000000FF, NULL);

    }
    return gaus_graph_item;
}

static gchar *
format_spike_string(gdouble best_spike,
                    PanelAppletOrient orientation)
{
    gchar      *spike;

    if(orientation == PANEL_APPLET_ORIENT_LEFT
       || orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        spike = g_strdup_printf("%3.0f", best_spike);
    }
    else
    {
        spike = g_strdup_printf("%.2f", best_spike);
    }
    return spike;
}

GnomeCanvasItem *
default_spike(skin_data *data,
              GnomeCanvasGroup * group,
              gdouble best_spike,
              seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *spike_item = GNOME_CANVAS_ITEM(GNOME_CANVAS_GROUP
                                                    (gnome_canvas_item_new
                                                     (group,
                                                      gnome_canvas_group_get_type
                                                      (), "x", 0.0, "y",
                                                      0.0, NULL)));

    static gint spikeThresholds[12] =
        { 0, 25, 50, 75, 100, 125, 150, 175, 200, 225, 250, 275 };
    static gint spikeColoursA[12] =
        { 0x55CC00FF, 0x55CC00FF, 0x55CC00FF, 0x55CC00FF, 0xFFEE00FF,
0xFFEE00FF,
        0xFFEE00FF, 0xFFEE00FF, 0xFF0000FF, 0xFF0000FF, 0xFF0000FF, 0xFF0000FF
    };
    static gint spikeColoursU[12] =
        { 0x287A00FF, 0x287A00FF, 0x287A00FF, 0x287A00FF, 0x7A7200FF,
0x7A7200FF,
        0x7A7200FF, 0x7A7200FF, 0x7C0000FF, 0x7C0000FF, 0x7C0000FF, 0x7C0000FF
    };
    double      xpos = 0.0;
    double      xpos2 = 0.0;
    int         count = 0;
    gint        current_colour;
    gchar      *spike_value;
    gdouble     spike_width = LINE_3_OFFSET - LINE_2_OFFSET;
    gdouble     full_spike_width;
    gdouble     spike_elt_width;
    gdouble     spike_gap_width;
    gdouble     spike_text_y_offset = (LINE_3_G_OFFSET - LINE_2_G_OFFSET) / 2.0;

    if(applet_size->only_radar != 0)
    {
        //We are leaking things here - FIXME
        return NULL;
    }

    if(applet_size->auto_size == 1)
    {
        //The width of the canvas we can draw on is the width of the applet
        //less the height (the radar is as wide as high)
        full_spike_width = 128 * 2;
        //We have 12 items in the spike display, each of which has equal
        //width with equal gap between it, so we have 12 large items plus 12
        //small gaps
        //Divide by 14, each spike elt is 1/16th of the width of the display,
        //with the remaining 2 widths used for gaps
        spike_elt_width = full_spike_width / 16.0;
        //Divide up the 4 remaining spike widths between the 12 gaps
        spike_gap_width = (4.0 * (full_spike_width / 16.0)) / 12.0;
    }

    if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        xpos2 = ALIGN_ITEM_2_SIDE_X;
    }
    else
    {
        xpos2 = (128.0 * 2) / 2.0;
    }
    spike_value = format_spike_string(best_spike, applet_size->orientation);
    if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        gnome_canvas_item_new(GNOME_CANVAS_GROUP(spike_item),
                              gnome_canvas_text_get_type(),
                              "text", _("Spike:"),
                              "x", 0.0,
                              "y", ALIGN_ITEM_1_SIDE_Y,
                              "size_points", FONT_POINT_SIZE,
                              "anchor", GTK_ANCHOR_NW,
                              "fill_color_rgba", 0x000000FF, NULL);

    }
    else
    {
        gnome_canvas_item_new(GNOME_CANVAS_GROUP(spike_item),
                              gnome_canvas_text_get_type(),
                              "text", _("Spike:"),
                              "x", 0.0,
                              "y", LINE_1_OFFSET,
                              "size_points", FONT_POINT_SIZE,
                              "anchor", GTK_ANCHOR_NW,
                              "fill_color_rgba", 0x000000FF, NULL);

        if(applet_size->auto_size)
        {
            xpos = 0.0;
            for(count = 0; count < 12; count++)
            {
                if(best_spike > spikeThresholds[count])
                {
                    current_colour = spikeColoursA[count];
                }
                else
                {
                    current_colour = spikeColoursU[count];
                }
                gnome_canvas_item_new(GNOME_CANVAS_GROUP
                                      (spike_item),
                                      gnome_canvas_rect_get_type(),
                                      "x1", xpos,
                                      "y1", LINE_2_G_OFFSET - 1,
                                      "x2", xpos + spike_elt_width,
                                      "y2", LINE_3_G_OFFSET - 1,
                                      "fill_color_rgba", current_colour,
                                      "outline_color_rgba", current_colour,
                                      NULL);
                gnome_canvas_item_new(GNOME_CANVAS_GROUP
                                      (spike_item),
                                      gnome_canvas_rect_get_type(),
                                      "x1", xpos,
                                      "y1", LINE_2_G_OFFSET - 1,
                                      "x2", xpos + spike_elt_width,
                                      "y2", LINE_3_G_OFFSET - 1,
                                      "fill_color", NULL,
                                      "outline_color_rgba", 0x000000FF,
                                      "width_units", 1.0, NULL);
                xpos += spike_elt_width + spike_gap_width;
            }
        }
        gnome_canvas_item_new(GNOME_CANVAS_GROUP(spike_item),
                              gnome_canvas_text_get_type(),
                              "text", spike_value,
                              "x", xpos2,
                              "y", LINE_2_OFFSET + 5.0,
                              "size_points", FONT_POINT_SIZE,
                              "anchor", GTK_ANCHOR_NORTH,
                              "fill_color_rgba", 0x999999FF, NULL);
    }
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(spike_item),
                          gnome_canvas_text_get_type(),
                          "text", spike_value,
                          "x", xpos2 + 1.0,
                          "y", LINE_2_OFFSET + 5.0 - 1.0,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NORTH,
                          "fill_color_rgba", 0x000000FF, NULL);
    g_free(spike_value);
    return spike_item;
}

GnomeCanvasItem *
default_units(skin_data *data,
              GnomeCanvasGroup * group,
              gdouble units_processed,
              seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *num_units_item = GNOME_CANVAS_ITEM(GNOME_CANVAS_GROUP
                                                        (gnome_canvas_item_new
                                                         (group,
                                                          gnome_canvas_group_get_type
                                                          (), "x", 0.0, "y",
                                                          0.0, NULL)));
    gchar      *units_label = _("Units:");
    gchar      *units_value = g_strdup_printf("%.0f", units_processed);

    if(applet_size->only_radar != 0)
    {
        //We are leaking things here - FIXME
        return NULL;
    }
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(num_units_item),
                          gnome_canvas_text_get_type(),
                          "text", units_label,
                          "x", 0.0,
                          "y", LINE_1_OFFSET,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NW,
                          "fill_color_rgba", 0x000000FF, NULL);
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(num_units_item),
                          gnome_canvas_text_get_type(),
                          "text", units_value,
                          "x", 0.0,
                          "y", LINE_2_OFFSET,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NW,
                          "fill_color_rgba", 0x003300FF, NULL);
    g_free(units_value);
    return num_units_item;
}

static gchar *
format_percentage_string(gdouble progress,
                         PanelAppletOrient orient)
{
    gchar      *percentage;

    if(orient == PANEL_APPLET_ORIENT_LEFT
       || orient == PANEL_APPLET_ORIENT_RIGHT)
    {
        percentage = g_strdup_printf("%2.1f%c", progress * 100, '%');
    }
    else
    {
        percentage = g_strdup_printf("%3.2f%c", progress * 100, '%');
    }
    return percentage;
}

GnomeCanvasItem *
default_progress(skin_data *data,
                 GnomeCanvasGroup * group,
                 gdouble progress,
                 seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *percentage_item = GNOME_CANVAS_ITEM(GNOME_CANVAS_GROUP
                                                         (gnome_canvas_item_new
                                                          (group,
                                                           gnome_canvas_group_get_type
                                                           (),
                                                           "x", 0.0, "y", 0.0,
                                                           NULL)));
    gchar      *percentage_label;
    gchar      *percentage_value;

    if(applet_size->only_radar != 0)
    {
        //We are leaking the percentage item here - fixme.
        return NULL;
    }

    if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        percentage_label = _("Prog:");
    }
    else
    {
        percentage_label = _("Progress:");
    }
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(percentage_item),
                          gnome_canvas_text_get_type(),
                          "text", percentage_label,
                          "x", 0.0,
                          "y", LINE_1_OFFSET,
                          "size_points", FONT_POINT_SIZE,
                          "anchor", GTK_ANCHOR_NW,
                          "fill_color_rgba", 0x000000FF, NULL);
    percentage_value =
        format_percentage_string(progress, applet_size->orientation);
    gnome_canvas_item_new(GNOME_CANVAS_GROUP(percentage_item),
                          gnome_canvas_text_get_type(), "text",
                          percentage_value, "x", 0.0, "y", LINE_2_OFFSET,
                          "size_points", FONT_POINT_SIZE, "anchor",
                          GTK_ANCHOR_NW, "fill_color_rgba", 0x003300FF, NULL);
    g_free(percentage_value);

    return percentage_item;
}

static gdouble
find_radar_x(seti_applet_size_t * applet_size)
{
    if(applet_size->auto_size)
    {
        if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
           || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
        {
            return 0.0;
        }
        else
        {
            return (gdouble) 128.0 *2;
        }
    }
    else
    {
        return 0.0;
    }
}

static gdouble
find_radar_y(seti_applet_size_t * applet_size)
{
    return 0.0;
}

GnomeCanvasItem *
default_red_radar(skin_data *data,
                  GnomeCanvasGroup * group,
                  seti_applet_size_t * applet_size)
{
    i_skin_data_t *the_images = (i_skin_data_t *)data;
    //First scale the radar image
    GdkPixbuf  *scaled_radar_red =
        gdk_pixbuf_scale_simple(the_images->the_red_radar, 128, 128, GDK_INTERP_BILINEAR);

    gdouble     radar_x = find_radar_x(applet_size);
    gdouble     radar_y = find_radar_y(applet_size);

    GnomeCanvasItem *rv;


    rv = gnome_canvas_item_new(group,
                               gnome_canvas_pixbuf_get_type(),
                               "pixbuf", scaled_radar_red,
                               "x", radar_x,
                               "y", radar_y,
                               "anchor", GTK_ANCHOR_NW,
                               "width", 128.0, "height", 128.0, NULL);
    g_message("(red)x,y %f,%f dx,dy %d,%d", radar_x, radar_y,
              applet_size->panel_size - 1, applet_size->panel_size - 1);

    return rv;
}

GnomeCanvasItem *
default_green_radar(skin_data *data,
                    GnomeCanvasGroup * group,
                    seti_applet_size_t * applet_size)
{
    i_skin_data_t *the_images = (i_skin_data_t *)data;
    //First scale the radar image
    GdkPixbuf  *scaled_radar_green =
        gdk_pixbuf_scale_simple(the_images->the_green_radar, 128, 128, GDK_INTERP_BILINEAR);

    gdouble     radar_x = find_radar_x(applet_size);
    gdouble     radar_y = find_radar_y(applet_size);

    GnomeCanvasItem *rv = gnome_canvas_item_new(group,
                                                gnome_canvas_pixbuf_get_type(),
                                                "pixbuf", scaled_radar_green,
                                                "x", radar_x,
                                                "y", radar_y,
                                                "width", 128.0,
                                                "height", 128.0,
                                                "anchor", GTK_ANCHOR_NW,
                                                NULL);

    g_message("(green)x,y %f,%f dx,dy %d,%d", radar_x, radar_y,
              applet_size->panel_size - 1, applet_size->panel_size - 1);

    return rv;
}

GnomeCanvasItem *
default_grey_radar(skin_data *data,
                   GnomeCanvasGroup * group,
                   seti_applet_size_t * applet_size)
{
    i_skin_data_t *the_images = (i_skin_data_t *)data;
    //First scale the radar image
    GdkPixbuf  *scaled_radar_grey =
        gdk_pixbuf_scale_simple(the_images->the_grey_radar, 128, 128, GDK_INTERP_BILINEAR);

    gdouble     radar_x = find_radar_x(applet_size);
    gdouble     radar_y = find_radar_y(applet_size);

    GnomeCanvasItem *rv = gnome_canvas_item_new(group,
                                                gnome_canvas_pixbuf_get_type(),
                                                "pixbuf", scaled_radar_grey,
                                                "x", radar_x,
                                                "y", radar_y,
                                                "width", 128.0,
                                                "height", 128.0,
                                                "anchor", GTK_ANCHOR_NW,
                                                NULL);

    g_message("(grey)x,y %f,%f dx,dy %hd,%hd", radar_x, radar_y,
              applet_size->panel_size - 1, applet_size->panel_size - 1);

    return rv;
}

gchar      *
default_tooltip(skin_data *data,
                setiapplet_current_unit * unit,
                setiapplet_user_info * user)
{
    gchar      *seti_state;
    gchar      *time_string;

    time_string =
        format_cpu_string(user->cumulative_cpu_time, PANEL_APPLET_ORIENT_UP);
    seti_state =
        g_strdup_printf("%.2f%%, %s%.2f, %s %.2f, %s %.2f, %s%d, %s%s",
                        unit->percent_complete * 100, _("Spike:"),
                        unit->best_spike, _("Gaussian Power"),
                        unit->best_gaussian_power, _("Gaussian Score"),
                        unit->best_gaussian_score, _("Units:"),
                        (gint) user->units_processed, _("CPU Time:"),
                        time_string);

    g_free(time_string);
    return seti_state;
}

GnomeCanvasItem *
default_title(skin_data *data,
              GnomeCanvasGroup * group,
              seti_applet_size_t * applet_size)
{
    GnomeCanvasItem *text_item;

    if(applet_size->only_radar != 0)
    {
        return NULL;
    }

    text_item = gnome_canvas_item_new(group, gnome_canvas_text_get_type(),
                                      "text",
                                      "seti@home",
                                      "x", 0.0,
                                      "y", TITLE_OFFSET,
                                      "anchor", GTK_ANCHOR_NW,
                                      "fill_color_rgba", 0x003300FF,
                                      "size_points", FONT_POINT_SIZE, NULL);

    if(applet_size->orientation == PANEL_APPLET_ORIENT_LEFT
       || applet_size->orientation == PANEL_APPLET_ORIENT_RIGHT)
    {
        gdouble     theAffine[6];
        //Rotate the sucker through 90 degrees (anti-clockwise)
        art_affine_identity(theAffine);
        art_affine_rotate(theAffine, 270.0);
        gnome_canvas_item_affine_relative(text_item, theAffine);
    }
    return text_item;
}

seti_applet_skin_t *
new_default_skin(void)
{
    seti_applet_skin_t *rv = g_new0(seti_applet_skin_t, 1);

    rv->private = internal_default_setup();
    rv->setup = default_setup;
    rv->cleanup = default_cleanup;
    rv->cputimeoverall = default_cputime_overall;
    rv->cputimecurrent = default_cputime_current;
    rv->gaussian = default_gaussian;
    rv->spike = default_spike;
    rv->units = default_units;
    rv->progress = default_progress;
    rv->radar_red = default_red_radar;
    rv->radar_green = default_green_radar;
    rv->radar_grey = default_grey_radar;
    rv->tooltip = default_tooltip;
    rv->title = default_title;
    return rv;
}

free_default_skin(seti_applet_skin_t *skin)
{
    if(skin)
    {
        if(skin->private)
        {
            g_free(skin->private);
        }
        g_free(skin);
    }
}
