# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2010 Lukas Hetzenecker <LuHe@gmx.at>

from PyQt4.QtCore import *

class Enum(object):
    def __init__(self, names, separator=None,  beginning=0):
        self.names = names.split(separator)
        for value, name in enumerate(self.names):
            setattr(self, name, value+beginning)
    def tuples(self):
        return tuple(enumerate(self.names))

Roles = Enum("MessageRole ContactRole DeviceRole ApplicationRole VersionRole TypeRole LinkRole",  beginning=Qt.UserRole+10)
MessageType = Enum("Unknown Incoming Outgoing All")
MessageState = Enum("Unknown Created Pending Sending SendFailed SendOk")
MessagePriority = Enum("Low High Medium")
StatisticFor = Enum("Unknown Periods Contacts")
StatisticPeriod = Enum("Unknown Years YearsAndMonths Months Days Weekdays Hours")
StatisticOrderBy = Enum("Unknown Number Name")
StatisticGraphLayout = Enum("Stacked Separate")
PopupTypes = Enum("Unknown Message ConnectionCompleted ConnectionClosed")

# For message export
Contacts = Enum("All ContactsWithMessages Filter None_")
Period = Enum("Daily Monthly Yearly All")
Order = Enum("ASC DESC")
Graph = Enum("Yes No")
GraphFormat = Enum("PNG SVG")
Legend = Enum("Yes No")
Thumbnails = Enum("Yes No")

class HistoryDates(object):
    def __init__(self):
        self.__dates = dict()

    def addYear(self,  year):
        if not year in self.__dates:
            self.__dates[year] = dict()

    def addMonth(self,  year,  month):
        self.addYear(year)

        if not month in self.__dates[year]:
           self.__dates[year][month] = list()

    def addDay(self,  year,  month,  day):
        self.addMonth(year,  month)

        if not day in self.__dates[year][month]:
            self.__dates[year][month].append(day)

    def years(self):
        years = self.__dates.keys()
        years.sort(reverse=True)
        return years

    def months(self,  year):
        months = self.__dates[year].keys()
        months.sort(reverse=True)
        return months

    def days(self,  year,  month):
        days = self.__dates[year][month]
        days.sort(reverse=True)
        return days

class ContactField(QObject):
    def __init__(self,  type,  location):
        super(ContactField,  self).__init__()

        self.setType(type)
        self.setLocation(location)

    def __repr__(self):
        return "ContactField('%s', '%s')" % (self.__type,  self.__location)

    def __hash__(self):
        return hash(self.__type + ":" + self.__location)

    def __eq__(self,  b):
        return self.type() == b.type() and self.location() == b.location()

    def setLocation(self,  location):
        assert location in ["none",  "home",  "work"],  "wrong argument '%s'" % location
        self.__location = location

    @staticmethod
    def types():
        return ["last_name",  "first_name",  "job_title",  "company_name",  "phone_number",  "mobile_number",
                               "pager_number",  "fax_number",  "email_address",  "url",  "postal_address", "po_box",
                               "extended_address",  "street_address",  "postal_code",  "city",  "state",  "country",
                               "dtmf_string",  "date",  "note",  "thumbnail_image",  "prefix",  "suffix",  "second_name",
                               "video_number",  "voip",  "push_to_talk",  "share_view",  "sip_id",  "personal_ringtone",
                               "locationid_indication",  "picture",  "first_name_reading",  "last_name_reading",
                               "speed_dial",  "voice_tag",  "wvid"]

    def setType(self,  type):
        # There was a Typo in 0.3.90:
        # TODO: Remove
        if type == "dmtf_string":
            type = "dtmf_string"
            
        assert type in self.types(),  "wrong argument '%s'" % type

        self.__type = type

    def isPicture(self):
        return self.type() == "thumbnail_image"
    
    def isDate(self):
        return self.type() == "date"
    
    def location(self):
        return self.__location

    def type(self):
        return self.__type

    def toString(self,  num=1,  printLocation=False):
        if self.type() == "last_name":
            string = self.tr("Last name:")
        elif self.type() == "first_name":
            string = self.tr("First name:")
        elif self.type() == "job_title":
            string = self.tr("Job title:")
        elif self.type() == "company_name":
            string = self.tr("Company:")
        elif self.type() == "phone_number":
            string = self.tr("Telephone:")
        elif self.type() == "mobile_number":
            string = self.tr("Mobile:")
        elif self.type() == "pager_number":
            string = self.tr("Pager:")
        elif self.type() == "fax_number":
            string = self.tr("Fax:")
        elif self.type() == "email_address":
            string = self.tr("E-mail:")
        elif self.type() == "url":
            string = self.tr("Web address:")
        #elif self.type() == "postal_address":
        #    string = self.tr("Street:")
        elif self.type() == "po_box":
            string = self.tr("Post-office box:")
        elif self.type() == "extended_address":
            string = self.tr("Extension:")
        elif self.type() == "street_address":
            string = self.tr("Street:")
        elif self.type() == "postal_code":
            string = self.tr("Postal/ZIP code:")
        elif self.type() == "city":
            string = self.tr("City:")
        elif self.type() == "state":
            string = self.tr("State/Province:")
        elif self.type() == "country":
            string = self.tr("Country/Region:")
        elif self.type() == "dtmf_string":
            string = self.tr("DTMF:")
        elif self.type() == "date":
            string = self.tr("Birthday:")
        elif self.type() == "note":
            string = self.tr("Note:")
        elif self.type() == "thumbnail_image":
            string = self.tr("Thumbnail image:")
        elif self.type() == "prefix":
            string = self.tr("Title:")
        elif self.type() == "suffix":
            string = self.tr("Suffix:")
        elif self.type() == "second_name":
            string = self.tr("Nickname:")
        elif self.type() == "video_number":
            string = self.tr("Video call:")
        elif self.type() == "voip":
            string = self.tr("Internet telephone:")
        elif self.type() == "push_to_talk":
            string = self.tr("Push to talk:")
        elif self.type() == "share_view":
            string = self.tr("Share view:")
        elif self.type() == "sip_id":
            string = self.tr("SIP:")
        elif self.type() == "personal_ringtone":
            string = self.tr("Personal ringtone:")
        else:
            string = self.type() + ":"

        location = ""
        if printLocation:
            if self.location() == "home":
                location = self.tr(" (%1) ").arg(self.tr("Home"))
            elif self.location() == "work":
                location = self.tr(" (%1) ").arg(self.tr("Business"))

        if num > 1:
            string = self.tr("%1 (%2):").arg(unicode(string)[:-1] + location).arg(num)
        else:
            string = unicode(string)[:-1] + location + ":"

        return string

class __ContactFields(object): pass
ContactFields = __ContactFields()
for name in ContactField.types():
    setattr(ContactFields, name.title().replace("_",  ""), ContactField(name,  "none"))

class Contact(object):
    def __init__(self,  id = 0,  idOnPhone = 0,  name = "", ignore=False,  favorite=False,  values=list(),  internalValues=list()):
        self.setId(id)
        self.setIdOnPhone(idOnPhone)
        self.setName(name)
        self.setIgnored(ignore)
        self.setFavorite(favorite)
        #self.__values = values
        self.__values = list()
        self.__internalValues = dict()

    def __repr__(self):
        return "Contact(%i, %i, '%s', %s, %s, %s, %s)" % (self.__id, self.__idOnPhone,  self.__name,  self.__ignored,  self.__favorite,
                                                          "",  self.__internalValues)

    def __contains__(self,  field):
        return self.contains(field)

    def __eq__(self,  b):
        if self.id() and b.id():
            return self.id() == b.id()
        elif self.name() and b.name():
            return self.name() == b.name()
        else:
            return False

    def __ne__(self,  b):
        return not self.__eq__(b)

    def __iter__(self):
        for item in self.__values:
            yield item

    def __nonzero__(self):
        return self.id() != 0 or self.idOnPhone() != 0

    def setId(self,  id):
        assert isinstance(id,  int),  "expected an int"
        self.__id = id

    def setIdOnPhone(self,  idOnPhone):
        assert isinstance(idOnPhone,  int),  "expected an int"
        self.__idOnPhone = idOnPhone

    def setName(self,  name):
        assert isinstance(name,  basestring),  "expected a string or unicode"
        self.__name = name

    def setIgnored(self,  ignore):
        self.__ignored = ignore

    def setFavorite(self,  favorite):
        self.__favorite = favorite

    def addValue(self,  field,  value):
        assert isinstance(field,  ContactField),  "expected a ContactField instance"
        assert isinstance(value,  unicode),  "expected a unicode"
        self.__values.append( (field,  value) )

    def addInternalValue(self,  key,  value):
        self.__internalValues[key] = value

    def internalValue(self,  key):
        return self.__internalValues[key]

    def internalValues(self):
        return self.__internalValues.keys()

    def ignore(self):
        self.setIgnored(True)

    def id(self):
        return self.__id

    def idOnPhone(self):
        return self.__idOnPhone

    def name(self):
        return self.__name

    def values(self):
        return self.__values

    def types(self,  location=None):
        if location == None:
            return set( value[0].type() for value in self.__values )
        else:
            ret = list()
            for value in self.__values:
                if value[0].location() == location:
                    ret.append(value[0].type())
            return set(ret)

    def isIgnored(self):
        return self.__ignored

    def isFavorite(self):
        return self.__favorite

    def value(self,  type,  location=None):
        tmp = list()
        for item in self.__values:
            if (location and item[0].location() == location) or (location == None):
                if item[0].type() == type:
                    tmp.append( item[1] )

        return tmp

#    def remove(self,  type):
#        tmp = list()
#        for item in self.__values:
#            if item[0] != type:
#                tmp.append(item)
#        self.__values = tmp

    def contains(self,  type):
        return type in self.types()

class Device(object):
    def __init__(self,  id = 0, name = "",  bluetoothAddress = "00:00:00:00:00:00",  port = 0,  values=list()):
        self.setId(id)
        self.setName(name)
        self.setBluetoothAddress(bluetoothAddress)
        self.setPort(port)
        #self.__values = values
        self.__values = list()

    def __repr__(self):
        return "Device(%i, '%s', '%s', %i, %s)" % (self.__id,  self.__name,  self.__bluetoothAddress,  self.__port,  self.__values)

    def __contains__(self,  type):
        return self.contains(type)

    def setId(self,  id):
        assert isinstance(id,  int),  "expected an int"
        self.__id = id

    def setName(self,  name):
        assert isinstance(name,  basestring),  "expected a string or unicode"
        self.__name = name

    def setBluetoothAddress(self,  bluetoothAddress):
        assert isinstance(bluetoothAddress,  str),  "expected a string"
        self.__bluetoothAddress = bluetoothAddress

    def setPort(self,  port):
        assert isinstance(port,  int),  "expected an int"
        self.__port = port

    def addValue(self,  type,  value):
        if (type != None) and (value != None):
            self.__values.append( [type,  value] )

    def id(self):
        return self.__id

    def name(self):
        return self.__name

    def bluetoothAddress(self):
        return self.__bluetoothAddress

    def port(self):
        return self.__port

    def values(self):
        return self.__values

    def types(self):
        return set( value[0] for value in self.__values )

    def value(self,  type):
        return self.__values[ [ value[0] for value in self.__values ].index(type) ][1]

    def contains(self,  type):
        return type in self.types()

    def clear(self):
        self.__values = list()

class MessageStates(object):
    def __init__(self,  states=list()):
        # self.__states = states
        self.__states = list()
        self.__state = MessageState.Unknown

    def __repr__(self):
        return "MessageStates(%s)" % self.__states

    def __iter__(self):
        for i in self.__states:
            yield i

    def addState(self, state, message="",  date=None):
        if not date:
            date = QDateTime.currentDateTime()
        tmp = [date, state,  message]
        self.__states.append(tmp)
        if state != MessageState.Unknown:
            self.__state = state
        return tmp

    def created(self):
        return self.__state == MessageState.Created

    def sending(self):
        return self.__state == MessageState.Sending

    def pending(self):
        return self.__state == MessageState.Pending

    def sendOk(self):
        return self.__state == MessageState.SendOk

    def sendFailed(self):
        return self.__state == MessageState.SendFailed

    def states(self):
        for i in self.__states:
            yield i

class Message(QObject):
    def __init__(self, id = 0,  idOnPhone = 0,  type = MessageType.Unknown, priority = MessagePriority.Medium,  \
                  device = Device(),  contact = Contact(), states = MessageStates(),  dateTime = 0,  message = ""):
        super(Message,  self).__init__()

        self.setContact(contact)
        self.setDevice(device)
        self.setStates(MessageStates())

        self.setId(id)
        self.setIdOnPhone(idOnPhone)
        self.setType(type)
        self.setPriority(priority)
        self.setDateTime(dateTime)
        self.setMessage(message)

    def __repr__(self):
        return "Message(%i, %i, %s, %s, %s, %s, %s, %s, '%s')" % (self.__id,  self.__idOnPhone,  repr(self.__type),
                    repr(self.__priority),  repr(self.__device),  repr(self.__contact), repr(self.__states), repr(self.__dateTime),
                    self.__message)

    def __cmp__(self,  b):
        if self.id() and b.id():
            return self.id() - b.id()
        elif self.idOnPhone() and b.idOnPhone():
            return self.idOnPhone() - b.idOnPhone()
        else:
            return -1

    def setId(self,  id):
        assert isinstance(id,  int),  "expected an int"
        self.__id = id

    def setIdOnPhone(self,  idOnPhone):
        assert isinstance(idOnPhone,  int),  "expected an int"
        self.__idOnPhone = idOnPhone

    def setType(self,  _type):
        assert isinstance(_type,  int),  "expected an int"
        self.__type = _type

    def setPriority(self,  priority):
        assert isinstance(priority,  int),  "expected an int"
        self.__priority = priority

    def setDevice(self,  device):
        assert isinstance(device,  Device),  "expected a Device instance"
        self.__device = device

    def setContact(self,  contact):
        assert type(contact) == type(Contact()),  "expected a Contact instance"
        self.__contact = contact

    def setStates(self, states):
        assert isinstance(states,  MessageStates),  "expected a MessageStates instance"
        self.__states = states

    def setDateTime(self,  dateTime):
        if isinstance(dateTime,  float):
            self.__dateTime = QDateTime.fromTime_t(int(dateTime))
        elif isinstance(dateTime,  int):
            self.__dateTime = QDateTime.fromTime_t(int(float(dateTime)))
        elif isinstance(dateTime,  QDateTime):
            self.__dateTime = dateTime
        else:
            assert False,  "expected a float, QDateTime or int"

    def setMessage(self,  message):
        assert isinstance(message,  basestring),  "expected a string or unicode"
        self.__message = message

    def addState(self,  state, message="",  date=None):
        last = self.states().addState(state,  message,  date)
        self.emit(SIGNAL("messageStateAdded"),  self,  last)

    def id(self):
        return self.__id

    def idOnPhone(self):
        return self.__idOnPhone

    def type(self):
        return self.__type

    def priority(self):
        return self.__priority

    def device(self):
        return self.__device

    def contact(self):
        return self.__contact

    def states(self):
        return self.__states

    def dateTime(self):
        return self.__dateTime

    def message(self):
        return self.__message

    def messageForPhone(self):
        return self.__message.replace(u'\n', u'\u2029') # LINE FEED (\u000a) replaced by PARAGRAPH SEPARATOR (\u2029)

class StatisticsRequest(object):
    def __init__(self,  statisticFor = StatisticFor.Unknown,  contact = Contact(),  period = StatisticPeriod.Unknown,  \
                 type = MessageType.Unknown,  start = None,  duration = None, orderBy = StatisticOrderBy.Number,  \
                 year = None,  month = None,  day = None):

        self.setStatisticFor(statisticFor)
        self.setContact(contact)
        self.setPeriod(period)
        self.setType(type)
        self.setRange(start, duration)
        self.setOrderBy(orderBy)
        self.setYear(year)
        self.setMonth(month)
        self.setDay(day)

    def __nonzero__(self):
        return self.statisticsFor() != StatisticFor.Unknown

    def setStatisticFor(self,  statisticFor):
        assert isinstance(statisticFor,  int),  "expected an int"
        self.__statisticsFor = statisticFor

    def setContact(self,  contact):
        assert isinstance(contact,  Contact),  "expected a Contact instance"
        self.__contact = contact

    def setPeriod(self,  period):
        assert isinstance(period,  int),  "expected a QDateTime instance"
        self.__period = period

    def setType(self,  _type):
        assert isinstance(_type,  int),  "expected an int"
        self.__type = _type

    def setRange(self,  start,  duration):
        assert type(start) in  [ type(int()), type(None) ],  "expected an int or NoneType"
        assert type(duration) in  [ type(int()), type(None) ],  "expected an int or NoneType"
        self.__start = start
        self.__duration = duration

    def setOrderBy(self,  orderBy):
        assert isinstance(orderBy,  int),  "expected an int"
        self.__orderBy = orderBy

    def setYear(self,  year):
        assert type(year) in  [ type(int()), type(None) ],  "expected an int or NoneType"
        self.__year = year

    def setMonth(self,  month):
        assert type(month) in  [ type(int()), type(None) ],  "expected an int or NoneType"
        self.__month = month

    def setDay(self,  day):
        assert type(day) in  [ type(int()), type(None) ],  "expected an int or NoneType"
        self.__day = day

    def statisticsFor(self):
        return self.__statisticsFor

    def contact(self):
        return self.__contact

    def period(self):
        return self.__period

    def type(self):
        return self.__type

    def range(self):
        return self.__start,  self.__duration

    def orderBy(self):
        return self.__orderBy

    def year(self):
        return self.__year

    def month(self):
        return self.__month

    def day(self):
        return self.__day

class StatisticResponseLine(object):
    def __init__(self,  values):
        self.__values = values

    def __str__(self):
        return "StatisticResponseLine('%s')" % repr(self.__values)

    def year(self):
        return self.__values[0]

    def month(self):
        return self.__values[1]

    def day(self):
        return self.__values[2]

    def weekday(self):
        return self.__values[3]

    def hour(self):
        return self.__values[4]

    def incoming(self):
        return self.__values[5]

    def outgoing(self):
        return self.__values[6]

    def total(self):
        return self.__values[7]

class StatisticResponse(object):
    def __init__(self,  request):
        self.__values = list()
        self.__request = request
        self.__containsChecked = False
        self.__year = False
        self.__month = False
        self.__day = False
        self.__weekday = False
        self.__hour = False
        self.__incoming = False
        self.__outgoing = False
        self.__total = False

    def __len__(self):
        return len(self.__values)
    
    def __getitem__(self,  key):
        return self.__values[key]

    def __iter__(self):
        for item in self.__values:
            yield item

    def __bool__(self):
        return len(self) != 0

    def append(self,  year,  month,  day,  weekday,  hour,  incoming,  outgoing,  total):
        self.__values.append( StatisticResponseLine( [year,  month,  day,  weekday,  hour,  incoming,  outgoing,  total] ) )

        if not self.__containsChecked:
            self.__year = year != None
            self.__month = month != None
            self.__day = day != None
            self.__weekday = weekday != None
            self.__hour = hour != None
            self.__incoming = incoming != None
            self.__outgoing = outgoing != None
            self.__total = total != None

            self.__containsChecked = True

    def request(self):
        return self.__request

    def values(self):
        for i in self.__values:
            yield i

    def containsYear(self):
        return self.__year

    def containsMonth(self):
        return self.__month

    def containsDay(self):
        return self.__day

    def containsWeekday(self):
        return self.__weekday

    def containsHour(self):
        return self.__hour

    def containsIncoming(self):
        return self.__incoming

    def containsOutgoing(self):
        return self.__outgoing

    def containsTotal(self):
        return self.__total
