#ifdef __APPLE__
#include <Carbon/Carbon.h>
#include "seaview.h"
#include <FL/x.H>
#include <FL/Fl_Sys_Menu_Bar.H>
#include <FL/Fl_Help_View.H>
#include <FL/filename.H>
#include "FL/fl_draw.H"
#include "src/Fl_Font.H"
#include <sys/stat.h>
#include <unistd.h>

// included functions
char *mac_fname_to_roman(const char *in);
void fl_ringbell(int a);
extern "C" {
	char *MG_GetBundleResourcesDir(void);
	void PtoC(const void *in, void *out);
	void CtoP(const void *in, void *out);
	void MGinit_apple_events(void);
	void add_apropos(const char *progname);
	void show_apropos(Fl_Widget *, void *unused);
	Boolean AppleEventsInstalled ();
	OSErr  MyHandleODoc (const AppleEvent *theAppleEvent, AppleEvent* reply, long handlerRefCon);
	OSErr  MyHandlePDoc (const AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon);
	OSErr  MyHandleOApp (const AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon);
	OSErr MyGotRequiredParams (const AppleEvent *theAppleEvent);
	void MG_apple_inits(void);
}
void set_seaview_modified(SEA_VIEW *view, int ismodified);
void cre_sys_menu(void );
void windowmenuitem_callback(Fl_Widget *o, void *data);
int find_windowmenuitem(Fl_Window *w);
int add_windowmenuitem(const char *name, Fl_Window *w);
void rename_windowmenuitem(const char *name, int rank);
void delete_windowmenuitem(int rank);
static void MyNavEventProc_Plus (NavEventCallbackMessage callBackSelector,
										NavCBRecPtr callBackParms, void *callBackUD);
char *mac_GetOutputFName_Plus(const char *dfault, const char *message, int, const char *directory);
void MG_fl_draw(const char *txt, int x, int y);
void MG_fl_line(int x, int y, int x1, int y1);
void MG_fl_rect(int x, int y, int w, int h);
void MG_fl_rectf(int x, int y, int w, int h);
void MG_fl_font(int fontrank, int size);
void MG_fl_color(Fl_Color color);
float MG_fl_width(char *txt, int l);
void *MG_PrepareCopy(int w, int h);
void MG_CompleteCopy(void *mypicture);
static size_t MyPutBytes (void* info, const void* buffer, size_t count);
QDPictRef MyPictToQDPict(PicHandle mypicture);
void MyCopyPictToClipboard (PicHandle mypicture);
int MG_StartPrinting (int pagecount, int *frompage, int *topage);
int MG_PrintingRect(int *x, int *y, int *w, int *h);
int MG_StartPage (void);
int MG_ClipPrintingRect(int x, int y, int w, int h);
int MG_EndPage (void);
void MG_EndPrinting (void);
void *prepare_copy_pdf_and_pict(int w, int h);
void complete_copy_pdf_and_pict(void *data);
void draw_rotated_string(const char *txt, int x, int y, float degrees /*clockwise*/);

/* extern functions */
extern void hide_window_callback(Fl_Widget *ob, void *data);
extern char *get_res_value(const char *name, const char *def_value);
extern Fl_Window *use_initial_file(SEA_VIEW *view, char *masename, int doing_dnd);
extern char *get_next_help_line(void *in, char *line, int lline);
extern int printout_block, printout_fontsize;
extern int printout_vary, printout_black;
extern paperformat printout_pageformat;

/* globals */
static int MG_doing_copy = 0;

void fl_ringbell(int a)
{
	AlertSoundPlay();
}


char *mac_fname_to_roman(const char *in)
/* passage codage pathname vers codage MacRoman qui semble necessaire pour display ecran 
 */
{
	static char out[250];
	CFStringRef mycfs;
	Boolean ok;
	/* j'ai cru comprendre que les pathnames sont codes en UTF8 */
	mycfs = CFStringCreateWithCString(NULL, in, kCFStringEncodingUTF8);
	/* et que MacRoman est utilise pour display */
	ok = CFStringGetCString(mycfs, out, sizeof(out), kCFStringEncodingMacRoman);
	CFRelease(mycfs);
	return (ok ? out : (char *)in);
}

/*
pid_t get_desc_process(pid_t ascend)
//returns the pid of the first descending process from process of pid ascend
//or 0 if no descendant
{
	FILE *pipe;
	char command[] = "/bin/ps -U $USER -o pid,ppid ";
	char line[2000], *p;
	long long pid, ppid;
	
	pipe = popen(command, "r");
	fgets(line, sizeof(line), pipe); // skip first line
	while(TRUE)	{
		p = fgets(line, sizeof(line), pipe);
		pid = 0;
		if(p == NULL) break;
		sscanf(line, "%lld%lld", &pid, &ppid);
		if( (pid_t)ppid == ascend) break;
	}
	pclose(pipe);
	return (pid_t) pid;
}


void kill_last_desc(pid_t ascend)
//kills the deepest descendant process from process ascend, if any
{
	pid_t son, father;
	
	father = ascend;
	do {
		son = get_desc_process(father);
		if(son != 0) father = son;
	}
	while(son != 0);
	if(father != ascend) kill(father, SIGKILL);
}
*/

char *MG_GetBundleResourcesDir(void)
{
	ProcessSerialNumber psn;
	FSRef fsref;
	static char fname[300];
	
	GetCurrentProcess(&psn);
	GetProcessBundleLocation(&psn, &fsref);
	FSRefMakePath(&fsref, (UInt8 *)fname, sizeof(fname)-1);
	strcat(fname, "/Contents/Resources"); 
	return fname;
}


static char dialog_start_dir[FL_PATH_MAX]; //shared by MyNavEventProc_Plus & mac_GetOutputFName_Plus

static void MyNavEventProc_Plus (NavEventCallbackMessage callBackSelector,
										NavCBRecPtr callBackParms,
										void *callBackUD)
{
	OSErr err;
	static Handle gDitlList = NULL;
	static DialogItemIndex index;
	static DialogRef ref;
	static ControlRef var_control = NULL;
	Rect rect; Handle h; DialogItemType dtype; Str255 text; ControlRef control;
	char text2[256];
	
	if(callBackSelector == kNavCBCustomize) {
		if(callBackParms->customRect.bottom == 0) 
			callBackParms->customRect.bottom = callBackParms->customRect.top + 58;
		if(callBackParms->customRect.right == 0) 
			callBackParms->customRect.right = callBackParms->customRect.left + 425;
	}
	else if(callBackSelector == kNavCBStart) {
		FSRef fsref;
		if(dialog_start_dir[0] != 0 && FSPathMakeRef((const UInt8 *)dialog_start_dir, &fsref, NULL) == noErr) {
			AEDesc desc;
			err = AECreateDesc(typeFSRef, &fsref, sizeof(FSRef), &desc);
			if(err == noErr)  {
				NavCustomControl(callBackParms->context, kNavCtlSetLocation, &desc);
				AEDisposeDesc(&desc);
				}
		}
		gDitlList = GetResource ('DITL', 128);
		if(gDitlList != NULL) err = NavCustomControl(callBackParms->context, 
													 kNavCtlAddControlList, gDitlList);
		err = NavCustomControl(callBackParms->context, kNavCtlGetFirstControlID, &index);
		index++;
		ref = GetDialogFromWindow(callBackParms->window);
		sprintf(text2,"%2d", printout_block);
		CtoP(text2, text);
		GetDialogItem(ref, index + 1, &dtype, &h, &rect);
		SetDialogItemText(h, text);
		sprintf(text2,"%2d", printout_fontsize);
		CtoP(text2, text);
		GetDialogItem(ref, index + 3, &dtype, &h, &rect);
		SetDialogItemText(h, text);
		if(printout_black == PDF_COLOR) {
			err = GetDialogItemAsControl(ref, index + 4, &control);
			if(err == 0) SetControl32BitValue(control, 1);
		}
		else if(printout_black == PDF_BW) {
			err = GetDialogItemAsControl(ref, index + 8, &control);
			if(err == 0) SetControl32BitValue(control, 1);
		}
		else if(printout_black == TEXT_ONLY) {
			err = GetDialogItemAsControl(ref, index + 9, &control);
			if(err == 0) SetControl32BitValue(control, 1);
		}
		
		err = GetDialogItemAsControl(ref, index + 5, &control);
		if(err == 0) SetControl32BitValue(control, printout_pageformat == 0);
		err = GetDialogItemAsControl(ref, index + 6, &control);
		if(err == 0) SetControl32BitValue(control, printout_pageformat == 1);
		
		err = GetDialogItemAsControl(ref, index + 7, &var_control);
		if(err == 0) SetControl32BitValue(var_control, printout_vary);
		if(*(int *)callBackUD) ActivateControl(var_control);
		else DeactivateControl(var_control);
	}
	else if(callBackSelector == kNavCBTerminate) {
		if(gDitlList != NULL) {
			GetDialogItem(ref, index + 1, &dtype, &h, &rect);
			GetDialogItemText(h, text);
			PtoC(text, text2);
			sscanf(text2,"%d", &printout_block);
			GetDialogItem(ref, index + 3, &dtype, &h, &rect);
			GetDialogItemText(h, text);
			PtoC(text, text2);
			sscanf(text2,"%d", &printout_fontsize);
			ReleaseResource( gDitlList );
		}
		var_control = NULL;
	}
	else if(callBackSelector == kNavCBEvent ) {
		if(var_control != NULL && callBackParms->eventData.itemHit == index + 7) {
			printout_vary = (printout_vary + 1) % 2;
			SetControl32BitValue(var_control, printout_vary);
		}
		else if(callBackParms->eventData.itemHit == index + 4) {
			if(printout_black != PDF_COLOR) {
				printout_black = PDF_COLOR;
				GetDialogItemAsControl(ref, index + 4, &control);
				SetControl32BitValue(control, 1);
				GetDialogItemAsControl(ref, index + 8, &control);
				SetControl32BitValue(control, 0);
				GetDialogItemAsControl(ref, index + 9, &control);
				SetControl32BitValue(control, 0);
			}
		}
		else if(callBackParms->eventData.itemHit == index + 8) {
			if(printout_black != PDF_BW) {
				printout_black = PDF_BW;
				GetDialogItemAsControl(ref, index + 4, &control);
				SetControl32BitValue(control, 0);
				GetDialogItemAsControl(ref, index + 8, &control);
				SetControl32BitValue(control, 1);
				GetDialogItemAsControl(ref, index + 9, &control);
				SetControl32BitValue(control, 0);
			}
		}
		else if(callBackParms->eventData.itemHit == index + 9) {
			if(printout_black != TEXT_ONLY) {
				printout_black = TEXT_ONLY;
				GetDialogItemAsControl(ref, index + 4, &control);
				SetControl32BitValue(control, 0);
				GetDialogItemAsControl(ref, index + 8, &control);
				SetControl32BitValue(control, 0);
				GetDialogItemAsControl(ref, index + 9, &control);
				SetControl32BitValue(control, 1);
			}
		}
		else if(callBackParms->eventData.itemHit == index + 5) {
			if(printout_pageformat == LETTER) {
				GetDialogItemAsControl(ref, index + 5, &control);
				SetControl32BitValue(control, 1);
				GetDialogItemAsControl(ref, index + 6, &control);
				SetControl32BitValue(control, 0);
				printout_pageformat = A4;
			}
		}
		else if(callBackParms->eventData.itemHit == index + 6) {
			if(printout_pageformat == A4) {
				GetDialogItemAsControl(ref, index + 6, &control);
				SetControl32BitValue(control, 1);
				GetDialogItemAsControl(ref, index + 5, &control);
				SetControl32BitValue(control, 0);
				printout_pageformat = LETTER;
			}
		}
	}
	return;
}


char *mac_GetOutputFName_Plus(const char *dfault, const char *message, int use_only_button, const char *directory)
{
    OSErr               anErr = noErr;
    NavReplyRecord      reply;
    NavDialogRef        navDialog;
    NavDialogCreationOptions    dialogOptions;
    OSType              fileTypeToSave = 'TEXT';
    OSType              creatorType;
    NavEventUPP         eventProc = NewNavEventUPP (MyNavEventProc_Plus);
	char *rsult = NULL;
	static int *p_use;
    Boolean ok;
	static char pathname[FL_PATH_MAX];
	
    anErr = NavGetDefaultDialogCreationOptions (&dialogOptions);
	if(anErr != noErr) return FALSE;
    dialogOptions.windowTitle = CFStringCreateWithCString(NULL, message, kCFStringEncodingMacRoman);
    dialogOptions.optionFlags |= kNavNoTypePopup;
    dialogOptions.optionFlags &= ~ kNavAllowStationery;
    dialogOptions.saveFileName = CFStringCreateWithCString(NULL, dfault, kCFStringEncodingUTF8); 
	if(directory != NULL) strcpy(dialog_start_dir, directory);
	else dialog_start_dir[0] = 0;
	
	creatorType = kNavGenericSignature;
	p_use = &use_only_button;
	anErr = NavCreatePutFileDialog( &dialogOptions, fileTypeToSave, creatorType, eventProc, p_use, &navDialog );
	CFRelease(dialogOptions.saveFileName);
	if(anErr != noErr) return NULL;
	
	anErr = NavDialogRun( navDialog );
	if(anErr != noErr) return NULL;
	
	anErr = NavDialogGetReply( navDialog, &reply );
	
	if (anErr == noErr && reply.validRecord) {
		FSRef fileRef;
		CFURLRef directoryURL, fileURL;
		
		anErr = AEGetNthPtr( &reply.selection, 1, typeFSRef, NULL, NULL, &fileRef, sizeof( FSRef ), NULL );
		if (anErr == noErr)
		{
			directoryURL = CFURLCreateFromFSRef( kCFAllocatorDefault, &fileRef );
			// append new filename to this path
			fileURL = CFURLCreateCopyAppendingPathComponent( kCFAllocatorDefault, directoryURL, reply.saveFileName, false );
			ok = CFURLGetFileSystemRepresentation(fileURL, true, (UInt8 *)pathname, FL_PATH_MAX);
			CFRelease(fileURL); CFRelease(directoryURL);
			if(ok) rsult = pathname;
		}
	}
	DisposeNavEventUPP(eventProc);
	NavDisposeReply( &reply );
	NavDialogDispose( navDialog );
	return rsult;
}


void PtoC(const void *in, void *out)
{
	char *vin = (char *)in;
	char *vout = (char *)out;
	int l = *vin;
	
	if(l > 0) memcpy(vout, vin + 1, l);
	vout[l] = 0;
}


void CtoP(const void *in, void *out)
{
	char *vout = (char *)out;
	int l;
	if(in == NULL) l = 0;
	else	{
		l = strlen( (char *)in );
		memcpy(vout + 1, in, l);
	}
	vout[0] = l;
}


void MGinit_apple_events(void)
{
	OSErr err;
	AEEventHandlerUPP	gAEEventHandlerUPPODoc;
	AEEventHandlerUPP	gAEEventHandlerUPPOApp;
	AEEventHandlerUPP	gAEEventHandlerUPPPDoc;
    Boolean aEvents;
	aEvents = AppleEventsInstalled();
	if (aEvents) {
		gAEEventHandlerUPPODoc = NewAEEventHandlerUPP(MyHandleODoc);
		gAEEventHandlerUPPOApp = NewAEEventHandlerUPP(MyHandleOApp);
		gAEEventHandlerUPPPDoc = NewAEEventHandlerUPP(MyHandlePDoc);
		err = AEInstallEventHandler (kCoreEventClass, kAEOpenDocuments, gAEEventHandlerUPPODoc,0, 0);
		err = AEInstallEventHandler (kCoreEventClass, kAEOpenApplication, gAEEventHandlerUPPOApp,0, 0);
		err = AEInstallEventHandler (kCoreEventClass, kAEPrintDocuments, gAEEventHandlerUPPPDoc,0, 0);
	}
}


void add_apropos(const char *progname)
{  
	static Fl_Menu_Item item = {"", 0, show_apropos, 0, 0};
	OSStatus err;
	MenuRef mr;
	const  char apple_glyph[2] = "\024";
	
	Fl_Sys_Menu_Bar *smb = new Fl_Sys_Menu_Bar(0,0,0,0);
	Fl_Menu_Item empty = {0, 0, NULL, 0, 0};
	smb->menu( &empty ); // indispensable pour initialiser fl_sys_menu_bar
	err = CreateNewMenu(1, 0, &mr);
	CFStringRef cftitle = CFStringCreateWithCString(NULL, apple_glyph, kCFStringEncodingMacRoman);
	err = SetMenuTitleWithCFString(mr, cftitle);
	CFRelease(cftitle);
	InsertMenu(mr, 0);
	CFStringRef format = CFStringCreateWithCString(NULL, "About %s", kCFStringEncodingMacRoman);
	cftitle = CFStringCreateWithFormat(NULL, NULL, format, progname);
	CFRelease(format);
	err = InsertMenuItemTextWithCFString(mr, cftitle, 0, 0,0);
	CFRelease(cftitle);
	err = SetMenuItemRefCon(mr, 1, (UInt32) (&item));
	item.user_data_ = NULL;
}


extern Fl_Sys_Menu_Bar *fl_sys_menu_bar;
void cre_sys_menu(void)
{
	static const Fl_Menu_Item items[] = {
		{"Window", 0, NULL, 0, FL_SUBMENU} ,
		{0},
		{0}
	};
	
	fl_sys_menu_bar->menu(items);
}

void windowmenuitem_callback(Fl_Widget *o, void *data)
{
	((Fl_Window *)data)->show();
	const char *c = ((Fl_Window *)data)->xclass();
	if(c == NULL) return;
	if(strcmp(c, SEAVIEW_WINDOW) == 0) {
		SEA_VIEW *view = (SEA_VIEW *)((Fl_Window *)data)->user_data();
		view->DNA_obj->take_focus();
		}
}

int find_windowmenuitem(Fl_Window *w)
{
	MenuRef ref = GetMenuRef(2);
	UInt32 p;
	Fl_Menu_Item *pitem;
	int i, count = CountMenuItems(ref);
	for(i = 1; i <= count; i++) {
		GetMenuItemRefCon(ref, i, &p);
		pitem = (Fl_Menu_Item *)p;
		if(pitem->user_data_ == w) return i;
	}
	return 0;
}

int add_windowmenuitem(const char *name, Fl_Window *w)
//returns rank of mew menu item or -1 if error
{
	MenuRef ref = GetMenuRef(2);
	MenuItemIndex rank;
	CFStringRef mycfs = CFStringCreateWithCString(NULL, (name == NULL ? "<empty>" : name), kCFStringEncodingUTF8);
	if(mycfs == NULL) return -1;
	AppendMenuItemTextWithCFString(ref, mycfs, 0, 0, &rank);
	CFRelease(mycfs);
	Fl_Menu_Item *item = (Fl_Menu_Item *)calloc(1, sizeof(Fl_Menu_Item));
	item->user_data_ = w;
	item->callback_ = windowmenuitem_callback;
	SetMenuItemRefCon(ref, rank, (UInt32)item);
	return rank;
}

void rename_windowmenuitem(const char *name, int rank)
{
	MenuRef ref = GetMenuRef(2);
	CFStringRef mycfs = CFStringCreateWithCString(NULL, name, kCFStringEncodingUTF8);
	SetMenuItemTextWithCFString(ref, rank, mycfs);
	CFRelease(mycfs);
}

void delete_windowmenuitem(int rank)
{
	UInt32 p;
	GetMenuItemRefCon(GetMenuRef(2), rank, &p);
	free((void *)p);
	DeleteMenuItem( GetMenuRef(2) , rank);
}


void show_apropos(Fl_Widget *w, void *unused)
{
	static Fl_Window *about = NULL;
	char line[100], *p;
	FILE *data;
	static char text[5000];
	
	if(about == NULL) {
		about = new Fl_Window(600, 450, "About seaview");
		Fl_Help_View *br = new Fl_Help_View(1, 1, about->w() - 2, about->h() - 2);
		Fl_Button *b = new Fl_Button(1, 1, 1, 1, NULL);//only to allow the shortcut
		b->callback(hide_window_callback, NULL);
		b->shortcut(myFL_CTRL | 'w');
		about->end();
		p = get_full_path(get_res_value("helpfile", "seaview.html"));
		if(p == NULL) return;
		data = fopen(p, "r");
		if(data == NULL) return;
		p = text;
		int doit = 0;
		while(TRUE) {
			fgets(line, sizeof(line), data);
			if(strncmp(line, "<a name=", 8) == 0) doit = 1;
			if(strncmp(line, "<hr>", 4) == 0) break;
			if(doit) {
				strcpy(p, line);
				p += strlen(p);
				}
		}
		fclose(data);
		br->value(text);
		about->resizable(br);
	}
	about->show();
}



/***********************************************************************/
Boolean AppleEventsInstalled ()
{
	OSErr err;
	long  result;
	
	err = Gestalt (gestaltAppleEventsAttr, &result);
	return (!err && ((result >> gestaltAppleEventsPresent) & 0x0001));
	// return TRUE if there is no
	// error and the proper bit of
	// result is set
}

/***********************************************************************/
OSErr  MyHandleODoc (const AppleEvent *theAppleEvent, AppleEvent* reply, long handlerRefCon)
{
	FSRef myFSRef;	
	AEDescList	docList;
	OSErr	err;
	long	itemsInList, rank;
	char masename[300];
	SEA_VIEW *view;
	
	err = AEGetParamDesc (theAppleEvent, keyDirectObject, typeAEList,
						  &docList);
	if (err) return err;
	
	err = MyGotRequiredParams (theAppleEvent);
	if (err) return err;
	
	err = AECountItems (&docList, &itemsInList);
	for(rank = 1; rank <= itemsInList; rank++) {
		err = AEGetNthPtr (&docList, rank, typeFSRef, NULL, NULL, &myFSRef, sizeof(FSRef), NULL);
		if (err) continue;
		err = FSRefMakePath(&myFSRef, (UInt8 *)masename, sizeof(masename));
		if (err) continue;
		Fl_Window *w = Fl::first_window();
		while(w != NULL) {
			const char *c = w->xclass();
			if(c != NULL && strcmp(c, SEAVIEW_WINDOW) == 0) break;
			w = Fl::next_window(w);
		}
		view = (w != NULL ? (SEA_VIEW *)w->user_data() : NULL);
		use_initial_file(view, masename, false);
		}
	err = AEDisposeDesc (&docList);
	return noErr;
}

/***********************************************************************/
OSErr  MyHandlePDoc (const AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	AEDescList	docList;
	OSErr	err;
	long	itemsInList;
	
	// get the direct parameter--a descriptor list--and put it into a docList
	err = AEGetParamDesc (theAppleEvent, keyDirectObject, typeAEList,
						  &docList);
	if (err)
		return err;
	
	// check for missing parameters
	err = MyGotRequiredParams (theAppleEvent);
	if (err)
		return err;
	
	// count the number of descriptor records in the list
	err = AECountItems (&docList, &itemsInList);
	
	// now get each descriptor record from the list, coerce the returned
	// data to an FSSpec record, and open the associated file
	
	return noErr;
	
}

/***********************************************************************/
OSErr  MyHandleOApp (const AppleEvent *theAppleEvent, AppleEvent *reply, long handlerRefCon)
{
	// Ouverture de l'appli
	return noErr;
}


/***********************************************************************/
OSErr MyGotRequiredParams (const AppleEvent *theAppleEvent)
{
	DescType	returnedType;
	Size	actualSize;
	OSErr	err;
	
	err = AEGetAttributePtr (theAppleEvent, keyMissedKeywordAttr,
							 typeWildCard, &returnedType, nil, 0,
							 &actualSize);
	if (err == errAEDescNotFound)	// you got all the required parameters
		return noErr;
	else if (!err)				// you missed a required parameter
		return errAEEventNotHandled;
	else						// the call to AEGetAttributePtr failed
		return err;
}


void MG_apple_inits(void)
{
	Fl::set_font(FL_COURIER,"Courier");
	Fl::set_font(FL_COURIER_BOLD,"Courier Bold");
	Fl::set_font(FL_COURIER_ITALIC,"Courier Oblique");
	Fl::set_font(FL_COURIER_BOLD_ITALIC,"Courier Bold Oblique");
}


void set_seaview_modified(SEA_VIEW *view, int ismodified)
{
	view->modif_but_not_saved = ismodified;
	SetWindowModified(fl_xid(view->dnawin), (Boolean)ismodified);
}

void MG_fl_draw(const char *txt, int x, int y)
{
	if(txt == NULL) return;
	if(!MG_doing_copy){
		fl_draw(txt, x, y);
		return;
	}
	unsigned char copy[strlen(txt) + 1];
	CtoP(txt, copy);
	MoveTo(x, y);
	DrawString(copy);
}

void MG_fl_line(int x, int y, int x1, int y1)
{
	if(!MG_doing_copy){
		fl_line(x, y, x1, y1);
		return;
	}
	MoveTo(x, y);
	LineTo(x1, y1);
}

void MG_fl_rect(int x, int y, int w, int h)
{
	if(!MG_doing_copy){
		fl_rect(x, y, w, h);
		return;
	}
	MoveTo(x, y);
	LineTo(x, y+h-1);
	LineTo(x+w-1, y+h-1);
	LineTo(x+w-1, y);
	LineTo(x, y);
}

void MG_fl_rectf(int x, int y, int w, int h)
{
	if(!MG_doing_copy){
		fl_rectf(x, y, w, h);
		return;
	}
	Rect r;
	r.top = (short)y;
	r.left = (short)x;
	r.bottom = (short)(y+h);
	r.right = (short)(x+w);
	PaintRect(&r);
}

void MG_fl_font(int fontrank, int size)
{
	if(!MG_doing_copy){
		fl_font(fontrank, size);
		return;
	}
	Style mystyle = normal;
	int macfont, fontstyle;
	if(fontrank < FL_SYMBOL) {
		fontstyle = fontrank % 4;
		fontrank -= fontstyle;
	}
	if(fontrank == FL_COURIER)
		macfont=kFontIDCourier/*22*/;
	else if(fontrank == FL_HELVETICA)
		macfont=kFontIDHelvetica/*21*/;
	else if(fontrank == FL_TIMES)
		macfont=kFontIDTimes/*20*/;
	else if(fontrank == FL_SYMBOL)
		macfont=kFontIDSymbol/*23*/;
	else
		macfont=21; 
	if(fontrank < FL_SYMBOL) {
		if(fontstyle == 0) mystyle = normal;
		else if(fontstyle == 1) mystyle = bold;
		else if(fontstyle == 2) mystyle = italic;
		else if(fontstyle == 3) mystyle = bold+italic;
	}
	TextFont(macfont);
	TextSize(size);
    TextFace(mystyle);
}

void MG_fl_color(Fl_Color color)
{
	if(!MG_doing_copy){
		fl_color(color);
		return;
	}
	unsigned char r, g, b;
	Fl::get_color(color, r, g, b);
	RGBColor maccolor;
	maccolor.red = r << 8;
	maccolor.green = g << 8;
	maccolor.blue = b << 8;
	RGBForeColor(&maccolor);
}

float MG_fl_width(char *txt, int l)
{
	if(!MG_doing_copy){
		return fl_width(txt, l);
	}
	return (float)TextWidth( txt, 0, l );
}

float MG_fl_width(char *txt)
{
	return MG_fl_width(txt, strlen(txt));
}

void *MG_PrepareCopy(int w, int h)
{
	Rect myrect;
	myrect.top=0;
	myrect.left=0;
	myrect.right=w;
	myrect.bottom=h;
#ifdef __APPLE_QUARTZ__
	Fl_X::q_release_context();
#endif
	PicHandle mypicture = OpenPicture(&myrect);
	ClipRect(&myrect);
	PenNormal();
	MG_doing_copy = 1;
	return mypicture;
}


void MG_CompleteCopy(void *mypicture)
{
	ClosePicture();
	MG_doing_copy = 0;
	MyCopyPictToClipboard((PicHandle)mypicture);
	KillPicture((PicHandle)mypicture);
}


static size_t MyPutBytes (void* info, const void* buffer, size_t count)
{
    CFDataAppendBytes ((CFMutableDataRef) info, (const UInt8 *)buffer, count);
    return count;
}

QDPictRef MyPictToQDPict(PicHandle mypicture)
{
	CFIndex pictlen;
	CGDataProviderRef provider;
	/* conversion picture to QDPict */
	pictlen = GetHandleSize( (Handle)  mypicture);
	HLock( (Handle) mypicture);
	provider = CGDataProviderCreateWithData (NULL, *mypicture, pictlen, NULL);
	HUnlock( (Handle) mypicture);
    if (provider != NULL) {
        QDPictRef myqdpict = QDPictCreateWithProvider (provider);
		CFRelease(provider);
		return myqdpict;
	}
	else return NULL;
}

void MyCopyPictToClipboard (PicHandle mypicture)
{
    static CGDataConsumerCallbacks callbacks = { MyPutBytes, NULL };
	CFDataRef  data = NULL;
	PasteboardRef clipboard = NULL;
	CFIndex pictlen;
	QDPictRef myqdpict;	
	
	pictlen = GetHandleSize( (Handle)  mypicture);
	PasteboardCreate (kPasteboardClipboard, &clipboard);
	PasteboardClear(clipboard);
	HLock( (Handle) mypicture);
	data = CFDataCreate( kCFAllocatorDefault, (UInt8*)*mypicture, pictlen );
	HUnlock( (Handle) mypicture);
	PasteboardPutItemFlavor(clipboard, (PasteboardItemID)1, 
							/* kUTTypePICT */ CFSTR("com.apple.pict"),
							data,
							kPasteboardFlavorNoFlags );
    CFRelease (data);    
	
	myqdpict = MyPictToQDPict(mypicture);
	data = CFDataCreateMutable (kCFAllocatorDefault, 0);
    if (data != NULL)
    {
        CGDataConsumerRef consumer = NULL;
        consumer = CGDataConsumerCreate ((void*) data, &callbacks);
        if (consumer != NULL) 
        {
            CGContextRef context = NULL;
            CGRect bounds = QDPictGetBounds (myqdpict);
            bounds.origin.x = 0;
            bounds.origin.y = 0;
            context = CGPDFContextCreate (consumer, &bounds, NULL);
            CGDataConsumerRelease (consumer);
            if (context != NULL) 
            {
                /* convert PICT to PDF */
                CGContextBeginPage (context, &bounds);
                (void) QDPictDrawToCGContext (context, bounds, myqdpict);
                CGContextEndPage (context);
                CGContextRelease (context);
				
                /* copy PDF to clipboard */
                (void) PasteboardPutItemFlavor (clipboard, 
												(PasteboardItemID)1, 
												/* kUTTypePDF */ CFSTR("com.adobe.pdf"), 
												data, kPasteboardFlavorNoFlags);
            }
        }
		CFRelease (data); 
    }
	CFRelease (clipboard);
	if(myqdpict != NULL) QDPictRelease(myqdpict);
}


extern paperformat printout_pageformat;
static PMPrintSession  printSession;
static PMPageFormat    pageFormat = kPMNoPageFormat;
static PMPrintSettings printSettings = kPMNoPrintSettings;
int MG_StartPrinting (int pagecount, int *frompage, int *topage)
//printing using a Quartz graphics context
//returns 0 iff OK
{
	OSStatus status;
	Boolean accepted;
	
	Fl_X::q_release_context();
	status = PMCreateSession(&printSession);
	if (status != noErr) return 1;
/*	failed attempt to set papersize from GUI
	PMPaper mypaper;
	CFIndex num, count;
	if(PMPrinterGetPaperList == NULL) {//if running before 10.3
		num = count = 0;
		}
	else {
		//set mypaper to value in GUI
		PMPrinter myprinter;
		status = PMSessionGetCurrentPrinter(printSession, &myprinter);
		if (status != noErr) return 1;
		CFArrayRef mypaperlist;
		status = PMPrinterGetPaperList(myprinter, &mypaperlist); //10.3
		if (status != noErr) return 1;
		count = CFArrayGetCount(mypaperlist);
		CFStringRef desiredpaper = printout_pageformat == A4 ? CFSTR("A4") : CFSTR("Letter");
		for(num = 0; num < count; num++) {
			mypaper = (PMPaper)CFArrayGetValueAtIndex(mypaperlist, num);
			CFStringRef papername;
			PMPaperGetName(mypaper, &papername); //10.3
			if( CFStringCompare(papername, desiredpaper, kCFCompareCaseInsensitive) == kCFCompareEqualTo) break;
			}
		}
	if(num < count) {//desired papersize was found
		status = PMCreatePageFormatWithPMPaper(&pageFormat, mypaper); //10.3
		if (status != noErr || pageFormat == kPMNoPageFormat) return 1;
		}
	else {//desired papersize was not found
		status = PMCreatePageFormat(&pageFormat);
		status = PMSessionDefaultPageFormat(printSession, pageFormat);
		if (status != noErr) return 1;
		status = PMSessionPageSetupDialog(printSession, pageFormat, &accepted);
		if (status != noErr || !accepted) return 1;
		} */
	status = PMCreatePageFormat(&pageFormat);
	status = PMSessionDefaultPageFormat(printSession, pageFormat);
	if (status != noErr) return 1;
	status = PMSessionPageSetupDialog(printSession, pageFormat, &accepted);
	if (status != noErr || !accepted) return 1;
	status = PMCreatePrintSettings(&printSettings);
	if (status != noErr || printSettings == kPMNoPrintSettings) return 1;
	status = PMSessionDefaultPrintSettings (printSession, printSettings);
	if (status != noErr) return 1;
	PMSetPageRange(printSettings, 1, (UInt32)kPMPrintAllPages);
	status = PMSessionPrintDialog(printSession, printSettings, pageFormat, &accepted);
	if (!accepted) status = kPMCancel;
	if (status != noErr) return 1;
	UInt32 from32, to32;
	PMGetFirstPage(printSettings, &from32); *frompage = (int)from32;
	PMGetLastPage(printSettings, &to32); *topage = (int)to32;
	if(*topage > pagecount) *topage = pagecount;
#if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4
	status = PMSessionBeginCGDocument(printSession, printSettings, pageFormat);
#else
	CFStringRef mystring[1];
	mystring[0] = kPMGraphicsContextCoreGraphics;
	CFArrayRef array = CFArrayCreate(NULL, (const void **)mystring, 1, &kCFTypeArrayCallBacks);
	status = PMSessionSetDocumentFormatGeneration(printSession, kPMDocumentFormatDefault, array, NULL);
	CFRelease(array);
	status = PMSessionBeginDocument(printSession, printSettings, pageFormat);
#endif
	if (status != noErr) return 1;
	return 0;
}


int MG_PrintingRect(int *x, int *y, int *w, int *h)
//returns TRUE iff OK
{
	OSStatus status;
	PMRect pmRect;
	
	status = PMGetAdjustedPageRect(pageFormat, &pmRect);
	if (status != noErr) return FALSE;
	
	*x = (int)pmRect.left;
	*y = (int)pmRect.top;
	*w = (int)pmRect.right - *x + 1;
	*h = (int)pmRect.bottom - *y + 1;
	return TRUE;
}

extern CGContextRef fl_gc;
extern void fl_quartz_restore_line_style_();
int MG_StartPage (void)
{	
	Fl_X::q_release_context();
	OSStatus status = PMSessionBeginPage(printSession, pageFormat, NULL);
#if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4
	status = PMSessionGetCGGraphicsContext(printSession, &fl_gc);
#else
	status = PMSessionGetGraphicsContext(printSession,NULL,(void **)&fl_gc);
#endif
	PMRect pmRect;
	status = PMGetAdjustedPageRect(pageFormat, &pmRect);
	double h = pmRect.bottom - pmRect.top;
	CGContextTranslateCTM(fl_gc, 20, 20 + h -0.5f);
	CGContextScaleCTM(fl_gc, 1.0f, -1.0f);
	fl_quartz_restore_line_style_();
	CGContextSaveGState(fl_gc);
	return status == noErr;
}

int MG_ClipPrintingRect(int x, int y, int w, int h)
{
	Rect r;
	
	r.top = y - 1; r.left = x - 1; r.bottom = y + h + 1; r.right = x + w + 1;
	CGRect cgr;
	cgr.origin.x = r.left;
	cgr.origin.y = r.top;
	cgr.size.width = r.right - r.left;
	cgr.size.height = r.bottom - r.top;
	CGContextRestoreGState(fl_gc);
	CGContextSaveGState(fl_gc);
	CGContextClipToRect(fl_gc, cgr); 
	return TRUE;
}

int MG_EndPage (void)
{	
	OSStatus status = PMSessionEndPage(printSession);
	CGContextFlush(fl_gc);
	CGContextRestoreGState(fl_gc);
	return status == noErr;
}

void MG_EndPrinting (void)
{
	OSStatus status;
	
		status = PMSessionError(printSession);
		if (status != noErr) {
			fl_alert ("PM Session error %d", status);
		}
		PMSessionEndDocument (printSession);
	
	if (pageFormat != kPMNoPageFormat) {
		PMRelease(pageFormat);
		pageFormat = kPMNoPageFormat;
	}
	if (printSettings != kPMNoPrintSettings) {
		PMRelease(printSettings);
		printSettings = kPMNoPrintSettings;
	}
	PMRelease(printSession);
	fl_gc = 0;
}


//next 2 functions to copy quartz data to clipboard both as pdf and as PICT bitmap
#if MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_4
#include <QuickTime/QuickTimeComponents.h>
#endif
void *prepare_copy_pdf_and_pict(int w, int h)
{
	Fl_X::q_release_context();
	//prepare to draw in pdf context
	CGRect bounds = CGRectMake(0, 0, w, h );	
	CFMutableDataRef pdfdata = CFDataCreateMutable(NULL, 0);
	CGDataConsumerRef myconsumer;
#if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4
	myconsumer = CGDataConsumerCreateWithCFData (pdfdata);
#else
    static CGDataConsumerCallbacks callbacks = { MyPutBytes, NULL };
	myconsumer = CGDataConsumerCreate ((void*) pdfdata, &callbacks);
#endif
	fl_gc = CGPDFContextCreate (myconsumer, &bounds, NULL);
	CGDataConsumerRelease (myconsumer);
	if (fl_gc == NULL) return NULL;
	CGContextBeginPage (fl_gc, NULL);
	CGContextTranslateCTM(fl_gc, 0, h);
	CGContextScaleCTM(fl_gc, 1.0f, -1.0f);
	return (void *)pdfdata;
}


void complete_copy_pdf_and_pict(void *data)
{
	int w, h;
	CFMutableDataRef pdfdata = (CFMutableDataRef)data;
	CGRect clip = CGContextGetClipBoundingBox(fl_gc);
	w = (int)(clip.size.width + 0.5);
	h = (int)(clip.size.height + 0.5);
	CGContextEndPage (fl_gc);
	CGContextRelease (fl_gc);
	PasteboardRef clipboard = NULL;
	PasteboardCreate (kPasteboardClipboard, &clipboard);
	PasteboardClear(clipboard); //	copy PDF to clipboard 
	PasteboardPutItemFlavor (clipboard, (PasteboardItemID)1, 
							 CFSTR("com.adobe.pdf"), // kUTTypePDF
							 pdfdata, kPasteboardFlavorNoFlags);
	
	
	//second, transform this PDF to a bitmap image and put it as PICT in clipboard
#if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4
	CGDataProviderRef prov = CGDataProviderCreateWithCFData(pdfdata);
#else
	CGDataProviderRef prov = CGDataProviderCreateWithData(NULL, CFDataGetBytePtr(pdfdata), CFDataGetLength(pdfdata), NULL);
#endif
	CGPDFDocumentRef pdfdoc = CGPDFDocumentCreateWithProvider(prov);
	CGDataProviderRelease(prov);
	CGColorSpaceRef space = CGColorSpaceCreateWithName(/*kCGColorSpaceGenericRGB*/ kCGColorSpaceUserRGB);
	const int scale = 2;
	w *= scale; h *= scale;
	void *mem = malloc(w * h * sizeof(int));
	fl_gc = CGBitmapContextCreate(mem, w, h, 8, w * 4, space, kCGImageAlphaPremultipliedFirst /*| kCGBitmapByteOrder32Host*/);
	CFRelease(space);
	if (fl_gc == NULL) { free(mem); return; }
	CGRect rect = CGRectMake(0, 0, w, h);
	CGContextSetRGBFillColor(fl_gc,  1,1,1,1);//need to clear background
	CGContextFillRect(fl_gc, rect);
	CGContextDrawPDFDocument(fl_gc, rect, pdfdoc, 1);
	CGPDFDocumentRelease(pdfdoc);
	CFRelease(pdfdata);
#if MAC_OS_X_VERSION_MAX_ALLOWED >= MAC_OS_X_VERSION_10_4
	//other way for >= 10.4 that doesn't use QuickTime:
	CGImageRef image = CGBitmapContextCreateImage(fl_gc);
	CFMutableDataRef pictdata = CFDataCreateMutable(NULL, 0);
	CGImageDestinationRef dest = CGImageDestinationCreateWithData(pictdata, CFSTR("com.apple.pict"), 1, NULL);
	CGImageDestinationAddImage(dest, image, NULL);
	CGImageDestinationFinalize(dest);
	CGImageRelease(image);
	CFRelease(dest);
	CFRange range = CFRangeMake(0, 512);
	CFDataDeleteBytes(pictdata, range);// The clipboard REQUIRES only the image data, not the header.
	PasteboardPutItemFlavor(clipboard, (PasteboardItemID)1, CFSTR("com.apple.pict"), pictdata,
							kPasteboardFlavorNoFlags ); 
	CFRelease (pictdata);    
#else
	GraphicsExportComponent exporter;
	OSErr err = OpenADefaultComponent(GraphicsExporterComponentType, kQTFileTypePicture /*'PICT'*/, &exporter);
	err = GraphicsExportSetInputCGBitmapContext(exporter, fl_gc);
	Handle dataHandle = NewHandle(0);
	err = GraphicsExportSetOutputHandle(exporter, dataHandle);
	unsigned long size;
	err = GraphicsExportDoExport(exporter, &size);
	err = CloseComponent(exporter); 
	if(GetHandleSize(dataHandle) > 512) {
		HLock(dataHandle);
		// The clipboard REQUIRES only the image data, not the header.
		CFDataRef pictdata = CFDataCreate(NULL, (const UInt8 *)*dataHandle + 512, GetHandleSize(dataHandle) - 512);
		HUnlock(dataHandle);
		PasteboardPutItemFlavor(clipboard, (PasteboardItemID)1, 
								CFSTR("com.apple.pict"),
								pictdata,
								kPasteboardFlavorNoFlags ); 
		CFRelease (pictdata);    
	}
	DisposeHandle(dataHandle);
#endif
    CFRelease (clipboard);    
	CGContextRelease(fl_gc);
	fl_gc = NULL;
	free(mem);
}


void draw_rotated_string(const char *txt, int x, int y, float degrees /*clockwise*/)
{
	OSStatus err;
	unsigned n = strlen(txt);
    // convert to UTF-16
	unsigned short *uniStr = new unsigned short[3 * n];
	n = fl_utf8toUtf16(txt, n, uniStr, 3 * n);
	
	ATSUTextLayout layout = fl_fontsize->layout;
	Fixed fangle = FloatToFixed(degrees);
	ByteCount iSize[] = {sizeof(Fixed), sizeof(CGContextRef)};
	ATSUAttributeTag iTag[] = {kATSULineRotationTag, kATSUCGContextTag};
	ATSUAttributeValuePtr aAttr[] = { &fangle,  &fl_gc};
	err = ATSUSetLayoutControls (layout, 2, iTag, iSize, aAttr );
	err = ATSUSetTextPointerLocation(layout, uniStr, kATSUFromTextBeginning, n, n);
	err = ATSUDrawText(layout, kATSUFromTextBeginning, n, FloatToFixed((float)x), FloatToFixed((float)y));
	delete uniStr;
	fangle = FloatToFixed(0); // reset angle
	err = ATSUSetLayoutControls (layout, 1, iTag, iSize, aAttr );
}

#endif //__APPLE__
