/*
 * robrating.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// robrating.cpp: Implementierung der Klasse RatingPlugin.
//
//////////////////////////////////////////////////////////////////////

#include "robrating.h"
#include "robtrans.h"


using namespace lrt;

namespace rt {

FpInterpretePos RatingPlugin::getInterpretePos()
{
	return fpInterPosBefore;
}

String RatingPlugin::getCodeName()
{
	return "RatingPlugin";
}

String RatingPlugin::getName()
{
	return _("Robot Rating Agency");
}

String RatingPlugin::getHelpText()
{
	return _("  -rate N    Produce a rating for the first bot on cycle N and end \n             the game. If N < 0, wait for timeout. ");
}

bool RatingPlugin::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	for(int i = 0; i < params.length(); i++)
		if(params[i].lowerCase() == "-rate") // Do rating.
		{
			active = true;
			used[i] = true;
			// get our cycle parameter
			if(used[i+1]) 
				return false;
			endCycles = params[i+1].intValue(-1);
			used[i+1] = true;
		}
	return true;
}

void RatingPlugin::fillOptions(SimOptions& options)
{
	if(active) 
		options.supervisors += new RatingSimSupervisor(endCycles);
}

RatingPlugin::RatingPlugin(Frontend* parent) : FrontendPlugin(parent), endCycles(-1)
{
}

///////////////////////// RatingSimSupervisor

RatingSimSupervisor::RatingSimSupervisor(int endCycles) : endCycles(endCycles)
{
	retState.affectedProgram = 0;
	retState.event = gameNothing;
}

SimSuperPosition RatingSimSupervisor::getPreferredPosition()
{
	return supPosDontCare;
}

unsigned long RatingSimSupervisor::getStepCycles()
{
	return 100;
}

SimSupervisor::GameState RatingSimSupervisor::exec(Simulation* const curSim)
{
	if(endCycles < 0)
		endCycles = curSim->glob.timeout;

	if(curSim->curCycle >= (unsigned long) endCycles)
	{
		retState.event = gameTie;
		return retState;
	}
	else 
	{
		retState.event = gameNothing;
		return retState;
	}
}

void RatingSimSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	double rating = 0; 

	if(curSim->getNumPrograms() < 1) 
		return; 

	if(curSim->getNumPrograms() == 1) { // single-player rating based on bot count
		if(curSim->curCycle >= (unsigned long) endCycles) { // survival to finish => positive rating
			rating = ((double) curSim->getProgram(0)->vars[progMybots]) /  // field fill rate
						  (curSim->fieldHeight * curSim->fieldWidth);
		}
		else { // died before rating time => negative rating
			rating = (((double) curSim->curCycle) / endCycles / 2.) - 1.0; // survival time rate
		}
	}
	else { // multi-player game rating based on winner & bot count
		if(simResult.event == gameFinished) { // there is a winner
			rating = 1. - (((double) curSim->curCycle) / endCycles * 0.66); // range (0.33 .. 1.0)
			if(simResult.affectedProgram != 0) // oops, it's not us
				rating = -rating; // negative score for the loser
		}
		else { // timeout or such
			double mybots = curSim->getProgram(0)->vars[progMybots];
			double otherbots = curSim->getProgram(0)->vars[progOtherbots];
			double maxbots = (curSim->fieldHeight * curSim->fieldWidth);
			rating = mybots / ((otherbots + maxbots) / 2); // this value has range (0..2)
			rating = rating * 0.33 - 0.33; // transform to (-0.33 .. 0.33)
		}
	}

	System::println(String("X-Rating: ") + String(rating));
}



} // namespace

