/*
 * robpackage.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements search path-related functions (looking for data files)
// that are most required when being installed as a RPM (or similar) package

// For Linux packaging purposes, besides looking in the executable folder, 
// or the subfolders /sounds, /help, /optsets, 
// we also check for
// EXTRA_DATA_DIR (/sounds, /help, /optsets)
// and
// EXTRA_HELP_DIR (without subfolder) 

#include "robpackage.h"
#include "robtrans.h"

#include <rtsound.h>
#include <rtfile.h>
#include <rtresource.h>

using namespace lrt; 

namespace rt {

// Should be called at startup, after lrt was initialized!
void initPackage() 
{
#ifdef EXTRA_DATA_DIR
	String extraDataDir = EXTRA_DATA_DIR; 
	if(!extraDataDir.endsWith(File::separator))
		extraDataDir += File::separator; 

	Sound::addSoundFolder(extraDataDir + "sounds/"); 

#endif
}

// Search for something everywhere
// xfolder: subfolder where to look
lrt::String findSomething(const lrt::String& xfolder, const lrt::String& file) 
{
	File execfolder = File::getExecutableFile().getParentFile();

	if(File(file).exists()) return file;

	if(File(execfolder, file).exists()) 
		return File(execfolder, file).getName();
	if(File(execfolder, xfolder + file).exists()) 
		return File(execfolder, xfolder + file).getName();


#ifdef EXTRA_DATA_DIR
	String extraDataDir = EXTRA_DATA_DIR; 
	if(!extraDataDir.endsWith(File::separator))
		extraDataDir += File::separator; 

	if(File(extraDataDir + xfolder + file).exists())
		return File(extraDataDir + xfolder + file).getName(); 
#endif

	return String(); 
}

// Search for an option set
lrt::String findOptionSet(const lrt::String& file)
{
	lrt::InputStream* is = lrt::Resource::getResource(file); 
	if(is) {
		delete is; 
		return file; // just the resource name
	}

	return findSomething("optsets/", file); 
}

// The sound finding is done by lrt::Sound itself. 
//lrt::String findSound(const lrt::String& baseName);

lrt::String findTranslatedFile(const lrt::String& xfolder, const lrt::String& file)
{
	String lcode = getLanguageCode(); 
	if(lcode.indexOf('_') >= 0) { // fully specified language
		// search for it there
		String filen = findSomething(xfolder + lcode + "/", file);
		if(filen.length())
			return filen; 

		// now shorten to short form of the language code (without country) 
		lcode = lcode.substring(0, lcode.indexOf('_')); 
	}
	if(lcode.length()) { // some language was specified
		String filen = findSomething(xfolder + lcode + "/", file); 
		if(filen.length())
			return filen; 
	}
	// search for the default also 
	return findSomething(xfolder, file); 
}

lrt::String findHelpFile(const lrt::String& file)
{
	String filen = findTranslatedFile("help/", file); 
#ifdef EXTRA_HELP_DIR
	String extraHelpDir = EXTRA_HELP_DIR; 
	if(!extraHelpDir.endsWith(File::separator))
		extraHelpDir += File::separator; 

	if(!filen.length()) {
		filen = findTranslatedFile(extraHelpDir, file); 
	}
#endif
	return filen; 
}



} // namespace
