/*
 * robinstr.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

// Implements all the instructions 

#include "robvars.h"
#include "robglob.h"
#include "robbase.h"
#include "robinstr.h"

#include <rtstring.h>
#include <rtcollect.h>
#include <rtsystem.h>
#include <rtmath.h>

using namespace lrt; 

namespace rt {

///////////////// derived instructions //////////////////

/*******/
/* DIE */
/*******/

ExecReturnType InstrDie::exec(Task*)
{
	return failDie;
}

String InstrDie::getName()
{
	return String("Die");
}

InstrDie::InstrDie(const Globals& glob) : Instr(0)
{
	cycles = glob.durDie;
}

InstrDie::~InstrDie(){};

/********/
/* MOVE */
/********/

rint InstrMove::getNumCycles(Task* task)
{
	// Check mobility
	if(task->bot->vars[botMobile] == 0)
	{
		return failMobile;
	}
	return cycles;
}

ExecReturnType InstrMove::exec(Task* task)
{
	Bot* bot = task->bot;
	Bot* refBot = task->getRefBot();
	if(refBot == 0)
	{
		rint x,y;
		task->getRefBotPos(x, y);

		bot->owner->sim->field[bot->vars[botPosX] +
				   bot->vars[botPosY] * bot->owner->sim->fieldWidth] = 0;

		bot->owner->sim->field[x +
				   y * bot->owner->sim->fieldWidth] = bot;

		bot->vars[botPosX] = x;
		bot->vars[botPosY] = y;
	}
	return execOK;
}

String InstrMove::getName()
{
	return String("Move");
}

InstrMove::InstrMove(const Globals& glob) : Instr(0)
{
	cycles = glob.durMove;
}

InstrMove::~InstrMove(){};

/********/
/* TURN */
/********/

ExecReturnType InstrTurn::exec(Task* task)
{
	rint op1;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.

	if(op1 == 0)
		task -> vars[taskDir] = (task -> vars[taskDir] + turnLeft) % 4;
	else
		task -> vars[taskDir] = (task -> vars[taskDir] + turnRight) % 4;
	return execOK;
}

String InstrTurn::getName()
{
  return String("Turn");
}

InstrTurn::InstrTurn(const Globals& glob, const Array<Op>& ops) : Instr(1)
{
  cycles = glob.durTurn;
  if(ops[0].isRemote) cycles += glob.durRemote;

  initOps(ops);
}
InstrTurn::~InstrTurn(){};

/********/
/* SCAN */
/********/

rint InstrScan::getNumCycles(Task* task)
{
	// Check Instruction Set
	if(task->bot->vars[botInstrSet] < 1)
	{
	  return failInstrSet;
	}
	return cycles;
}

ExecReturnType InstrScan::exec(Task* task)
{
	rint res;

	Bot* bot = task->bot;
	Bot* refBot = task->getRefBot();
	if(refBot == 0)
		res = 0;
	else
	{
		if(bot->owner == refBot->owner)
			res = 2;
		else
			res = 1;
	}

	setValue(task, 0, res);

	return execOK;
}

String InstrScan::getName()
{
  return String("Scan");
}

InstrScan::InstrScan(const Globals& glob, const Array<Op>& ops) : Instr(1)
{
  cycles = glob.durScan;
  if(ops[0].isRemote) cycles += glob.durRemote;

  initOps(ops);
}
InstrScan::~InstrScan(){};


/***********/
/* FarScan */
/***********/

rint InstrFarscan::getNumCycles(Task* task)
{
	// Check Instruction Set
	if(task->bot->vars[botInstrSet] < 1)
	{
	  return failInstrSet;
	}
	rint maxDist = getValue(task, 2);
	if(maxDist < 0) // check param maxDist
	{
		return failParam;
	}

	int addCycles = maxDist * task->sim->glob.durFarscanPerField;

	return (rint)(Math::min(cycles + addCycles, MAX_RINT));
}

ExecReturnType InstrFarscan::exec(Task* task)
{
	rint maxDist = getValue(task, 2);
	if(maxDist < 0)
		return failParam;

	rint type = 0;
	rint dist = 0;

	rint x, y, dir;
	Bot* bot = task->bot;
	x = bot->vars[botPosX];
	y = bot->vars[botPosY];
	dir = task->vars[taskDir];

	Bot* refBot = 0;
	for(dist = 1; dist <= maxDist; dist++)
	{
		task->sim->getPosBefore(x, y, dir);
		refBot = task->sim->field[ x + y * task->sim->fieldWidth ];
		if(refBot != 0) break;
	}
	// now, refBot is the closest bot (or 0 if none in range)
	if(refBot == 0) 
	{
		dist = 0;
	}
	else 
	{
		if(bot->owner == refBot->owner)
			type = 2;
		else
			type = 1;
	}

	setValue(task, 0, type);
	setValue(task, 1, dist);

	return execOK;
}

String InstrFarscan::getName()
{
  return String("FarScan");
}

InstrFarscan::InstrFarscan(const Globals& glob, const Array<Op>& ops) : Instr(3)
{
  cycles = glob.durFarscan;
  if(ops[0].isRemote) cycles += glob.durRemote;
  if(ops[1].isRemote) cycles += glob.durRemote;
  if(ops[2].isRemote) cycles += glob.durRemote;

  initOps(ops);
}
InstrFarscan::~InstrFarscan(){};


/*******/
/* SET */
/*******/

ExecReturnType InstrSet::exec(Task* task)
{
	rint op = getValue(task, 1);
	setValue(task, 0, op);
	return execOK;
}

String InstrSet::getName()
{
	return String("Set");
}

InstrSet::InstrSet(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durSet;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}

InstrSet::~InstrSet(){};

/*******/
/* MIN */
/*******/

ExecReturnType InstrMin::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	rint v2 = op1 < op2 ? op1:op2;

	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrMin::getName()
{
	return String("Min");
}

InstrMin::InstrMin(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durMin;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrMin::~InstrMin(){};

/*******/
/* MAX */
/*******/

ExecReturnType InstrMax::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	rint v2 = op1 > op2 ? op1:op2;

	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrMax::getName()
{
	return String("Max");
}

InstrMax::InstrMax(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durMax;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrMax::~InstrMax(){};

/*******/
/* ADD */
/*******/

ExecReturnType InstrAdd::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	rint v1 = op1, v2 = op1 + op2;

	// Elimination trigger kontrolle
	if(task->sim->glob.enableElim) {
		bool i1 = (Math::abs(v1) > task->sim->glob.elim);
		bool i2 = (Math::abs(v2) > task->sim->glob.elim);

		if(i1 != i2)
			return failElim;
	}

	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrAdd::getName()
{
	return String("Add");
}

InstrAdd::InstrAdd(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durAdd;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrAdd::~InstrAdd(){};

/*******/
/* SUB */
/*******/

ExecReturnType InstrSub::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	rint v1 = op1, v2 = op1 - op2;

	// Elimination trigger kontrolle
	if(task->sim->glob.enableElim) {
		bool i1 = (Math::abs(v1) > task->sim->glob.elim);
		bool i2 = (Math::abs(v2) > task->sim->glob.elim);

		if(i1 != i2)
			return failElim;
	}

	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrSub::getName()
{
	return String("Sub");
}

InstrSub::InstrSub(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durSub;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrSub::~InstrSub(){};


/*******/
/* MUL */
/*******/

ExecReturnType InstrMul::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	rint v2 = op1 * op2;

	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrMul::getName()
{
  return String("Mul");
}

InstrMul::InstrMul(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durMul;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrMul::~InstrMul(){};

/*******/
/* DIV */
/*******/

ExecReturnType InstrDiv::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	// Division durch 0 abfangen
	if(op2 == 0)
	{
		return failDiv0;
	}

	rint v2 = op1 / op2;
	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrDiv::getName()
{
	return String("Div");
}

InstrDiv::InstrDiv(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durDiv;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrDiv::~InstrDiv(){};

/*******/
/* MOD */
/*******/

ExecReturnType InstrMod::exec(Task* task)
{
	rint op1, op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	// Avoid modulus of 0
	if(op2 == 0)
	{
		return failDiv0;
	}

	rint v2 = op1 % op2;
	setValue(task, 0, v2);    // Set the variable in the first parameter to the
	// result value
	return execOK;
}

String InstrMod::getName()
{
	return String("Mod");
}

InstrMod::InstrMod(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durMod;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrMod::~InstrMod(){};

/*********/
/* TRANS */
/*********/

rint InstrTrans::getNumCycles(Task* task)
{
	Bot* bot = task->bot;
	// Check Instruction Set
	if(bot -> vars[botInstrSet] < 1)
	{
		return failInstrSet;
	}

	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if((op1 < 1) || (op1 > bot->vars[botNumBanks]))
	{
		return failBank;
	}

	rint numInstr = bot->banks[op1 - 1]->instr.length();

	return cycles + numInstr * task->sim->glob.durTransPerInstr;
}

ExecReturnType InstrTrans::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.
	Bot* bot = task->bot;

	if((op1 < 1) || (op1 > bot->vars[botNumBanks]))
	{
		return failBank;
	}

	Bot* refBot = task->getRefBot();
	if(refBot != 0)
	{
		// Hat der Zielbototer die Zielbank?
		if((op2 >= 1) && (op2 <= refBot -> vars[botNumBanks]) )
		{
			// Bank kopieren
			refBot -> banks[op2 - 1] = bot -> banks[op1 - 1];
		}
	}
	return execOK;
}

String InstrTrans::getName()
{
	return String("Trans");
}

InstrTrans::InstrTrans(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durTrans;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrTrans::~InstrTrans(){};

/**********/
/* RTRANS */ 
/**********/

rint InstrRtrans::getNumCycles(Task* task)
{
	Bot* bot = task->bot;
	// Check Instruction Set
	if(bot -> vars[botInstrSet] < 1)
	{
		return failInstrSet;
	}

	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if((op1 < 1) || (op1 > task->sim->glob.maxBanks))
	{
		return failBank;
	}
	
	if((op2 < 1) || (op2 > bot->vars[botNumBanks]))
	{
		return failBank;
	}

	rint numInstr;
	Bot* refBot = task->getRefBot();
	if(!refBot) numInstr = 0;
	else if((op1 < 1) || (op1 > refBot->vars[botNumBanks])) numInstr = 0;
	else numInstr = refBot->banks[op1 - 1]->instr.length();

	return cycles + numInstr * task->sim->glob.durRtransPerInstr;
}

ExecReturnType InstrRtrans::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.
	Bot* bot = task->bot;

	if((op1 < 1) || (op1 > task->sim->glob.maxBanks))
	{
		return failBank;
	}
	
	if((op2 < 1) || (op2 > bot->vars[botNumBanks]))
	{
		return failBank;
	}

	Bot* refBot = task->getRefBot();
	if(refBot != 0)
	{
		// Has the reference bot got the bank?
		if((op1 >= 1) && (op1 <= refBot->vars[botNumBanks]) )
		{
			// Copy bank
			bot -> banks[op2 - 1] = refBot -> banks[op1 - 1];
		}
	}
	return execOK;
}


String InstrRtrans::getName()
{
	return String("RTrans ");
}

InstrRtrans::InstrRtrans(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durRtrans;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrRtrans::~InstrRtrans(){};

/*********/
/* AJUMP */
/*********/

ExecReturnType InstrAjump::exec(Task* task)
{
  rint op1;
  op1 = getValue(task, 0, true); // Interpretiert den Typ zusammen mit dem Param.

  task->vars[taskCurInstr] = op1 - 2; // "ajump 1" must result in -1

  return execOK;
}

String InstrAjump::getName()
{
	return String("AJump");
}

InstrAjump::InstrAjump(const Globals& glob, const Array<Op>& ops) : Instr(1)
{
	cycles = glob.durAjump;
	if(ops[0].isRemote) cycles += glob.durRemote;

	initOps(ops);
}
InstrAjump::~InstrAjump(){};

/********/
/* JUMP */
/********/

ExecReturnType InstrJump::exec(Task* task)
{
	rint op1;
	op1 = getValue(task, 0); // Interpretiert den Typ zusammen mit dem Param.

	task->vars[taskCurInstr] += op1 - 1; // "jump 1" darf nichts aendern!
	return execOK;
}

String InstrJump::getName()
{
	return String("Jump");
}

InstrJump::InstrJump(const Globals& glob, const Array<Op>& ops) : Instr(1)
{
	cycles = glob.durJump;
	if(ops[0].isRemote) cycles += glob.durRemote;

	initOps(ops);
}
InstrJump::~InstrJump(){};

/*********/
/* BJUMP */
/*********/

ExecReturnType InstrBjump::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // jump to op1'th bank
	op2 = getValue(task, 1, true);   // absolute jump to the op2'th instr in the op1'th bank

	if((op1 < 0) || (op1 > task->sim->glob.maxBanks))
	{
		return failBank;
	}

	task->vars[taskCurBank] = op1 - 1;
	task->vars[taskCurInstr] = op2 - 2;

	return execOK;
}

String InstrBjump::getName()
{
	return String("BJump");
}

InstrBjump::InstrBjump(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durBjump;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrBjump::~InstrBjump(){};

/********/
/* SLEEP */
/********/

TimingMode InstrSleep::getTimingMode()
{
	return timingNumWaited;
}

rint InstrSleep::getNumCycles(Task* task)
{
	rint op1 = getValue(task, 0);
	if(op1 < 1) op1 = 1;			// sleep must take at least one cycle
	if(ops[0].isRemote) // or durRemoteAcc if we've got   SLEEP %VAR 
		op1 = Math::max(op1, task->sim->glob.durRemote);

	if(op1 > task->sim->glob.maxSleepDur) op1 = task->sim->glob.maxSleepDur;
	return op1;
}

ExecReturnType InstrSleep::exec(Task* task)
{
	return execSleeps;
}

String InstrSleep::getName()
{
	return String("Sleep");
}

InstrSleep::InstrSleep(const Globals& glob, const Array<Op>& ops) : Instr(1)
{
	initOps(ops);
	cycles = ops[0].ptr;
}
InstrSleep::~InstrSleep(){};

/********/
/* COMP */
/********/

ExecReturnType InstrComp::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if(op1 == op2)
		task->vars[taskCurInstr] ++; // Zeile ueberspringen

	return execOK;
}

String InstrComp::getName()
{
	return String("Comp");
}

InstrComp::InstrComp(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durComp;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrComp::~InstrComp(){};

/*********/
/* NCOMP */
/*********/

ExecReturnType InstrNcomp::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if(op1 != op2)
		task->vars[taskCurInstr] ++; // Zeile ueberspringen

	return execOK;
}

String InstrNcomp::getName()
{
	return String("NComp");
}

InstrNcomp::InstrNcomp(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durNcomp;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrNcomp::~InstrNcomp(){};

/*********/
/* LCOMP */
/*********/

ExecReturnType InstrLcomp::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if(op1 <= op2) // LCOMP is a less-or-equal comparison!!
		task->vars[taskCurInstr] ++; // Zeile ueberspringen

	return execOK;
}

String InstrLcomp::getName()
{
	return String("LComp");
}

InstrLcomp::InstrLcomp(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durLcomp;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrLcomp::~InstrLcomp(){};

/*********/
/* GCOMP */
/*********/

ExecReturnType InstrGcomp::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.

	if(op1 >= op2) // GCOMP is a greater-or-equal comparison!!!
		task->vars[taskCurInstr] ++; // Zeile ueberspringen

	return execOK;
}

String InstrGcomp::getName()
{
	return String("GComp");
}

InstrGcomp::InstrGcomp(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durGcomp;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrGcomp::~InstrGcomp(){};

/**********/
/* CREATE */
/**********/

rint InstrCreate::getNumCycles(Task* task)
{
	// Check Instruction Set
	if(task->bot->vars[botInstrSet] < 2)
	{
		return failInstrSet;
	}

	rint op1,op2,op3;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.
	op3 = getValue(task, 2);   // Interpretiert den Typ zusammen mit dem Param.

	// Check new bot instruction set
	if((op1 < 0) || (op1 > 2))
	{
		return failParam;
	}

	// Check new bot bank count 
	if((op2 < 1) || (op2 > task->sim->glob.maxBanks))
	{
		return failParam;
	}

	// Check new bot mobile value
	if((op3 < 0) || (op3 > 1))
	{
		return failParam;
	}

	rint newcycles = cycles;
	newcycles += op2 * task->sim->glob.durCreatePerBank;

	if(op3) {
		newcycles *= task->sim->glob.durCreateMulMobile;
		newcycles += task->sim->glob.durCreateAddMobile;
	}

	if(op1 == 1)
		newcycles += task->sim->glob.durCreateAdvISet;

	if(op1 == 2)
		newcycles += task->sim->glob.durCreateSuperISet;

	if(newcycles > task->sim->glob.maxCreateDur)
		newcycles = task->sim->glob.maxCreateDur;

	return newcycles;
}

ExecReturnType InstrCreate::exec(Task* task)
{
	rint op1,op2,op3;
	op1 = getValue(task, 0);   // Interpretiert den Typ zusammen mit dem Param.
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.
	op3 = getValue(task, 2);   // Interpretiert den Typ zusammen mit dem Param.
	Bot* bot = task->bot;

	// Check new bot instruction set
	if((op1 < 0) || (op1 > 2))
	{
		return failParam;
	}

	// Check new bot bank count 
	if((op2 < 1) || (op2 > task->sim->glob.maxBanks))
	{
		return failParam;
	}

	// Check new bot mobile value
	if((op3 < 0) || (op3 > 1))
	{
		return failParam;
	}

	// Check MaxGeneration
	if(bot->vars[botGeneration] >= task->sim->glob.maxGeneration) return execOK;
	// Check MaxNumBots
	if(task->bot->owner->vars[progMybots] >= task->sim->glob.maxMybots) return execOK;

	rint x,y;
	task->getRefBotPos(x, y);

	Simulation *sim = bot->owner->sim;

	if(sim->field[x + y * sim->fieldWidth] == 0)
		new Bot(op1, op2, op3, x, y,
			   task->vars[taskDir], bot->vars[botGeneration]+1,
			   bot->owner);
	return execOK;
}

String InstrCreate::getName()
{
	return String("Create");
}

InstrCreate::InstrCreate(const Globals& glob, const Array<Op>& ops) : Instr(3)
{
	cycles = glob.durCreate;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;
	if(ops[2].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
	
}

InstrCreate::~InstrCreate(){};

/**********/
/* RANDOM */
/**********/

ExecReturnType InstrRandom::exec(Task* task)
{
	rint op2,op3;
	op2 = getValue(task, 1);   // Interpretiert den Typ zusammen mit dem Param.
	op3 = getValue(task, 2);   // Interpretiert den Typ zusammen mit dem Param.

	// Check random range
	if(op3 < 0)
	{
		return failParam;
	}

	rint op1 = (rint)Math::rand(op2, op2 + op3);

	setValue(task, 0, op1);

	return execOK;
}

String InstrRandom::getName()
{
	return String("Random");
}

InstrRandom::InstrRandom(const Globals& glob, const Array<Op>& ops) : Instr(3)
{
	cycles = glob.durRandom;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;
	if(ops[2].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
	
}
InstrRandom::~InstrRandom(){};

////////////////////////////// Multitasking ///////////////////////////////

/*********/
/* Break */
/*********/

ExecReturnType InstrBreak::exec(Task* task)
{
	// delete all tasks, except our task
	for(int t = task->bot->tasks.length() - 1; t >= 0; t--)
		if(task->bot->tasks[t] != task)
			delete task->bot->tasks[t];
	return execOK;
}

String InstrBreak::getName()
{
	return String("Break");
}

InstrBreak::InstrBreak(const Globals& glob) : Instr(0)
{
	cycles = glob.durBreak;
}

InstrBreak::~InstrBreak(){};


/*********/
/* Seize */
/*********/

ExecReturnType InstrSeize::exec(Task* task)
{
	task->seized = true;
	return execOK;
}

String InstrSeize::getName()
{
	return String("Seize");
}

InstrSeize::InstrSeize(const Globals& glob) : Instr(0)
{
	cycles = glob.durSeize;
}

InstrSeize::~InstrSeize(){};


/**********/
/* Resume */
/**********/

ExecReturnType InstrResume::exec(Task* task)
{
	task->seized = false;
	return execOK;
}

String InstrResume::getName()
{
	return String("Resume");
}

InstrResume::InstrResume(const Globals& glob) : Instr(0)
{
	cycles = glob.durResume;
}

InstrResume::~InstrResume(){};


/*********/
/* Quit */
/*********/

ExecReturnType InstrQuit::exec(Task* task)
{
	// Do NOT kill the task here! We're still inside its exec() function!

	ExecReturnType ret = execOK;

	if(task->bot->tasks.length() == 1)
		ret = failUnemploy; // kill the bot
	else
		ret = failQuit; // leave it alive

	return ret;
}

String InstrQuit::getName()
{
	return String("Quit");
}

InstrQuit::InstrQuit(const Globals& glob) : Instr(0)
{
	cycles = glob.durQuit;
}

InstrQuit::~InstrQuit(){};

/*********/
/* Init  */
/*********/

ExecReturnType InstrInit::exec(Task* task)
{
	rint op1,op2;
	op1 = getValue(task, 0);   // init at op1'th bank
	op2 = getValue(task, 1, true);   // init at op2'th instr in the op1'th bank

	if(task->sim->glob.maxTasks <= 1)  // new Task() would fail
		return execOK;

	// op1 cannot be a bank! so kill bot
	if((op1 < 0) || (op1 > task->sim->glob.maxBanks))
	{
		return failBank;
	}

	Task* newTask = new Task(task->bot, task, task->vars[taskDir]);

	newTask->vars[taskCurBank] = op1 - 1;
	newTask->vars[taskCurInstr] = op2 - 2;

	return execOK;
}

String InstrInit::getName()
{
	return String("Init ");
}

InstrInit::InstrInit(const Globals& glob, const Array<Op>& ops) : Instr(2)
{
	cycles = glob.durInit;
	if(ops[0].isRemote) cycles += glob.durRemote;
	if(ops[1].isRemote) cycles += glob.durRemote;

	initOps(ops);
	
}
InstrInit::~InstrInit(){};







} // namespace

