/*
 * wxext.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

#include <wx/wx.h>
#include <wx/config.h>
#include <wx/toolbar.h>
#include <wx/mimetype.h>

#include "wxext.h"

#include "bitmaps/miniclose.xpm"

// Windows implementation
#ifdef __WXMSW__
#include <windows.h>
#include <wx/msw/winundef.h>

wxLongLong wxxFileSize(const wxString& file)
{
	wxLongLong ret = -1;

	WIN32_FIND_DATA data;
	HANDLE hFile;
	if((hFile = FindFirstFile(file.c_str(), &data)) 
		!= INVALID_HANDLE_VALUE) {
		ret = data.nFileSizeHigh;
		ret <<= 32;
		ret += data.nFileSizeLow;
		FindClose(hFile);
	}
	return ret;
}
#else // generic version

// UNIX implementation 
#include <unistd.h>
#include <sys/stat.h>
wxLongLong wxxFileSize(const wxString& file)
{
	wxLongLong ret = -1;

	struct stat status;
	if( !wxStat( file.c_str(), &status ) )
	{
       ret = status.st_size;
	}
	return ret;
}
#endif // OS


bool wxxShowInBrowser(const wxString& location, const wxString& defaultCmd)
{
	wxFileType* ftHtml = wxTheMimeTypesManager->GetFileTypeFromMimeType(wxT("text/html"));
	wxString browserCmd; 
	if(ftHtml) browserCmd = ftHtml->GetOpenCommand(location); 
	delete ftHtml; ftHtml = 0; 

	if(browserCmd.IsEmpty()) { // resort to asking the user... 
		browserCmd = wxxReadString(wxT(""), wxT("/BrowserCmd")); 
		if(browserCmd.IsEmpty())
			browserCmd = wxGetTextFromUser(_("Enter the command to start your favourite browser:"), _("Browser selection"), wxT("firefox")); 
		if(browserCmd.IsEmpty()) // user doesn't want to enter browser
			return false; 
		else {
			wxxStoreString(browserCmd, wxT("/BrowserCmd")); 
			browserCmd += wxT(" ") + location;
		}
	}
	if(!wxExecute(browserCmd)) {
		wxLogError(_("Cannot start browser for showing HTML file. Please open %s yourself."), location.c_str()); 
		wxxStoreString(wxT(""), wxT("/BrowserCmd")); // browser does not work, ask again next time
		return false; 
	}
	return true; 
}



void wxxAddCloseBar(wxWindow* parent, wxWindow* closeW, wxSizer* parentSizer, 
					wxToolBar* bar, wxCommandEventFunction closeFunc)
{
#ifdef __WXGTK__
	wxSizer* closeSizer = new wxBoxSizer(wxHORIZONTAL);
	closeSizer->Add(bar, 1, wxEXPAND, 0);
	wxBitmap closeBmp(miniclose_xpm);
	wxBitmapButton* closeButton = new wxBitmapButton(parent, -1, closeBmp);
	closeW->Connect(closeButton->GetId(), wxEVT_COMMAND_BUTTON_CLICKED, (wxObjectEventFunction) closeFunc);
	closeSizer->Add(closeButton, 0, wxALIGN_CENTER | wxRIGHT | wxLEFT, 2);
	parentSizer->Add(closeSizer, 0, wxEXPAND | wxALL, 2);
#else	
	parentSizer->Add( bar, 0,	wxEXPAND | wxALL, 2 );
#endif
}



wxString wxxReadString(const wxString& def, const wxString& key)
{
	wxConfigBase* cfg = wxConfig::Get();
	return cfg->Read(key, def);
}

void wxxStoreString(const wxString& val, const wxString& key)
{
	wxConfigBase* cfg = wxConfig::Get();
	cfg->Write(key, val);
}




// for toplevel windows (frames dialogs)
// * handle "MAXIMIZED" case specially
void wxxStoreRect(wxWindow* win, const wxString& cfgKey)
{
	wxString val; 
	wxRect rect = win->GetRect();
	wxConfigBase* cfg = wxConfig::Get();

	if(wxDynamicCast(win, wxTopLevelWindow)) // top level windows can be iconized or maximized which must be treated separately
	{
		// In iconized or maximized state, the GetRect() function return values are not really helpful. 
		wxTopLevelWindow* twin = wxDynamicCast(win, wxTopLevelWindow);
		if(twin->IsIconized()) // we don't want to save this state. It would be confusing. 
			return; 
		else if(twin->IsMaximized())
		{
			// just re-use previous state
			val = cfg->Read(cfgKey, wxT(""));
			if(val.AfterLast(',') != wxT("max")) // was not already saved max'ed => append ",max"
				val += wxT(",max"); 
			cfg->Write(cfgKey, val);
			return;
		}
	}
	// window is no toplevel, or is not in a special state
	val = wxString::Format(wxT("%d,%d,%d,%d"), rect.x, rect.y, rect.width, rect.height);
	cfg->Write(cfgKey, val);
}

// for toplevel windows (frames dialogs)
// * handle "MAXIMIZED" case specially
// * make sure that the window is not positioned offscreen due to resolution changes
void wxxRestoreRect(wxWindow* win, const wxString& cfgKey)
{
	wxString str = wxxReadString(wxT(""), cfgKey); 
	wxString arr[5]; 

	for(int p = 0; p < 5; p++) {
		arr[p] = str.BeforeFirst(','); 
		str = str.AfterFirst(','); 
	}

	long x = -1, y = -1, w = -1, h = -1; 
	arr[0].ToLong(&x); 
	arr[1].ToLong(&y);
	arr[2].ToLong(&w); 
	arr[3].ToLong(&h); 

	// validate sizes (-1 is OK and does nothing)
	wxSize screenSize = wxSize(wxSystemSettings::GetMetric(wxSYS_SCREEN_X), wxSystemSettings::GetMetric(wxSYS_SCREEN_Y));
	if(x < -100 || x > screenSize.x - 100) x = -1;
	if(y < -10  || y > screenSize.y - 100) y = -1;
	if(w < 10   || w > screenSize.x) w = -1;
	if(h < 10   || h > screenSize.y) h = -1; 

	wxTopLevelWindow* twin = wxDynamicCast(win, wxTopLevelWindow);
	if(!twin || !twin->IsMaximized())
		win->SetSize(x, y, w, h);

	// handle maximization
	if(arr[4] == wxT("max"))
	{
		if(twin) twin->Maximize(true);
	}
		
}



// /////////////////////////////////////////////////////////////// //
// ////////////////////////////////////////////////////////////// //



wxxMDIChildFrame::wxxMDIChildFrame(wxMDIParentFrame* parent, const wxString& title, const wxString& cfgID, long style) 
	: wxMDIChildFrame(parent, -1, title, wxDefaultPosition, wxDefaultSize, style), cfgID(cfgID) 
{
	RestoreRect(); 

	Connect(-1, wxEVT_SIZE, (wxObjectEventFunction) &wxxMDIChildFrame::OnSize);
}

bool wxxMDIChildFrame::Destroy()
{
	StoreRect();
	return wxMDIChildFrame::Destroy(); 
}

void wxxMDIChildFrame::RestoreRect()
{
	wxxRestoreRect(this, wxT("/Rect/") + cfgID);
}

void wxxMDIChildFrame::StoreRect()
{
	wxxStoreRect(this, wxT("/Rect/") + cfgID);
}

void wxxMDIChildFrame::OnSize(wxSizeEvent& evt)
{
	StoreRect(); 
	evt.Skip(); // we just want to get the new size here, continue processing... 
}

